/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.bpel.xpath.view.expression.impl;

import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.Logger;

import javax.swing.AbstractAction;
import javax.swing.JPopupMenu;
import javax.swing.JTree;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeExpansionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import org.netbeans.modules.bpel.model.api.BpelEntity;
import org.netbeans.modules.bpel.xpath.model.nodes.XPathLocationStepNode;
import org.netbeans.modules.bpel.xpath.model.nodes.XSDElementNode;

import org.openide.util.NbBundle;


/**
 *
 * @author radval
 *
 */
public abstract class MapperTreeHelper {
    
    protected static final Logger LOGGER = 
            Logger.getLogger(MapperTreeHelper.class.getName());
    
    private static final String ACTION_FIND =
            NbBundle.getMessage(MapperTreeHelper.class, "STR_FIND");  // NOI18N

    private static final String ACTION_EXPAND =
            NbBundle.getMessage(MapperTreeHelper.class, "STR_EXPAND");  // NOI18N
    
    private static final String ACTION_COLLAPSE =
            NbBundle.getMessage(MapperTreeHelper.class, "STR_COLLAPSE");    // NOI18N
    
    private static final String MAX_RECURSION_DEPTH = "10"; // NOI18N
    
    private static int effMaxRecursionDepth =
            Integer.parseInt(System.getProperty(
            "org.netbeans.modules.bpel.xpath.view.BPViewMapperManager.maxRecursionDepth",   // NOI18N
            MAX_RECURSION_DEPTH));
    
    private HashMap mExpandedMapperSrcTreeNodes = new HashMap();
    private BpelMapper mBpelMapper;
    private BpelEntity mSelectedElement;

    
    public MapperTreeHelper(BpelMapper bpelMapper) {
        this.mSelectedElement = bpelMapper.getSelectedElement();
        this.mBpelMapper = bpelMapper;
    }
    
    
    protected abstract TreeModel getTreeModel();
    
    
    public BpelEntity getSelectedElement() {
        return this.mSelectedElement;
    }
    
    public TreeSelectionListener getTreeSelectionListener() {
        return new MapperTreeSelectionListener();
    }
    
    public TreeExpansionListener getTreeExpansionListener() {
        return new MapperTreeExpansionListener();
    }
    
    public MouseListener getMouseListener() {
        return new MapperTreeMouseListener();
    }
    
    protected JPopupMenu createPopUpMenu(final JTree tree) {
        JPopupMenu popUpMenu = new JPopupMenu();
        final TreePath tPath = tree.getSelectionPath();
        popUpMenu.add(new AbstractAction(ACTION_EXPAND) {
            public void actionPerformed(ActionEvent ae) {
                if (tPath != null) {
                    expandTreeRecursive(tree, tPath, 0);
                }
            }
        });
        if (tPath == null) {
            return popUpMenu;
        }
        popUpMenu.addSeparator();
        popUpMenu.add(new AbstractAction(ACTION_COLLAPSE) {
            public void actionPerformed(ActionEvent ae) {
                tree.collapsePath(tPath);
            }
        });
        
        Object[] nodes = tPath.getPath();
        if (
                nodes != null &&
                nodes.length > 0 &&
                (nodes[nodes.length - 1] instanceof XSDElementNode) ||
                (nodes[nodes.length - 1] instanceof XPathLocationStepNode)) {
            if (nodes[nodes.length - 1] instanceof XSDElementNode) {
                final XSDElementNode selectedNode = (XSDElementNode) nodes[nodes.length - 1];
                if (selectedNode.getMaxOccurs() != 1 && selectedNode.getMaxOccurs() != 0) {
                    popUpMenu.add
                            (new AbstractAction(NbBundle.getMessage(MapperTreeHelper.class,
                            "STR_NEW_PREDICATE_ACTION")) {  // NOI18N
                        public void actionPerformed(ActionEvent ae) {
                            mBpelMapper.getPredicatesManager().insertNewPredicateInTree(
                                    mBpelMapper, selectedNode, tree, tPath);
                        }
                    });
                }
            } else if (nodes[nodes.length - 1] instanceof XPathLocationStepNode) {
                final XPathLocationStepNode selectedNode = (XPathLocationStepNode) nodes[nodes.length - 1];
                popUpMenu.add
                        (new AbstractAction(NbBundle.getMessage(MapperTreeHelper.class,
                        "STR_DELETE_PREDICATE_ACTION")) {   // NOI18N
                    public void actionPerformed(ActionEvent ae) {
                        mBpelMapper.getPredicatesManager().deletePredicateFromTree(
                                mBpelMapper, selectedNode, tree, tPath);
                    }
                });
                popUpMenu.add
                        (new AbstractAction(NbBundle.getMessage(MapperTreeHelper.class,
                        "STR_EDIT_PREDICATE_ACTION")) {     // NOI18N
                    public void actionPerformed(ActionEvent ae) {
                        mBpelMapper.getPredicatesManager().editPredicateInTree(
                                mBpelMapper, (XPathLocationStepNode) selectedNode, tree, tPath);
                    }
                });
            }
        }
        
        return popUpMenu;
        
    }
    
    /**
     * This expands the mapper tree recursively to a maximum defined depth.
     *
     * @param tree a <code>JTree</code> value
     * @param tPath a <code>TreePath</code> value
     * @param recursiveDepth an <code>int</code> value
     */
    private void expandTreeRecursive(JTree tree, TreePath tPath, int recursiveDepth) {
        recursiveDepth++;
        if (recursiveDepth >= effMaxRecursionDepth) {
            return;
        }
        if (tPath == null) {
            return;
        }
        TreeModel tModel = tree.getModel();
        Object expandedObj = tPath.getLastPathComponent();
        Object child = null;
        TreePath newTPath = null;
        int childrenCount = tModel.getChildCount(expandedObj);
        TreePath[] tPaths = new TreePath[childrenCount];
        for (int i = 0; i < childrenCount; i++) {
            child = tModel.getChild(expandedObj, i);
            newTPath = tPath.pathByAddingChild(child);
            tree.expandPath(newTPath);
            tree.makeVisible(newTPath);
            tPaths[i] = newTPath;
        }
        for (int j = 0; j < tPaths.length; j++) {
            expandTreeRecursive(tree, tPaths[j], recursiveDepth);
        }
    }
    
    
    class MapperTreeExpansionListener implements TreeExpansionListener {
        
        public void treeCollapsed(TreeExpansionEvent event) {
            BpelEntity selection = getSelectedElement();
            if (selection == null) {
                LOGGER.severe(NbBundle.getMessage(MapperTreeHelper.class,
                        "STR_LINK_CANNOT_BE_NULL_WHILE_TREE_MODIFIED"));    // NOI18N
                return;
            }
            
            Set s = null;
            s = (Set) mExpandedMapperSrcTreeNodes.get(selection);
            if (s == null) {
                return;
            }
            
            TreePath tp = event.getPath();
            for (Iterator itr = s.iterator(); itr.hasNext();) {
                TreePath obj = (TreePath) itr.next();
                if (tp.isDescendant((TreePath) obj)
                && !(tp.equals(obj))) {
                    itr.remove();
                }
            }
            
            mExpandedMapperSrcTreeNodes.put(selection, s);
        }
        
        public void treeExpanded(TreeExpansionEvent event) {
            LOGGER.finer("In Expand TreePath");     // NOI18N
            BpelEntity selection = getSelectedElement();
            
            if (selection == null) {
                LOGGER.severe(NbBundle.getMessage(MapperTreeHelper.class,
                        "STR_LINK_CANNOT_BE_NULL_WHILE_TREE_MODIFIED"));    // NOI18N
                return;
            }
            Set s = null;
            
            s = (Set) mExpandedMapperSrcTreeNodes.get(selection);
            
            if (s == null) {
                s = new HashSet();
            }
            // when someone expands we need to save the state of the next level
            // so get the first child of the expanded node and use that for the
            // state.
            TreePath tp = event.getPath();
            Object expandedObj = tp.getLastPathComponent();
            Object child = null;
            TreeModel tModel = null;
            
            tModel = getTreeModel();
            
            if (tModel.getChildCount(expandedObj) > 0) {
                child = tModel.getChild(expandedObj, 0);
                s.add(tp.pathByAddingChild(child));
            } else {
                s.add(tp);
            }
            
            mExpandedMapperSrcTreeNodes.put(selection, s);
        }
    }
    
    
    class MapperTreeSelectionListener implements TreeSelectionListener {
        
        public void valueChanged(TreeSelectionEvent event) {
            LOGGER.finer("In tree selection changed");  // NOI18N
            BpelEntity selection = getSelectedElement();
            
            if (selection == null) {
                LOGGER.severe(NbBundle.getMessage(MapperTreeHelper.class,
                        "STR_LINK_CANNOT_BE_NULL_WHILE_TREE_MODIFIED"));    // NOI18N
                return;
            }
            
            Set s = null;
            s = (Set) mExpandedMapperSrcTreeNodes.get(selection);
            
            if (s == null) {
                s = new HashSet();
            }
            
            s.add(event.getPath());
            mExpandedMapperSrcTreeNodes.put(selection, s);
        }
    }
    
    
    class MapperTreeMouseListener extends MouseAdapter {
        
        boolean isComponentInFocus = false;
        
        public void mouseEntered(MouseEvent evt) {
            isComponentInFocus = true;
            super.mouseEntered(evt);
        }
        
        public void mouseExited(MouseEvent evt) {
            isComponentInFocus = false;
            super.mouseEntered(evt);
        }
        
        public void mousePressed(MouseEvent e) {
            maybeShowPopup(e);
        }
        
        public void mouseReleased(MouseEvent e) {
            maybeShowPopup(e);
        }
        
        private void maybeShowPopup(MouseEvent e) {
            if (!isComponentInFocus) {
                return;
            }
            if (e.isPopupTrigger()) {
                JTree tree = (JTree) e.getComponent();
                
                // issue#80213 - commented below to allow node
                // selection via right click
                //TreePath tp = tree.getSelectionPath();
                // if selected treepath is null donot show the popup.
                //if (tp == null) {
                //    return;
                //}
                
                // if selected treepath is not null, then show the popup
                // only if right click happens on the selected path
                int selRow = tree.getRowForLocation(e.getX(), e.getY());
                
                // issue#80213 - allow an initial right click on a
                // node to 'select' and 'show' the popup in one single step
                if (!tree.isRowSelected(selRow)) {
                    tree.setSelectionRow(selRow);
                }
                
                if (selRow != -1) {
                    createPopUpMenu(tree).
                            show(e.getComponent(), e.getX(), e.getY());
                }
            }
        }
    }
}
