/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.sun.manager.jbi.management;

import java.net.MalformedURLException;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;
import javax.management.Attribute;

import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.TabularDataSupport;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.netbeans.modules.sun.manager.jbi.GenericConstants;
import org.netbeans.modules.sun.manager.jbi.management.connectors.HTTPServerConnector;
import org.netbeans.modules.sun.manager.jbi.management.model.ComponentInformationParser;
import org.netbeans.modules.sun.manager.jbi.management.model.JBIComponentDocument;
import org.netbeans.modules.sun.manager.jbi.management.model.JBIComponentStatus;
import org.netbeans.modules.sun.manager.jbi.management.model.JBIServiceAssemblyDocument;
import org.netbeans.modules.sun.manager.jbi.management.model.JBIServiceAssemblyStatus;
import org.netbeans.modules.sun.manager.jbi.management.model.JBIServiceUnitStatus;
import org.netbeans.modules.sun.manager.jbi.management.model.ServiceAssemblyInformationParser;
import org.netbeans.modules.sun.manager.jbi.util.FileTransferManager;
import org.netbeans.modules.sun.manager.jbi.util.ServerInstance;
import org.openide.util.NbBundle;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import static org.netbeans.modules.sun.manager.jbi.management.JBIComponentType.*;

/**
 * Standard API required to monitor and manage JBI Service assemblies,
 * JBI Binding components, JBI Shared Libraries, JBI Service Assemblies,
 * and JBI Service Units
 *
 * @author Graj
 * @author jqian
 */
public class AdministrationService implements Serializable {
        
    public static final String ADMIN_SERVICE_OBJECTNAME = 
            "com.sun.jbi:ServiceName=JbiAdminUiService,ComponentType=System"; // NOI18N
    public static final String REFERENCE_ADMIN_SERVICE_OBJECTNAME = 
            "com.sun.jbi:ServiceName=JbiReferenceAdminUiService,ComponentType=System"; // NOI18N
    private static final String SERVER_TARGET = "server"; // NOI18N

    private static final String PROTOCOL_CLASS = "com.sun.enterprise.admin.jmx.remote.protocol"; // NOI18N
    private static final String HTTP_AUTH_PROPERTY_NAME = "com.sun.enterprise.as.http.auth"; // NOI18N
    private static final String DEFAULT_HTTP_AUTH_SCHEME = "BASIC"; // NOI18N
    private static final String ADMIN_USER_ENV_PROPERTY_NAME = "USER"; // NOI18N
    private static final String ADMIN_PASSWORD_ENV_PROPERTY_NAME = "PASSWORD"; // NOI18N
    private static final String RTS_HTTP_CONNECTOR = "s1ashttp"; // NOI18N

    private static final String ENVIRONMENT_VARIABLES = "EnvironmentVariables";
    private static final String ENVIRONMENT_VARIABLE_NAME = "name";
    private static final String ENVIRONMENT_VARIABLE_VALUE = "value";
    
    private MBeanServerConnection serverConnection;
    private ObjectName objectName;
        
    private List<JBIComponentStatus> serviceEngineStatusList;
    private List<JBIComponentStatus> bindingComponentStatusList;
    private List<JBIComponentStatus> sharedLibraryStatusList;
    private List<JBIServiceAssemblyStatus> serviceAssemblyStatusList;
    
      
    public AdministrationService(HTTPServerConnector httpConnector) 
    throws MalformedObjectNameException, IOException {
        this(httpConnector.getConnection());
    }
    
    public AdministrationService(MBeanServerConnection connection) 
    throws MalformedObjectNameException, IOException {        
        this.serverConnection = connection;
        initObjectName();
    }
    
    public AdministrationService(ServerInstance serverInstance) 
            throws MalformedURLException, IOException, MalformedObjectNameException {
        this(serverInstance.getHostName(),
                serverInstance.getAdminPort(),
                serverInstance.getUserName(),
                serverInstance.getPassword(), 
                new JBIClassLoader(serverInstance));
    }
    
    private AdministrationService(
            String hostName, String port, String userName, String password,
            JBIClassLoader jbiClassLoader)
            throws MalformedURLException, IOException, MalformedObjectNameException {
        this(new HTTPServerConnector(
                hostName, port, userName, password, jbiClassLoader));
    }
    
    private void initObjectName() 
            throws MalformedObjectNameException, IOException {
        QueryExp queryExpression = null;
        objectName = new ObjectName(REFERENCE_ADMIN_SERVICE_OBJECTNAME);        
        
        if (this.serverConnection != null) {
            Set set = this.serverConnection.queryNames(objectName, queryExpression);
            Iterator iterator = set.iterator();
            if (iterator != null && iterator.hasNext()) {
                objectName = (ObjectName) iterator.next();
            }
        }
        
        if (objectName == null) {
            System.out.println(NbBundle.getMessage(AdministrationService.class, 
                    "Connection_Failed")); // NOI18N
        }
    }
  
    public List<JBIServiceAssemblyStatus> getServiceAssemblyStatusList() {
        // Use cache if available
        if (serviceAssemblyStatusList != null) {
            return serviceAssemblyStatusList;
        }

        String xml = (String) this.invoke(
                objectName,
                GenericConstants.LIST_SERVICE_ASSEMBLIES_OPERATION_NAME,
                new String[] {SERVER_TARGET});
        try {
            JBIServiceAssemblyDocument document = 
                    ServiceAssemblyInformationParser.parse(xml);
            if (document != null) {
                serviceAssemblyStatusList = document.getJbiServiceAssemblyList();
                return serviceAssemblyStatusList;
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        
        return new ArrayList<JBIServiceAssemblyStatus>();
    }
    
    public void clearServiceAssemblyStatusCache() {
        serviceAssemblyStatusList = null;
    }
    
    public List<JBIComponentStatus> getJBIComponentStatusList(
            JBIComponentType compType) {
        
        List<JBIComponentStatus> cachedList = null;
        String operationName = null;
        
        switch (compType) {
            case SERVICE_ENGINE:
                cachedList = serviceEngineStatusList;
                operationName = GenericConstants.LIST_SERVICE_ENGINES_OPERATION_NAME;
                break;
            case BINDING_COMPONENT:
                cachedList = bindingComponentStatusList;
                operationName = GenericConstants.LIST_BINDING_COMPONENTS_OPERATION_NAME;
                break;
            case SHARED_LIBRARY:
                cachedList = sharedLibraryStatusList;
                operationName = GenericConstants.LIST_SHARED_LIBRARIES_OPERATION_NAME;
                break;
            default:
                assert false : "Unknown component type: " + compType;
        }
                 
        // Use cache if available
        if (cachedList != null) {
            return cachedList;
        }
                    
        String[] params = new String[] {SERVER_TARGET};        
        String xml = (String) this.invoke(objectName, operationName, params);
         
        if (xml != null) {
            try {
                JBIComponentDocument document = ComponentInformationParser.parse(xml);
                if (document != null) {
                    List<JBIComponentStatus> ret = document.getJbiComponentList();
                    switch (compType) {
                        case SERVICE_ENGINE:
                            serviceEngineStatusList = ret;
                            break;
                        case BINDING_COMPONENT:
                            bindingComponentStatusList = ret;
                            break;
                        case SHARED_LIBRARY:
                            sharedLibraryStatusList = ret;
                    }
                    return ret;
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
        
        return new ArrayList<JBIComponentStatus>();
    }
           
    public void clearJBIComponentStatusCache(JBIComponentType compType) {
        switch (compType) {
            case SERVICE_ENGINE:
                serviceEngineStatusList = null;
                break;
            case BINDING_COMPONENT:
                bindingComponentStatusList = null;
                break;
            case SHARED_LIBRARY:
                sharedLibraryStatusList = null;
                break;
            default:
                assert false : "Unknown component type: " + compType;
        }
    }
        
    private Object invoke(ObjectName objectName, String operationName) {
        return invoke(objectName, operationName, null, null);
    }
    
    private Object invoke(ObjectName objectName, String operationName,
            Object[] parameters) {
        String[] signature = this.getSignatures(parameters);
        return invoke(objectName, operationName, parameters, signature);
    }

    private Object invoke(ObjectName objectName, String operationName, 
            Object[] parameters, String[] signature) {
        Object output = null; 
        try {
            if (this.serverConnection != null) {
                output = this.serverConnection.invoke(
                        objectName,
                        operationName,
                        parameters,
                        signature);
            }
        } catch (Exception e) {
            output = this.formatException(e, operationName);
        }
        
        return output;
    }
          
    /**
     *
     * @param e
     * @param operationName
     * @return
     */
    private String formatException(Exception e, String operationName) {
        String result = "Failed execution of " + operationName + ". "; // NOI18N
        Throwable throwable = e.getCause();
        if (throwable != null) {
            result += throwable.getMessage().trim();
        } else {
            result += e.getMessage();
        }
        return result;
    }
    
    /**
     * get parameter signatures based on the object type class
     *
     * @param params
     * @return
     */
    private String[] getSignatures(Object[] params) {
        if (params == null || params.length == 0) {
            return null;
        }
        String[] signatures = new String[params.length];
        for (int index = 0; index < params.length; index++) {
            if (params[index] == null) {
                signatures[index] = "java.lang.Object"; // NOI18N
            } else {
                String className = params[index].getClass().getName();
                
                // TODO: possible changes for other primitive types
                if (className.equals(Boolean.class.getName())) {
                    signatures[index] = "boolean"; // NOI18N
                } else {
                    signatures[index] = className;
                }
            }
        }
        return signatures;
    }
       
     /**
     * Retrieves the status of the given Service Assembly deployed on the
     * JBI Container on the Server.
     *
     * @param assemblyName  name of a Service Assembly
     * @return
     */
    public JBIServiceAssemblyStatus getServiceAssemblyStatus(String assemblyName) {
        for (JBIServiceAssemblyStatus assembly : getServiceAssemblyStatusList()) {
            if (assembly.getServiceAssemblyName().equals(assemblyName)) {
                return assembly;
            }
        }        
        return null;
    }
    
    /**
     * Retrieves the status of the given Service Unit on the
     * JBI Container on the Server.
     *
     * @param assemblyName  name of a Service Assembly
     * @param unitName  name of a Service Unit inside the Service Assembly
     * @return
     */
    public JBIServiceUnitStatus getServiceUnitStatus(String assemblyName,
            String unitName) {        
        JBIServiceAssemblyStatus assembly =
                getServiceAssemblyStatus(assemblyName);        
        if (assembly != null) {
            for (JBIServiceUnitStatus unit : assembly.getJbiServiceUnitStatusList()) {
                if (unit.getServiceUnitName().equals(unitName)) {
                    return unit;
                }
            }
        }        
        return null;
    }
    
    public String getServiceUnitDeploymentDescriptor(String assemblyName,
            String unitName) {        
        String xml = (String) this.invoke(
                objectName,
                GenericConstants.GET_SERVICE_UNIT_DEPLOYMENT_DESCRIPTOR_NAME,
                new String[] {assemblyName, unitName});        
        return xml;
    }
    
    public String getServiceAssemblyDeploymentDescriptor(String assemblyName) {        
        String xml = (String) this.invoke(
                objectName,
                GenericConstants.GET_SERVICE_ASSEMBLY_DEPLOYMENT_DESCRIPTOR_NAME,
                new String[] {assemblyName});        
        return xml;
    }
    /**
     * Retrieves the status of the given JBI Component installed on the
     * JBI Container on the Server.
     *
     * @param compType  type of JBI Component
     * @param compName  name of JBI Component
     * @return
     */
    public JBIComponentStatus getJBIComponentStatus(
            JBIComponentType compType, String compName) {
        for (JBIComponentStatus comp : getJBIComponentStatusList(compType)) {
            if (comp.getName().equals(compName)) {
                return comp;
            }
        }        
        return null;
    }
    
    /**
     * Installs a JBI component (service engine, binding component).
     *
     * @param zipFilePath archive file in a zip format
     * @return lifecycle object name string for service engine or binding component.
     */
    public String installComponent(String zipFilePath) {
        String result = ""; // NOI18N
        
        if (this.serverConnection != null) {
            FileTransferManager manager = null;
            try {
                manager = new FileTransferManager(this.serverConnection);
                File clientFile = new File(zipFilePath); 
                String serverPath = manager.uploadArchive(clientFile);
                String[] params = new String[] {serverPath, SERVER_TARGET};
                result = (String) this.invoke(objectName,
                        GenericConstants.INSTALL_COMPONENT_OPERATION_NAME,
                        params);
            } catch (Exception ex) {
                ex.printStackTrace();
                result = "Failed uploading of " + zipFilePath + ". \n" + ex.getMessage();    // NOI18N
            } finally {
                if (manager != null) {
                    manager.removeUploadedArchive(manager.getLastUploadId());
                }
            }
        }
        
        // TMP FIX for 114274 to start component automatically 
        // upon successful installation
        if (result != null) {
            String lowerCaseResult = result.toLowerCase();
            if (!lowerCaseResult.contains("error") &&        // NOI18N
                    !lowerCaseResult.contains("warning") &&  // NOI18N
                    !lowerCaseResult.contains("exception") &&  // NOI18N
                    !lowerCaseResult.contains("info")) {     // NOI18N
                String userHome = System.getProperty("user.home"); // NOI18N
                String hintFileName = ".autoStartJBIComponentUponInstallation"; // NOI18N
                if (new File(userHome, hintFileName).exists()) {
                    // assume the successful result is the component ID
                    try {
                        result = startComponent(result);
                    } catch (Exception e) {
                        // no biggie, user should still be able to start it manually
                        e.printStackTrace();
                    }
                }
            }
        }
        
        return result;
    }
    
     /**
     * Upgrades/updates a JBI component (service engine, binding component).
     *
     * @param componentName component name
     * @param zipFilePath archive file in a zip format
     * @return lifecycle object name string for service engine or binding component.
     */
    public String upgradeComponent(String componentName, String zipFilePath) {
        String result = ""; // NOI18N
        
        if (this.serverConnection != null) {
            FileTransferManager manager = null;
            try {
                manager = new FileTransferManager(this.serverConnection);
                File clientFile = new File(zipFilePath); 
                String serverPath = manager.uploadArchive(clientFile);
                //String[] params = new String[] {serverPath, SERVER_TARGET};
                String[] params = new String[] {componentName, serverPath};
                result = (String) this.invoke(objectName,
                        GenericConstants.UPGRADE_COMPONENT_OPERATION_NAME,
                        params);
            } catch (Exception ex) {
                ex.printStackTrace();
                result = "Failed uploading of " + zipFilePath + ". \n" + ex.getMessage();    // NOI18N
            } finally {
                if (manager != null) {
                    manager.removeUploadedArchive(manager.getLastUploadId());
                }
            }
        }
        
        return result;
    }
    
    /**
     * Uninstalls a JBI component (service engine, binding component).
     * @param componentName name of the component
     */
    public String uninstallComponent(String componentName, boolean force) {
        Object[] params = new Object[] {componentName, force, SERVER_TARGET};
        String result = (String) this.invoke(objectName,
                GenericConstants.UNINSTALL_COMPONENT_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Installs a shared library.
     * @param zipFilePath   archive file in a zip format
     */
    public String installSharedLibrary(String zipFilePath) {
        String result = ""; // NOI18N
        
        if (this.serverConnection != null) {
            FileTransferManager manager = null;
            try {
                manager = new FileTransferManager(this.serverConnection);
                File clientFile = new File(zipFilePath);
                String serverPath = manager.uploadArchive(clientFile);
                String[] params = new String[] {serverPath, SERVER_TARGET};
                result = (String) this.invoke(objectName,
                        GenericConstants.INSTALL_SHARED_LIBRARY_OPERATION_NAME,
                        params);
            } catch (Exception ex) {
                ex.printStackTrace();
                result = "Failed uploading of " + zipFilePath + ". \n" + ex.getMessage();    // NOI18N
            } finally {
                if (manager != null) {
                    manager.removeUploadedArchive(manager.getLastUploadId());
                }
            }
        }
        
        return result;
    }
    
    /**
     * Uninstalls a shared library.
     * @param name   name of the shared library
     */
    public String uninstallSharedLibrary(String name) {
        String[] params = new String[] {name, SERVER_TARGET};      
        String result = (String) this.invoke(objectName,
                GenericConstants.UNINSTALL_SHARED_LIBRARY_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Starts a component (service engine, binding component).
     * @param componentName name of the component
     */
    public String startComponent(String componentName) {
        String[] params = new String[] {componentName, SERVER_TARGET};    
        String result = (String) this.invoke(objectName,
                GenericConstants.START_COMPONENT_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Stops a JBI component (service engine, binding component).
     * @param componentName name of the component
     */
    public String stopComponent(String componentName) {
        String[] params = new String[] {componentName, SERVER_TARGET};
        String result = (String) this.invoke(objectName,
                GenericConstants.STOP_COMPONENT_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Shuts down a JBI component (service engine, binding component).
     * @param componentName name of the component
     */
    public String shutdownComponent(String componentName, boolean force) {
        Object[] params = new Object[] {componentName, force, SERVER_TARGET};
        String result = (String) this.invoke(objectName,
                GenericConstants.SHUTDOWN_COMPONENT_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Deploys a service assembly.
     * @return deployment status result as a management message xml
     * @param zipFilePath fie path
     */
    public String deployServiceAssembly(String zipFilePath) {
        String result = ""; // NOI18N
        
        if (this.serverConnection != null) {
            FileTransferManager manager = null;
            try {
                manager = new FileTransferManager(this.serverConnection);
                File clientFile = new File(zipFilePath);
                // e.x., serverpath is "C:\\build060730\\Sun\\AppServer\\domains\\domain1\\jbi\\system\\esb-repository\\tmp\\3d4fa36310cff5f718e7f16\\SynchronousSampleApplication.zip"
                String serverPath = manager.uploadArchive(clientFile);
                String[] params = new String[] {serverPath, SERVER_TARGET};
                result = (String) this.invoke(objectName,
                        GenericConstants.DEPLOY_SERVICE_ASSEMBLY_OPERATION_NAME,
                        params);
            } catch (Exception ex) {
                ex.printStackTrace();
                result = "Failed uploading of " + zipFilePath + ". \n" + ex.getMessage();    // NOI18N
            } finally {
                if (manager != null) {
                    manager.removeUploadedArchive(manager.getLastUploadId());
                }
            }
        }
        
        return result;
    }
        
    /**
     * Starts a service assembly.
     * @param serviceAssemblyName name of the service assembly
     */
    public String startServiceAssembly(String serviceAssemblyName) {
        String[] params = new String[] {
            serviceAssemblyName, SERVER_TARGET};
        String result = (String) this.invoke(objectName,
                GenericConstants.START_SERVICE_ASSEMBLY_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Stops a service assembly.
     * @param serviceAssemblyName name of the service assembly
     */
    public String stopServiceAssembly(String serviceAssemblyName) {
        String[] params = new String[] {serviceAssemblyName, SERVER_TARGET};
        String result = (String) this.invoke(objectName,
                GenericConstants.STOP_SERVICE_ASSEMBLY_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Shuts down a service assembly.
     * @param serviceAssemblyName name of the service assembly
     */
    public String shutdownServiceAssembly(String serviceAssemblyName, boolean force) {
        Object[] params = new Object[] {serviceAssemblyName, force, SERVER_TARGET};
        String result = (String) this.invoke(objectName,
                GenericConstants.SHUTDOWN_SERVICE_ASSEMBLY_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Undeploys a service assembly.
     * @param serviceAssemblyName name of the service assembly
     */
    public String undeployServiceAssembly(String serviceAssemblyName, boolean force) {
        Object[] params = new Object[] {serviceAssemblyName, force, SERVER_TARGET};     
        String result = (String) this.invoke(objectName,
                GenericConstants.UNDEPLOY_SERVICE_ASSEMBLY_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Gets service asssembly infos in a xml format that have the
     * service unit deployed on the specified component.
     * 
     * @param componentName to list all the service assemblies that have some 
     *                      deployments on this component.
     */
    public String listServiceAssemblies(String componentName) {
        String[] params = new String[] {componentName, SERVER_TARGET}; 
        String result = (String) this.invoke(objectName,
                GenericConstants.LIST_SERVICE_ASSEMBLIES_OPERATION_NAME,
                params);
        return result;
    }
    
    /**
     * Gets the name list of service asssemblies that have the
     * service unit deployed on the specified component.
     * 
     * @param componentName to list all the service assemblies that have some
     *                      deployments on this component.
     */
    public List<String> getServiceAssemblyNames(String componentName) {
        List<String> ret = new ArrayList<String>();
        
        String saXml = listServiceAssemblies(componentName);
        
        try { 
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document doc = builder.parse(new InputSource(new StringReader(saXml)));

            NodeList nodeList = doc.getElementsByTagName("service-assembly-info"); // NOI18N
            for (int i = 0; i < nodeList.getLength(); i++) {
                Element sa = (Element) nodeList.item(i);
                String name = sa.getAttribute("name"); // NOI18N
                // do we care about the state?
                String state = sa.getAttribute("state"); // NOI18N
                ret.add(name);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        
        return ret;
    }

            
    /**
     * Gets the identification properties of a JBI component (service engine or
     * binding component).
     */  
    public Map<Attribute, MBeanAttributeInfo> getComponentIdentificationProperties(
            String componentName)
            throws MalformedObjectNameException, AttributeNotFoundException,
            InstanceNotFoundException, MBeanException, ReflectionException,
            IOException, ParserConfigurationException, SAXException {
        return getIdentificationProperties(
                GenericConstants.GET_COMPONENT_INSTALLATION_DESCRIPTOR_NAME, 
                componentName);
    }
        
    /**
     * Gets the identification properties of a shared library.
     */
    public Map<Attribute, MBeanAttributeInfo> getSharedLibraryIdentificationProperties(
            String componentName)
            throws MalformedObjectNameException, AttributeNotFoundException,
            InstanceNotFoundException, MBeanException, ReflectionException,
            IOException, ParserConfigurationException, SAXException {
        return getIdentificationProperties(
                GenericConstants.GET_SHARED_LIBRARY_INSTALLATION_DESCRIPTOR_NAME, 
                componentName);
    }
    
    /**
     * Gets the identification properties of a JBI component or a shared library.
     */
    private Map<Attribute, MBeanAttributeInfo> getIdentificationProperties(
            String operation, String componentName)
            throws MalformedObjectNameException, AttributeNotFoundException,
            InstanceNotFoundException, MBeanException, ReflectionException,
            IOException, ParserConfigurationException, SAXException {
        
        Map<Attribute, MBeanAttributeInfo> map = 
                new HashMap<Attribute, MBeanAttributeInfo>();
        
        String metaData = (String) this.invoke(
                objectName,
                operation,
                new String[] {componentName});
        
        if (metaData != null && metaData.startsWith("<?xml")) { // NOI18N
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document doc = builder.parse(new InputSource(new StringReader(metaData)));
            
            NodeList nodeList = doc.getElementsByTagNameNS(
                    "http://www.sun.com/jbi/descriptor/identification", // NOI18N
                    "VersionInfo"); // NOI18N
            if (nodeList.getLength() > 0) {
                Element element = (Element) nodeList.item(0);
                
                String buildNumberLabel =
                        NbBundle.getMessage(AdministrationService.class,
                        "LBL_BUILD_NUMBER"); // NOI18N
                String buildNumberDesc =
                        NbBundle.getMessage(AdministrationService.class,
                        "DSC_BUILD_NUMBER"); // NOI18N
                
                String buildNumber = element.getAttribute("build-number"); // NOI18N
                if (buildNumber != null) {
                    Attribute attr = new Attribute(buildNumberLabel, buildNumber);
                    MBeanAttributeInfo info = new MBeanAttributeInfo(
                            buildNumberLabel,
                            "java.lang.String", // NOI18N
                            buildNumberDesc,
                            true, false, false);
                    map.put(attr, info);
                }
                
                String specVersionLabel =
                        NbBundle.getMessage(AdministrationService.class,
                        "LBL_SPECIFICATION_VERSION"); // NOI18N
                String specVersionDesc =
                        NbBundle.getMessage(AdministrationService.class,
                        "DSC_SPECIFICATION_VERSION"); // NOI18N
                
                String specVersion = element.getAttribute("specification-version"); // NOI18N
                if (specVersion != null) {
                    Attribute attr = new Attribute(specVersionLabel, specVersion);
                    MBeanAttributeInfo info = new MBeanAttributeInfo(
                            specVersionLabel,
                            "java.lang.String", // NOI18N
                            specVersionDesc,
                            true, false, false);
                    map.put(attr, info);
                }
            }
        }
    
        return map;
    }
    
    
    /**
     * Gets the logger properties of a JBI component (service engine or
     * binding component).
     */  
    public Map<Attribute, MBeanAttributeInfo> getComponentLoggerProperties(
            String componentName)
            throws MalformedObjectNameException, AttributeNotFoundException,
            InstanceNotFoundException, MBeanException, ReflectionException,
            IOException, ParserConfigurationException, SAXException {
        Map<Attribute, MBeanAttributeInfo> map = 
                new HashMap<Attribute, MBeanAttributeInfo>();
        
         this.invoke(
                objectName,
                "GenericConstants.GET_COMPONENT_LOGGER_LEVELS_NAME", // NOI18N
                new String[] {componentName, SERVER_TARGET, null},
                new String[] {"java.lang.String", "java.lang.String", "java.lang.String"}); // NOI18N
                
        // a Map of loggerCustomName to their log levels
        Object output = this.invoke(
                objectName,
                GenericConstants.GET_COMPONENT_LOGGER_LEVELS_NAME,
                new String[] {componentName, SERVER_TARGET, null},
                new String[] {"java.lang.String", "java.lang.String", "java.lang.String"}); // NOI18N
        
        Map<String, Level> loggerMap;
        
        if (output instanceof Map) {
            loggerMap = (Map<String, Level>) output;           
            
            for (String loggerCustomName : loggerMap.keySet()) {
                Level logLevel = loggerMap.get(loggerCustomName);
                int lastDotIndex = loggerCustomName.lastIndexOf("."); // NOI18N
                String shortName = lastDotIndex == -1 ?
                    loggerCustomName : loggerCustomName.substring(lastDotIndex + 1);
                
                Attribute attr = new Attribute(loggerCustomName, logLevel);
                MBeanAttributeInfo info = new MBeanAttributeInfo(
                        shortName,
                        "java.util.logging.Level", // NOI18N
                        loggerCustomName,
                        true, true, false);
                map.put(attr, info);
            }
        }
        
        return map;
    }
    
    /**
     * Sets a logger property of a JBI component (service engine or
     * binding component).
     */  
    public void setComponentLoggerProperty(
            String componentName, String loggerName, Object value)
            throws MalformedObjectNameException, AttributeNotFoundException,
            InstanceNotFoundException, MBeanException, ReflectionException,
            IOException, ParserConfigurationException, SAXException {
        
        Object[] params = new Object[] {
            componentName, loggerName, value, SERVER_TARGET, null};
        String[] signature = getSignatures(params);
        signature[4] = "java.lang.String"; // NOI18N
        this.invoke(objectName,
                GenericConstants.SET_COMPONENT_LOGGER_LEVEL_NAME,
                params, signature);
    }
       
    /**
     * Translates the given string using environment variables from JBI runtime.
     * 
     * @param string         the string to be translated
     * @param componentType  JBI component type: binding component or service engine
     * @param componentName  JBI component name
     * 
     * @return the translated string
     * @throws java.lang.Exception 
     */
    public String translateEnvrionmentVariables(String string, 
            String componentType, String componentName) throws Exception {
        if (string.indexOf("${") == -1 || string.indexOf("}") == -1) {
            return string;
        }
                
        System.out.println(); // TMP        
        System.out.println("Translating environment variables:");
        String oldString = string;        
        
        Map<String, String> map = getEnvrionmentVariables(this.serverConnection,
                componentType, componentName);
        //System.out.println(map);
        
        while (string.indexOf("${") != -1 && string.indexOf("}") != -1) {             
            int startIndex = string.indexOf("${");
            int endIndex = string.indexOf("}");
            String tokenName = string.substring(startIndex + 2, endIndex);
            String tokenValue = map.get(tokenName);
            if (tokenValue != null) {
                string = string.substring(0, startIndex) + tokenValue +
                        string.substring(endIndex + 1);
            } else {
                throw new RuntimeException("Unknown environment variable '" + 
                        tokenName + "' in '" + oldString + "'.");
            }
        }
        
        System.out.println("    " + oldString + " => " + string);
        return string;
    }
    
    /**
     * Gets environment variable name-value pairs for a particular JBI component.
     * 
     * @param connection    an MBeanServerConnection
     * @param componentType JBI component type: binding component or service engine
     * @param componentName JBI component name
     * 
     * @return a map mapping environment viariable names to values
     */
    private static Map<String, String> getEnvrionmentVariables(
            MBeanServerConnection connection,
            String componentType,
            String componentName) throws Exception {
        
        Map<String, String> map = new HashMap<String, String>();
        
        JBIComponentConfigurator configurator =
                new JBIComponentConfigurator(componentType, componentName, connection);
        Properties properties = configurator.getProperties();
        TabularDataSupport tabularData =
                (TabularDataSupport) properties.get(ENVIRONMENT_VARIABLES);
        if (tabularData != null) {
            for (CompositeDataSupport rowData : (Collection<CompositeDataSupport>)tabularData.values()) {
                String name = (String) rowData.get(ENVIRONMENT_VARIABLE_NAME);
                String value = (String) rowData.get(ENVIRONMENT_VARIABLE_VALUE);
                map.put(name, value);
            }
        }
        
        return map;
    }
    
    /**
     * Gets the component configurator for a given JBI Component.
     * 
     * @param type  JBI component type: binding component or service engine
     * @param name  JBI component name
     * 
     * @return a JBI component configurator
     */
    public JBIComponentConfigurator getComponentConfigurator(String type, String name) {
        JBIComponentConfigurator configurator = null;
        try {
            configurator = new JBIComponentConfigurator(type, name, serverConnection);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return configurator;
    }
    
    
    public static void main(String[] args) {        
        String hostName = "127.0.0.1"; // NOI18N
        String port = "4848"; // NOI18N
        port = "5651";
        String userName = "admin"; // NOI18N
        String password = "adminadmin"; // NOI18N
        
        MBeanServerConnection connection = null;
        
        try {
            HTTPServerConnector connector = new HTTPServerConnector(hostName, port,
                    userName, password, null);
            connection = connector.getConnection();
            test(connection);
            
        } catch (Exception exception) {
            exception.printStackTrace();
        }    
    }
    
    public static void test(MBeanServerConnection connection) {
        if(connection != null) {
            System.out.println("Connection Retrieved."+connection.toString()); // NOI18N
            try {
                AdministrationService adminService = new AdministrationService(connection);
//                adminService.printOut();
//                adminService.constructDocumentObject();
//                JBIServiceAssemblyDocument assemblyDocument = adminService.getJBIServiceAssemblyDocument();
//                assemblyDocument.dump();
//                
//                JBIComponentDocument document = adminService.getJBIComponentDocument();
//                document.dump();
                
                //testing SunSequencingEngine
                String componentName = "com.sun.bpelse-1.0-2"; // NOI18N
                String result = ""; // NOI18N
                
                result = adminService.shutdownComponent(componentName, false);
                System.out.println("shutdownComponent result: "+result); // NOI18N
                
                result = adminService.startComponent(componentName);
                System.out.println("startComponent result: "+result); // NOI18N
                
                
                result = adminService.stopComponent(componentName);
                System.out.println("stopComponent result: "+result); // NOI18N
                
                
            } catch (Exception ex) {
                ex.printStackTrace();
                System.out.println("Connection Failed "+ex.getMessage()); // NOI18N
            }
        } else {
            System.out.println("Connection Failed"); // NOI18N
        }
    }
    
}