/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.jdbc.builder;


import java.net.URL;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.DriverPropertyInfo;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;
import java.util.Vector;
import java.util.logging.Logger;

/**
 * Extracts database metadata information (table names and constraints, their 
 * associated columns, etc.)
 *
 * @author Susan Chen
 * @version $Revision: 1.4 $
 */

class DriverShim implements Driver {
    private Driver driver;
    DriverShim(Driver d) {
      this.driver = d;
    }
    public boolean acceptsURL(String u) throws SQLException {
      return this.driver.acceptsURL(u);
    }
    public Connection connect(String u, Properties p) throws SQLException {
      return this.driver.connect(u, p);
    }
    public int getMajorVersion() {
      return this.driver.getMajorVersion();
    }
    public int getMinorVersion() {
      return this.driver.getMinorVersion();
    }
    public DriverPropertyInfo[] getPropertyInfo(String u, Properties p) throws SQLException {
      return this.driver.getPropertyInfo(u, p);
    }
    public boolean jdbcCompliant() {
      return this.driver.jdbcCompliant();
    }
}

public class DBMetaData {
   // constants
   
   /** Index to the name field for results of table/view/procedure searches */
   public static final int NAME = 0;
   
   /** Index to the catalog field for results of table/view/procedure searches */
   public static final int CATALOG = 1;
   
   /** Index to the schema field for results of table/view/procedure searches */
   public static final int SCHEMA = 2;
   
   /** Index to the type field for results of table/view/procedure searches */
   public static final int TYPE = 3;
   
   /** Database OTD type for DB2 */
   public static final String DB2 = "DB2"; // NOI18N
   
   /** Database OTD type for Oracle */
   public static final String ORACLE = "ORACLE"; // NOI18N
   
   /** Database OTD type for Axion */
   public static final String AXION = "AXION"; // NOI18N
   
   /** Database OTD type for Derby */
   public static final String DERBY = "DERBY"; // NOI18N   
   
   /** Database OTD type for PostgreSQSL */
   public static final String PostgreSQL = "PostgreSQL"; // NOI18N  

   /** Database OTD type for MySQL */
   public static final String MYSQL = "MYSQL"; // NOI18N

   /** Database OTD type for SQL Server */
   public static final String SQLSERVER = "SQLSERVER"; // NOI18N
   
   /** Database OTD type for JDBC */
   public static final String JDBC = "JDBC"; // NOI18N
   
   /** Database OTD type for VSAM */
   public static final String VSAM_ADABAS_IAM = "LEGACY"; // NOI18N
   
   /** Database OTD type for JDBC-ODBC */
   public static final String JDBC_ODBC = "JDBC"; // NOI18N
   
   /** Database type display description for DB2 */
   public static final String DB2_TEXT = "DB2"; // NOI18N

   /** Database type display description for Oracle */
   public static final String ORACLE_TEXT = "ORACLE"; // NOI18N
   
   /** Database type display description for SQL Server */
   public static final String SQLSERVER_TEXT = "SQL SERVER"; // NOI18N
   
   /** Database type display description for JDBC */
   //public static final String JDBC_TEXT = "JDBC"; // NOI18N
   
   /** Database type display description for VSAM/ADABAS/IAM */
   public static final String VSAM_ADABAS_IAM_TEXT = "VSAM/ADABAS/IAM"; // NOI18N
   
   /** Database type display description for JDBC-ODBC */
   public static final String JDBC_TEXT = "JDBC-ODBC"; // NOI18N
   
   /** List of database type display descriptions */
   public static final String[] DBTYPES = {
       DB2_TEXT, ORACLE_TEXT, SQLSERVER_TEXT, JDBC_TEXT, 
       VSAM_ADABAS_IAM_TEXT, JDBC_TEXT, PostgreSQL
   };

   /** List of Java types */
   public static final String[] JAVATYPES = {
      "boolean", "byte", "byte[]", "double", "float", "int", 
      "java.lang.String", "java.lang.Object", "java.math.BigDecimal", 
      "java.net.URL", "java.sql.Array", 
      "java.sql.Blob", "java.sql.Clob", "java.sql.Date", "java.sql.Ref",  
      "java.sql.Struct", "java.sql.Time", "java.sql.Timestamp", 
      "long", "short"
   };
   
   /** List of JDBC SQL types */
   public static final String[] SQLTYPES = {
       "ARRAY", "BIGINT", "BINARY", "BIT", "BLOB", "BOOLEAN", 
       "CHAR", "CLOB", "DATALINK", "DATE", "DECIMAL", "DISTINCT", 
       "DOUBLE", "FLOAT", "INTEGER", "JAVA_OBJECT", "LONGVARBINARY", 
       "LONGVARCHAR", "NULL", "NUMERIC", "OTHER", "REAL", "REF", "SMALLINT", 
       "STRUCT", "TIME", "TIMESTAMP", "TINYINT", "VARBINARY", "VARCHAR"
   };

    public static final int[] SQLTYPE_CODES = {
        java.sql.Types.ARRAY,
        java.sql.Types.BIGINT,
        java.sql.Types.BINARY,
        java.sql.Types.BIT,
        java.sql.Types.BLOB,
        16, // java.sql.Types.BOOLEAN,
        java.sql.Types.CHAR,
        java.sql.Types.CLOB,
        70, //case java.sql.Types.DATALINK,
        java.sql.Types.DATE,
        java.sql.Types.DECIMAL,
        java.sql.Types.DISTINCT,
        java.sql.Types.DOUBLE,
        java.sql.Types.FLOAT,
        java.sql.Types.INTEGER,
        java.sql.Types.JAVA_OBJECT,
        java.sql.Types.LONGVARBINARY,
        java.sql.Types.LONGVARCHAR,
        java.sql.Types.NULL,
        java.sql.Types.NUMERIC,
        java.sql.Types.OTHER,
        java.sql.Types.REAL,
        java.sql.Types.REF,
        java.sql.Types.SMALLINT,
        java.sql.Types.STRUCT,
        java.sql.Types.TIME,
        java.sql.Types.TIMESTAMP,
        java.sql.Types.TINYINT,
        java.sql.Types.VARBINARY,
        java.sql.Types.VARCHAR
    };


   /** Map SQL type to Java type */
   public static final HashMap SQLTOJAVATYPES = new HashMap();
   static {
       SQLTOJAVATYPES.put("ARRAY", "java.sql.Array"); // NOI18N
       SQLTOJAVATYPES.put("BIGINT", "long"); // NOI18N
       SQLTOJAVATYPES.put("BINARY", "byte[]"); // NOI18N
       SQLTOJAVATYPES.put("BIT", "boolean"); // NOI18N
       SQLTOJAVATYPES.put("BLOB", "java.sql.Blob"); // NOI18N
       SQLTOJAVATYPES.put("BOOLEAN", "boolean"); // NOI18N
       SQLTOJAVATYPES.put("CHAR", "java.lang.String"); // NOI18N
       SQLTOJAVATYPES.put("CLOB", "java.sql.Clob"); // NOI18N
       SQLTOJAVATYPES.put("DATALINK", "java.net.URL"); // NOI18N
       SQLTOJAVATYPES.put("DATE", "java.sql.Date"); // NOI18N
       SQLTOJAVATYPES.put("DECIMAL", "java.math.BigDecimal"); // NOI18N
       SQLTOJAVATYPES.put("DISTINCT", "java.lang.String"); // NOI18N
       SQLTOJAVATYPES.put("DOUBLE", "double"); // NOI18N
       SQLTOJAVATYPES.put("FLOAT", "double"); // NOI18N
       SQLTOJAVATYPES.put("INTEGER", "int"); // NOI18N
       SQLTOJAVATYPES.put("JAVA_OBJECT", "java.lang.Object"); // NOI18N
       SQLTOJAVATYPES.put("LONGVARBINARY", "byte[]"); // NOI18N
       SQLTOJAVATYPES.put("LONGVARCHAR", "java.lang.String"); // NOI18N
       SQLTOJAVATYPES.put("NULL", "java.lang.String"); // NOI18N
       SQLTOJAVATYPES.put("NUMERIC", "java.math.BigDecimal"); // NOI18N
       SQLTOJAVATYPES.put("OTHER", "java.lang.String"); // NOI18N
       SQLTOJAVATYPES.put("REAL", "float"); // NOI18N
       SQLTOJAVATYPES.put("REF", "java.sql.Ref"); // NOI18N
       SQLTOJAVATYPES.put("SMALLINT", "short"); // NOI18N
       SQLTOJAVATYPES.put("STRUCT", "java.sql.Struct"); // NOI18N
       SQLTOJAVATYPES.put("TIME", "java.sql.Time"); // NOI18N
       SQLTOJAVATYPES.put("TIMESTAMP", "java.sql.Timestamp"); // NOI18N
       SQLTOJAVATYPES.put("TINYINT", "byte"); // NOI18N
       SQLTOJAVATYPES.put("VARBINARY", "byte[]"); // NOI18N
       SQLTOJAVATYPES.put("VARCHAR", "java.lang.String"); // NOI18N       
       //added abey for Procedure ResultSets
       SQLTOJAVATYPES.put("RESULTSET", "java.sql.ResultSet"); // NOI18N
    }

   // String used in java.sql.DatabaseMetaData to indicate system tables.
   private static final String SYSTEM_TABLE = "SYSTEM TABLE"; // NOI18N
   
   // String used in java.sql.DatabaseMetaData to indicate system tables.
   private static final String TABLE = "TABLE"; // NOI18N
   
   // String used in java.sql.DatabaseMetaData to indicate system tables.
   private static final String VIEW = "VIEW"; // NOI18N
   
   private Connection dbconn;           // db connection
   private DatabaseMetaData dbmeta;     // db metadata
  
   /**
    * Gets the primary keys for a table.
    *
    * @param newTable Table to get the primary key(s) for
    * @throws Exception DOCUMENT ME!
    */
   public void checkPrimaryKeys(Table newTable) throws Exception {

       try {
           // get the primary keys
           List primaryKeys = getPrimaryKeys(newTable.getCatalog(), 
                                             newTable.getSchema(), 
                                             newTable.getName());
            
           if (primaryKeys.size() != 0) {
               newTable.setPrimaryKeyColumnList(primaryKeys);
                
               // create a hash set of the keys
               java.util.Set primaryKeysSet = new java.util.HashSet();
               for (int i = 0; i < primaryKeys.size(); i++) {
                   KeyColumn key = (KeyColumn) primaryKeys.get(i);
                   primaryKeysSet.add(key.getColumnName());
               }
                
               // now loop through all the columns flagging the primary keys
               TableColumn[] columns = newTable.getColumns();
               if (columns != null) {
                   for (int i = 0; i < columns.length; i++) {
                       if (primaryKeysSet.contains(columns[i].getName())) {
                           columns[i].setIsPrimaryKey(true);
                       }
                   }
               }
           }
       } catch (Exception e) {
           e.printStackTrace();
  
           throw e;
       }         
   }

   /**
    * Gets the foreign keys for a table.
    *
    * @param newTable Table to get the foreign key(s) for
    * @throws Exception DOCUMENT ME!
    */
   public void checkForeignKeys(Table newTable) throws Exception {

       try {
           // get the foreing keys
           List foreignKeys = getForeignKeys(newTable.getCatalog(), 
                                             newTable.getSchema(), 
                                             newTable.getName());
           if (foreignKeys != null) {
               newTable.setForeignKeyColumnList(foreignKeys);
                
               // create a hash set of the keys
               java.util.Set foreignKeysSet = new java.util.HashSet();
               for (int i = 0; i < foreignKeys.size(); i++) {
                   ForeignKeyColumn key = (ForeignKeyColumn) foreignKeys.get(i);
                   foreignKeysSet.add(key.getColumnName());
               }
                
               // now loop through all the columns flagging the foreign keys
               TableColumn[] columns = newTable.getColumns();
               if (columns != null) {
                   for (int i = 0; i < columns.length; i++) {
                       if (foreignKeysSet.contains(columns[i].getName())) {
                           columns[i].setIsForeignKey(true);
                       }
                   }
               }
           }

       } catch (Exception e) {
           e.printStackTrace();
           throw e;
       }
   }

    private static Logger mLogger = Logger.getLogger("STC.eWay.DB.JDBC." + DBMetaData.class.getName());

   /**
    * Establishes a connection to the database.
    *
    * @param driver Driver class
    * @param url JDBC connection URL
    * @param userName User name
    * @param passWord Password
    * @throws Exception DOCUMENT ME!
    */
    public void connectDB(String driverClassName, String connectionURL, String userName, String passWord, URL[] jarFileURLs) throws Exception {


        // connect to the database
        try 
          {
            /** QAI 77126.
             * For Type 2 Drivers which require loading DLLs it will be documented
             * to leave the JarFileURLs empty in the wizard and copy the driver jars to the 
             * netbeans directory to prevent errors thrown when loading dll's twice.
             * We use ContextClassLoader to load those jars, which will use Netbeans classLoader.
             *  If the user specifies a jar file location, URLClassLoader will be used 
             * (which will link to SystemClassLoader).
             */ 
            if (jarFileURLs != null) {
                // DriverManager will only use driver classes loaded by the 
                // system ClassLoader.  We create a DriverShim class which is 
                // automatically loaded by system ClassLoader.  We
                // instantiate DriverShim class using the driver loaded by 
                // URLClassLoader.  We then register the DriverShim object
                // with DriverManager.  This way we can pick a JDBC driver at 
                // runtime without having to add the driver jar file to the class
                // path.
                URLClassLoader urlClassLoader = new URLClassLoader(jarFileURLs);;
                Driver d = (Driver)Class.forName(driverClassName, true, urlClassLoader).newInstance();
                DriverManager.registerDriver(new DriverShim(d));
            } else {
                Driver d = (Driver) Thread.currentThread().getContextClassLoader().loadClass(driverClassName).newInstance();
                DriverManager.registerDriver(new DriverShim(d));
            }

            if (userName != null && passWord != null) {
                dbconn = DriverManager.getConnection(connectionURL, userName, passWord);
            }
            else {
                dbconn = DriverManager.getConnection(connectionURL);
            }

            // once we've connected, get the metadata
            getDBMetaData();
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
            throw e;
        } catch (Exception e) {
            e.printStackTrace();

            throw e;
        }
    }
   /**
    * Establishes a connection to the database.
    *
    * @param conn JDBC connection
    * @throws Exception DOCUMENT ME!
    */
    public void connectDB(Connection conn) throws Exception {

        if (conn == null)
            throw new NullPointerException("Connection can't be null.");
 
        dbconn = conn;
        getDBMetaData();
    }
    
   /**
    * Disconnects from the database.
    *
    * @throws Exception DOCUMENT ME!
    */
    public void disconnectDB() throws Exception {

        // close connection to database
        try {
            if ((dbconn != null) && (!dbconn.isClosed())) {
                dbconn.close();
                dbconn = null;
            }
        } catch (SQLException e) {
            e.printStackTrace();

            throw e;
        }
    }
 
    private void getDBMetaData() throws Exception {

        // get the metadata
        try {
            dbmeta = dbconn.getMetaData();
        } catch (SQLException e) {
            e.printStackTrace();

            throw e;
        }
    }
    
    /** 
     * Returns the database product name
     *
     * @return String database product name
     * @throws Exception DOCUMENT ME!
     */
    public String getDBName() throws Exception {
        String dbname = "";
        

        // get the database product name
        try {
            dbname = dbmeta.getDatabaseProductName();
        } catch (SQLException e) {
            e.printStackTrace();

            throw e;
        }
        return dbname;
    }
    
    /**
     * Returns the database OTD type.
     *
     * @return String Database OTD type
     * @throws Exception DOCUMENT ME!
     */
    public String getDBType() throws Exception {
        String dbtype = "";
        
        // get the database type based on the product name converted to lowercase
        String dbname = getDBName().toLowerCase();
        if (dbname.equals("microsoft sql server")) {
            // Microsoft SQL Server
            dbtype = SQLSERVER;
        } else if ((dbname.equals("sql server")) || (dbname.indexOf("jdbc") > -1)) {
            // JDBC
            dbtype = JDBC;
        } else if ((dbname.indexOf("db2") > -1) || (dbname.equals("as"))) {
            // DB2
            dbtype = DB2;
        } else if ((dbname.equals("exadas")) || (dbname.equals("attunity connect driver"))) {
            // VSAM
            dbtype = VSAM_ADABAS_IAM;
        } else if (dbname.indexOf("orac") > -1) {
            // Oracle
            dbtype = ORACLE;
        } else if (dbname.indexOf("axion") > -1) {
            // Axion
            dbtype = AXION;
        } else if (dbname.indexOf("apache derby") > -1) {
            // Derby
            dbtype = DERBY;
        } else if (dbname.indexOf("postgresql") > -1) {
            // PostgreSQL
            dbtype = PostgreSQL;
        }else if (dbname.indexOf("mysql") > -1) {
            // MySQL
            dbtype = MYSQL;
		} else {
            // other type, default to JDBC-ODBC
            dbtype = JDBC_ODBC;
        }
        
        return dbtype;
    }

    
    private String getJDBCSearchPattern(String guiPattern) throws Exception {

        
        // Converts the passed in GUI pattern to one understood by the
        // JDBC driver:
        //   change _ to <escape char>_
        //   change % to <escape char>%
        //   change * to % = GUI uses * to represent 0 or more characters
        //   change ? to _ = GUI uses ? to represent any single character
        try {
            String jdbcPattern = guiPattern;
            String escapeChar = dbmeta.getSearchStringEscape();
            
            // change _ to <escape char>_
            //PP:See bug 10718. Disabling the escape character for _
            //jdbcPattern = replaceAllChars(jdbcPattern, '_', escapeChar + "_");

            // change % to <escape char>%
            jdbcPattern = replaceAllChars(jdbcPattern, '%', escapeChar + "%");
            
            // change * to %
            jdbcPattern = jdbcPattern.replace('*', '%');
            
            // change ? to _
            jdbcPattern = jdbcPattern.replace('?', '_');
            
            return jdbcPattern;
        } catch (Exception e) {
            e.printStackTrace();

            throw e;
        }
    }

    /**
     * Returns a list of schemas in the database.
     *
     * @return String[] List of schema names
     * @throws Exception DOCUMENT ME!
     */
    public String[] getSchemas() throws Exception {

        // get all schemas
        try {
            ResultSet rs = dbmeta.getSchemas();
            Vector v = new Vector();
            String[] schemaNames = null;
            
            while (rs.next()) {
                String schema = rs.getString("TABLE_SCHEM");
                v.add(schema);
            }
            
            if (v.size() > 0) {
                // copy into array to return
                schemaNames = new String[v.size()];
            	v.copyInto(schemaNames);
            }
            rs.close();
            return schemaNames;
        } catch (Exception e) {
            e.printStackTrace();

            throw e;
        }
    }
   
    /**
     * Returns a list of tables matching in the passed in filters.
     *
     * @param catalog Catalog name
     * @param schemaPattern Schema pattern
     * @param tablePattern Table name pattern
     * @param includeSystemTables Indicate whether to include system tables in search
     * @return String[][] List of tables matching search filters
     * @throws Exception DOCUMENT ME!
     */
    public String[][] getTablesOnly(String catalog, String schemaPattern, 
                                    String tablePattern, boolean includeSystemTables) 
            throws Exception {
        String[] tableTypes;
        
        if (includeSystemTables) {
            String[] types = {TABLE, SYSTEM_TABLE};
            tableTypes = types;
        } else {
            String[] types = {TABLE};
            tableTypes = types;
        }
        
        return getTables(catalog, schemaPattern, tablePattern, tableTypes);
    }
    
    /**
     * Returns a list of views matching in the passed in filters.
     *
     * @param catalog Catalog name
     * @param schemaPattern Schema pattern
     * @param viewPattern View name pattern
     * @param includeSystemTables Indicate whether to include system tables in search
     * @return String[][] List of views matching search filters
     * @throws Exception DOCUMENT ME!
     */    
    public String[][] getViewsOnly(String catalog, String schemaPattern, 
                                   String viewPattern, boolean includeSystemTables) 
            throws Exception {
        String[] tableTypes;
        
        if (includeSystemTables) {
            String[] types = {VIEW, SYSTEM_TABLE};
            tableTypes = types;
        } else {
            String[] types = {VIEW};
            tableTypes = types;
        }

        return getTables(catalog, schemaPattern, viewPattern, tableTypes);
    }

    /**
     * Returns a list of tables and views matching in the passed in filters.
     *
     * @param catalog Catalog name
     * @param schemaPattern Schema pattern
     * @param tablePattern Table/View name pattern
     * @param includeSystemTables Indicate whether to include system tables in search
     * @return String[][] List of tables and views matching search filters
     * @throws Exception DOCUMENT ME!
     */
    public String[][] getTablesAndViews(String catalog, String schemaPattern, 
                                        String tablePattern, 
                                        boolean includeSystemTables) 
            throws Exception {
        String[] tableTypes;
        
        if (includeSystemTables) {
            String[] types = {TABLE, VIEW, SYSTEM_TABLE};
            tableTypes = types;
        } else {
            String[] types = {TABLE, VIEW};
            tableTypes = types;
        }
        
        return getTables(catalog, schemaPattern, tablePattern, tableTypes);
    }
    
    /**
     * Returns a list of tables/views matching in the passed in filters.
     *
     * @param catalog Catalog name
     * @param schemaPattern Schema pattern
     * @param tablePattern Table/View name pattern
     * @param tableTypes List of table types to include (ex. TABLE, VIEW)
     * @return String[][] List of tables matching search filters
     * @throws Exception DOCUMENT ME!
     */
    public String[][] getTables(String catalog, String schemaPattern, 
                                String tablePattern, String[] tableTypes) 
            throws Exception {

        try {
            if (catalog.equals("")) {
                catalog = null;
            }
            if (schemaPattern.equals("")) {
                schemaPattern = null;
            }
            if (tablePattern.equals("")) {
                tablePattern = null;
            }
            
            if (tablePattern != null) {
                tablePattern = getJDBCSearchPattern(tablePattern);
            }
            
            ResultSet rs = dbmeta.getTables(catalog, schemaPattern, tablePattern, tableTypes);
            
            Vector v = new Vector();
            String[][] tables = null; // array of table structures: Name, Catalog, Schema
            
            while (rs.next()) {
                String tableCatalog = rs.getString("TABLE_CAT");
                String tableSchema = rs.getString("TABLE_SCHEM");
                String tableName = rs.getString("TABLE_NAME");
                String tableType = rs.getString("TABLE_TYPE");
                
                if (tableCatalog == null) {
                    tableCatalog = "";
                }
                if (tableSchema == null) {
                    tableSchema = "";
                }
                
                // fill in table info
                String[] tableItem = new String[4];    // hold info for each table
                tableItem[NAME] = tableName;
                tableItem[CATALOG] = tableCatalog;
                tableItem[SCHEMA] = tableSchema;
                tableItem[TYPE] = tableType;
                
                // add table to Vector
                v.add(tableItem);
            }
            
            // now copy Vector to array to return back
            if (v.size() > 0) {
                tables = new String[v.size()][4];
                v.copyInto(tables);
            }
            rs.close();
            return tables;
        } catch (Exception e) {
            e.printStackTrace();

            throw e;
        }
    }

    /**
     * Returns a list of primary keys for a table.
     *
     * @param tcatalog Catalog name
     * @param tschema Schema name
     * @param tname Table name
     * @return List List of primary keys
     * @throws Exception DOCUMENT ME!
     */
    public List getPrimaryKeys(String tcatalog, String tschema, String tname)
            throws Exception {
        List pkList = Collections.EMPTY_LIST;
        ResultSet rs = null;
                

        try {
            if (tcatalog.equals("")) {
                tcatalog = null;
            }
            if (tschema.equals("")) {
                tschema = null;
            }
            
            rs = dbmeta.getPrimaryKeys(tcatalog, tschema, tname);
            pkList = KeyColumn.createPrimaryKeyColumnList(rs);
        } catch (Exception e) {
            e.printStackTrace();

            throw e;
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                     /* Ignore */;   
                }
            }
        }
        
        return pkList;
    }

    /**
     * Returns a list of foreign keys for a table.
     *
     * @param tcatalog Catalog name
     * @param tschema Schema name
     * @param tname Table name
     * @return List List of foreign keys
     * @throws Exception DOCUMENT ME!
     */
    public List getForeignKeys(String tcatalog, String tschema, String tname) 
            throws Exception {

        List fkList = Collections.EMPTY_LIST;
        ResultSet rs = null;
        
        try {  
            if (tcatalog.equals("")) {
                tcatalog = null;
            }
            if (tschema.equals("")) {
                tschema = null;
            }
            try {
                rs = dbmeta.getImportedKeys(tcatalog, tschema, tname);
                fkList = ForeignKeyColumn.createForeignKeyColumnList(rs);
            }
            catch(Exception e) {
            	e.printStackTrace();
            	mLogger.warning("JDBC driver does not support java.sql.ParameterMetaData " + e.getMessage());
            }
        } catch (Exception e) {
            e.printStackTrace();

            throw e;
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                     /* Ignore */;   
                }
            }
        }
            
        return fkList;
    }

    /**
     * Gets the table metadata (columns).
     *
     * @param tcatalog Catalog name
     * @param tschema Schema name
     * @param tname Table name
     * @param ttype Table type
     * @return Table object
     * @throws Exception DOCUMENT ME!
     */
    public Table getTableMetaData(String tcatalog, String tschema, String tname, String ttype) 
            throws Exception {

        ResultSet rs = null;

        try {  
            // create a new Table object
            Table newTable = new Table(tname, tcatalog, tschema, ttype);
            Vector v = new Vector();
            
            if (tcatalog.equals("")) {
                tcatalog = null;
            }

            if (tschema.equals("")) {
                tschema = null;
            }
            
            // get table column information
            rs = dbmeta.getColumns(tcatalog, tschema, tname, "%");
            
            TableColumn[] columns = null;
            
            while (rs.next()) {
                String defaultValue = rs.getString("COLUMN_DEF");

                int sqlTypeCode = rs.getInt("DATA_TYPE");
                
                String colName = rs.getString("COLUMN_NAME");
                String sqlType = getSQLTypeDescription(sqlTypeCode);
               
                int position = rs.getInt("ORDINAL_POSITION");
                
                int scale = rs.getInt("DECIMAL_DIGITS");
                int precision = rs.getInt("COLUMN_SIZE");
                int radix = rs.getInt("NUM_PREC_RADIX");
                
                // create a table column and add it to the vector
                TableColumn col = new TableColumn(colName, null);
                boolean isNullable = false;
                if (rs.getString("IS_NULLABLE").equals("YES")) {
                    isNullable = true;
                }
                col.setSqlType(sqlType);
                col.setIsNullable(isNullable);
                col.setIsSelected(true);
                col.setIsPrimaryKey(false);
                col.setIsForeignKey(false);
                col.setSqlTypeCode(sqlTypeCode);
                
                col.setOrdinalPosition(position);
                col.setNumericPrecision(precision);
                col.setNumericScale(scale);
                col.setNumericRadix(radix);

                if (defaultValue != null) {
                    col.setDefaultValue(defaultValue.trim());
                }
                
                // add to vector
                v.add(col);
            }
            
            // now copy Vector to array
            if (v.size() > 0) {
                columns = new TableColumn[v.size()];
                v.copyInto(columns);
            }
            
            // now set up columns in the table to return
            newTable.setColumns(columns);
            
            // now check the columns that are primary keys
            checkPrimaryKeys(newTable);
            
            // now check the columns that are foreign keys
            checkForeignKeys(newTable);

            // catch exceptions for this as index only makes sense for
            // tables and not views (can't check the table type because it's dependent on driver)
            try {
               // get index info for this table
               rs = dbmeta.getIndexInfo(tcatalog, tschema, tname, false, true);
               newTable.setIndexList(IndexColumn.createIndexList(rs));
            } catch (Exception e) {
               // ignore and continue
   
            }
            
            return newTable;
        } catch (Exception e) {
            e.printStackTrace();

            throw e;
        } finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {
                    /* Ignore... */;
                }
            }
        }
    }
    
    /** 
     * Converts the numeric value of a JDBC SQL type to 
     * a display string.
     *
     * @param type JDBC numeric SQL type value
     * @return JDBC SQL type string
     */
    public static String getSQLTypeDescription(int type) {
        // returns a String representing the passed in numeric
        // SQL type
        switch (type) {
            case java.sql.Types.ARRAY:
                return "ARRAY";
            case java.sql.Types.BIGINT:
                return "BIGINT";
            case java.sql.Types.BINARY:
                return "BINARY";
            case java.sql.Types.BIT:
                return "BIT";
            case java.sql.Types.BLOB:
                return "BLOB";
            case 16:
                //case java.sql.Types.BOOLEAN:
                return "BOOLEAN";
            case java.sql.Types.CHAR:
                return "CHAR";
            case java.sql.Types.CLOB:
                return "CLOB";
            case 70:
                //case java.sql.Types.DATALINK:
                return "DATALINK";
            case java.sql.Types.DATE:
                return "DATE";
            case java.sql.Types.DECIMAL:
                return "DECIMAL";
            case java.sql.Types.DOUBLE:
                return "DOUBLE";
            case java.sql.Types.FLOAT:
                return "FLOAT";
            case java.sql.Types.INTEGER:
                return "INTEGER";
            case java.sql.Types.JAVA_OBJECT:
                return "JAVA_OBJECT";
            case java.sql.Types.LONGVARBINARY:
                return "LONGVARBINARY";
            case java.sql.Types.LONGVARCHAR:
                return "LONGVARCHAR";
            case java.sql.Types.NULL:
                return "NULL";
            case java.sql.Types.NUMERIC:
                return "NUMERIC";
            case java.sql.Types.OTHER:
                return "OTHER";
            case java.sql.Types.REAL:
                return "REAL";
            case java.sql.Types.REF:
                return "REF";
            case java.sql.Types.SMALLINT:
                return "SMALLINT";
            case java.sql.Types.STRUCT:
                return "STRUCT";
            case java.sql.Types.TIME:
                return "TIME";
            case java.sql.Types.TIMESTAMP:
                return "TIMESTAMP";
            case java.sql.Types.TINYINT:
                return "TINYINT";
            case java.sql.Types.VARBINARY:
                return "VARBINARY";
            case java.sql.Types.VARCHAR:
                return "VARCHAR";
        }
        // all others default to OTHER
        return "OTHER";
    }

    /** 
     * Converts a text representation of a JDBC SQL type to 
     * a display string.
     *
     * @param sqlText JDBC SQL type string
     * @return JDBC numeric SQL type value
     */
    public static int getSQLTypeCode(String sqlText) {
        if (sqlText == null) {
            throw new IllegalArgumentException(
                "Must supply non-null String value for sqlText.");
        }
        
        sqlText = sqlText.trim().toUpperCase();
        for (int i = 0; i < SQLTYPES.length; i++) {
            if (SQLTYPES[i].equals(sqlText)) {
                return SQLTYPE_CODES[i];
            }
        }
        
        return java.sql.Types.OTHER;
    }
    
    private String replaceAllChars(String orig, char oldChar, String replStr) {
        String newString = "";
        
        for (int i = 0; i < orig.length(); i++) {
            if (orig.charAt(i) == oldChar) {
                newString = newString + replStr;
            } else {
                newString = newString + orig.charAt(i);
            }
        }
        return newString;
    }

}

