/*
 *  Copyright (c) 2005, 2006 Imola Informatica.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the LGPL License v2.1
 *  which accompanies this distribution, and is available at
 *  http://www.gnu.org/licenses/lgpl.html
 */


package it.imolinfo.jbi4corba.netbeansplugin.idl2wsdlwizard;

import com.ibm.wsdl.util.xml.DOMUtils;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import javax.swing.JDialog;
import it.imolinfo.jbi4corba.exception.ClassGenerationException;
import it.imolinfo.jbi4corba.jbi.wsdl.Jbi4CorbaExtension;
import it.imolinfo.jbi4corba.netbeansplugin.idl2wsdlwizard.wsdl.Jbi4CorbaPartnerLink;
import it.imolinfo.jbi4corba.netbeansplugin.idl2wsdlwizard.wsdl.Jbi4CorbaPartnerLinkExtension;
import it.imolinfo.jbi4corba.netbeansplugin.idl2wsdlwizard.wsdl.Role;
import it.imolinfo.jbi4corba.webservice.generator.WSDLDescriptor;
import it.imolinfo.jbi4corba.webservice.generator.WSDLGenerator;
import java.io.FileWriter;
import java.util.Iterator;
import java.util.Map;
import java.util.logging.Logger;
import javax.wsdl.Definition;
import javax.wsdl.WSDLException;
import javax.wsdl.extensions.ExtensionRegistry;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLWriter;
import javax.xml.namespace.QName;
import org.openide.DialogDescriptor;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.nodes.Node;
import org.openide.util.HelpCtx;
import org.openide.util.NbBundle;
import org.openide.util.actions.CookieAction;

/**
 * Action to create WSDL file from a IDL file.
 *
 * @author <a href="mailto:mcimatti@imolinfo.it">Marco Cimatti</a>
 */
public final class CreateWSDLAction extends CookieAction {

    // the logger
    private java.util.logging.Logger log = 
        Logger.getLogger(this.getClass().getName());

    /**
     * Creates a new instance of this class.
     */
    public CreateWSDLAction() {
    }

    /**
     * Performs the action based on the currently activated nodes. Note that if
     * the source of the event triggering this action was itself a node, that
     * node will be the sole argument to this method, rather than the activated
     * nodes.
     *
     * @param  activatedNodes  current activated nodes, may be empty but not
     *                         <code>null</code>.
     */
    protected void performAction(final Node[] activatedNodes) {
        IDLDataObject c = (IDLDataObject)
                          activatedNodes[0].getCookie(IDLDataObject.class);
        final FileObject f = c.getPrimaryFile();
        final CreateWSDLPanel panel = new CreateWSDLPanel();
        DialogDescriptor descriptor = new DialogDescriptor(panel,
                NbBundle.getMessage(CreateWSDLAction.class,
                "LBL_CreateWSDLWindow"), true, new Object[] {
                panel.getCreateWSDLButton(), panel.getCancelButton() }, null, 0,
                null, new ActionListener() {
            public void actionPerformed(final ActionEvent e) {
                if (e.getSource() == panel.getCreateWSDLButton()) {
                    WSDLDescriptor desc = panel.createWSDLDescriptor();

                    try {
                        if (createWSDLfromIDLFile(f, desc)) {

                            // Close the JDialog that contains 'panel'
                            ((JDialog) panel.getTopLevelAncestor()).dispose();
                        }
                    } catch (IOException ex) {
                        ExceptionVisualizer.visualizeException(ex);
                    } catch (ClassGenerationException ex) {
                        ExceptionVisualizer.visualizeException(ex);
                    } catch (WSDLException ex) {
                        ExceptionVisualizer.visualizeException(ex);
                    } catch (Exception ex) {
                        ExceptionVisualizer.visualizeException(ex);
                    }
                } else {

                    // Close the JDialog that contains 'panel'
                    ((JDialog) panel.getTopLevelAncestor()).dispose();
                }
            }
        });
        DialogDisplayer displayer = DialogDisplayer.getDefault();
        JDialog dialog = (JDialog) displayer.createDialog(descriptor);

        dialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
        dialog.pack();
        dialog.setVisible(true);
    }

    /**
     * Shows an error dialog containing a generic error message and a specific
     * cause.
     *
     * @param  cause  the specific cause of the generic error.
     */
    private static void showErrorMessage(final String cause) {
        String msg = NbBundle.getMessage(CreateWSDLAction.class,
                                         "MSG_ErrorCreatingWsdl", cause);
        NotifyDescriptor nd = new NotifyDescriptor.Message(
                                        msg, NotifyDescriptor.ERROR_MESSAGE);

        DialogDisplayer.getDefault().notify(nd);
    }

    /**
     * Get the mode of the action: how strict it should be about cookie support.
     *
     * @return  the mode of the action. Possible values are disjunctions of the
     *          <code>MODE_XXX</code> constants.
     */
    protected int mode() {
        return CookieAction.MODE_EXACTLY_ONE;
    }

    /**
     * Get a human presentable name of the action. This may be presented as an
     * item in a menu.
     * <p>
     * Using the normal menu presenters, an included ampersand before a letter
     * will be treated as the name of a mnemonic.
     *
     * @return  the name of the action.
     */
    public String getName() {
        return NbBundle.getMessage(CreateWSDLAction.class,
                                   "CTL_CreateWSDLAction");
    }

    /**
     * Get the cookies that this action requires. The cookies are disjunctive,
     * i.e. a node must support AT LEAST ONE of the cookies specified by this
     * method.
     *
     * @return  a list of cookies.
     */
    protected Class[] cookieClasses() {
        return new Class[] { IDLDataObject.class };
    }

    /**
     * Specify the proper resource name for the action's icon. Typically this
     * should be a 16x16 color GIF. Do not use relative paths nor an initial
     * slash. If e.g. myIcon.gif is accompanied with myIcon_pressed.gif,
     * myIcon_disabled.gif and/or myIcon_rollover.gif these images are used to
     * call methods on <code>JButton.setPressedIcon()</code>,
     * <code>JButton.setDisabledIcon()</code> and/or
     * <code>JButton.setRolloverIcon()</code> with appropriate images. If you do
     * not want an icon, do not override this to return a blank icon. Leave it
     * <code>null</code>, but call <code>putValue("noIconInMenu",
     * Boolean.TRUE)</code> to make sure that no extra space is allotted for an
     * icon in the menu item.
     *
     * @return  the resource name for the icon, e.g.
     *          <code>com/mycom/mymodule/myIcon.gif</code>, or <code>null</code>
     *          to have no icon (make a text label).
     */
    protected String iconResource() {                           // Overridden
        return "it/imolinfo/jbi4corba/netbeansplugin/idl2wsdlwizard/"
               + "idl16x16.png";
    }

    /**
     * Get a help context for the action.
     *
     * @return  the help context for this action.
     */
    public HelpCtx getHelpCtx() {
        return HelpCtx.DEFAULT_HELP;
    }

    /**
     * If <code>true</code>, this action should be performed asynchronously in a
     * private thread. If <code>false</code>, it will be performed synchronously
     * as called in the event thread.
     *
     * @return  <code>false</code>.
     */
    protected boolean asynchronous() {                          // Overridden
        return false;
    }


    private boolean createWSDLfromIDLFile(final FileObject idlFile,
            final WSDLDescriptor desc)
    throws IOException, ClassGenerationException, WSDLException, Exception 
    {
        WSDLGenerator generator = WSDLGenerator.getWSDLGenerator();
        File idl = FileUtil.toFile(idlFile);
        String wsdlFileName = idl.getParent() + File.separatorChar
                             + desc.getCorbaServiceName() + ".wsdl";

        File wsdl = new File(wsdlFileName);
        Thread thread;
        ClassLoader origClassLoader;
        Definition definition;

        if (wsdl.exists()) {
            String msg = NbBundle.getMessage(CreateWSDLAction.class,
                                    "MSG_FileAlreadyExists", wsdl.getName());
            NotifyDescriptor nd = new NotifyDescriptor.Message(
                                    msg, NotifyDescriptor.ERROR_MESSAGE);

            DialogDisplayer.getDefault().notify(nd);
            return false;
        }

        /*
         * FIXME Patch to use CPY2WSDL and IDL2WSDL plugins together: we change
         *       the current thread context ClassLoader, because NetBeans set it
         *       as the "system class loader", able to load classes from any
         *       enabled module, as well as the JDK and all startup libraries
         */
        thread = Thread.currentThread();
        origClassLoader = thread.getContextClassLoader();
        try {
            thread.setContextClassLoader(WSDLGenerator.class.getClassLoader());
            definition = generator.generateWSDLfromIDL(idl, desc);

            // adding the partner link to the generated file
            if (definition != null) {
                 definition = addPartnerLinkToWSDL(definition, desc.getCorbaServiceName());
            }

            // TO FIX: when an exception occurs in the partner link generation,
            // the file is still created, but without the partnel link element

        } finally {
            thread.setContextClassLoader(origClassLoader);
        }

        if (definition == null) {
            showErrorMessage("???");
        }

        // creates the wsdl...
        boolean ret = createFileFromWSDL(definition, wsdl);
        
        return ret;
    }

    /**
     * Adds the deafult partner-link to the wsdl
     */ 
    private Definition addPartnerLinkToWSDL(Definition definition, String corbaServiceName) throws WSDLException, IOException {
       

        Jbi4CorbaPartnerLink ptnlnk = new Jbi4CorbaPartnerLink();        

        // check if name space is present
        String prefix = definition.getPrefix(Jbi4CorbaPartnerLinkExtension.NS_PTNLNK);
        if (prefix == null) {
            // add the namespace
            definition.addNamespace(Jbi4CorbaPartnerLinkExtension.NS_PREFIX, Jbi4CorbaPartnerLinkExtension.NS_PTNLNK);
            prefix = Jbi4CorbaPartnerLinkExtension.NS_PREFIX;
        }

        log.fine("The used prefix: " + prefix);

        // use the existing prefix
        ptnlnk.setPrefix(prefix);

        // set the partner link name as the corba service name
        ptnlnk.setName(corbaServiceName);

        // set the required to true
        ptnlnk.setRequired(true);

        // set the element type
        ptnlnk.setElementType(Jbi4CorbaPartnerLinkExtension.Q_ELEMENT_PTNLNK);

        // isolating the portType
        Map portTypes = definition.getAllPortTypes();
        Role role = null;
        if (portTypes.size() > 0) {
            Iterator iterKeys = portTypes.keySet().iterator();
            Iterator iterValues = portTypes.values().iterator();
            do {
                QName key = (QName) iterKeys.next();
                javax.wsdl.PortType value = (javax.wsdl.PortType) iterValues.next();

                // find the target name space
                String tns = value.getQName().getNamespaceURI();
                if (tns == null) {
                    // Impossible: the tns must be defined
                    String msg = NbBundle.getMessage(CreateWSDLAction.class, "MSG_TargetNamespaceNotFound");
                    NotifyDescriptor nd = new NotifyDescriptor.Message(msg, NotifyDescriptor.ERROR_MESSAGE);

                    DialogDisplayer.getDefault().notify(nd);
                    log.severe(msg);
                    return null;
                }
                role = new Role(corbaServiceName + Jbi4CorbaPartnerLinkExtension.ROLE_NAME_SUFFIX, DOMUtils.getPrefix(tns, definition) + ":" + value.getQName().getLocalPart());
                ptnlnk.addRole(role);
                portTypes.remove(key);
            } while (!portTypes.isEmpty());
        }
        
        // adds the extension               
        definition.addExtensibilityElement(ptnlnk);

        return definition;
    }
    
    /**
     * Creates the wsdl file from the <code>Definition</code>.
     */ 
    private boolean createFileFromWSDL(Definition definition, File wsdlFile) throws WSDLException, IOException {
        boolean result = false;
        
            FileWriter fw = null;

            try {
                WSDLFactory factory = WSDLFactory.newInstance();   
                // creating the registry for the WSDL4J Extension
                ExtensionRegistry registry = factory.newPopulatedExtensionRegistry();
                Jbi4CorbaPartnerLinkExtension.register(registry);
                Jbi4CorbaExtension.register(registry);
                definition.setExtensionRegistry(registry);

                // creating the writer
                WSDLWriter writer = factory.newWSDLWriter();

                // creating the new File
                fw = new FileWriter(wsdlFile);

                // write the file
                writer.writeWSDL(definition, fw);

                result = true;
            } finally {
                if (fw != null) {
                    fw.close();
                }
            }
            
        return result;
    }

}
