/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */ 
package org.netbeans.microedition.testme.bt;
        
import java.io.IOException;
import javax.bluetooth.*;
import java.util.Vector;

/**
 * Provides basic bluetooth features like device and service discovery. Isolates
 * upper layers from the bluetooth stack. Uses the singleton design pattern.
 *
 * @author PavelBenes
 */
final class BluetoothManager {
    public static final UUID UUID_SERIAL = new UUID(0x1101);

    private static BluetoothManager s_instance = null;

    public static synchronized BluetoothManager instance() throws BluetoothStateException {
        if ( s_instance == null) {
            s_instance = new BluetoothManager();
        }
        return s_instance;
    }

    private LocalDevice m_localDevice;

    public static interface ServiceComparator {
        boolean accept(ServiceRecord record) throws Exception;
    }

    private abstract class AbstractBluetoothFinder implements DiscoveryListener {
        protected boolean m_discovered;
        protected boolean m_completed;

        protected void search() {
            m_discovered = false;

            try{
                startSearch();
                startWait();
            }catch(BluetoothStateException e){
                throw new RuntimeException("Discovery Error");
            } catch (InterruptedException e) {
                e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
            }
        }

        private synchronized void startWait() throws InterruptedException  {
            m_completed = false;
            try {
                this.wait();
            } finally {
                m_completed = true;
            }
        }

        protected synchronized void endWait(boolean cancelInquiry) {
            if (!m_completed) {
                m_completed = true;
                this.notifyAll();
                if (cancelInquiry) {
                    m_localDevice.getDiscoveryAgent().cancelInquiry(this);
                }
            }
        }

        public void serviceSearchCompleted(int i, int i1) {
            //System.out.println( new Date(System.currentTimeMillis()) + "Service search completed.");
            endWait(false);
        }

        public void inquiryCompleted(int i) {
            //System.out.println( new Date(System.currentTimeMillis()) + "Inquiry completed.");
            endWait(false);
        }

        public void deviceDiscovered(RemoteDevice remoteDevice, DeviceClass deviceClass) {
        }

        public void servicesDiscovered(int i, ServiceRecord[] serviceRecords) {
        }

        protected abstract void startSearch() throws BluetoothStateException;
    }

    /*
     * In Series 40 2nd Edition devices, calling RemoteDevice.getFriendlyName() during
     * device discovery or service search will throw a java.io.IOException every time.
     */
    private final class BluetoothDeviceFinder extends AbstractBluetoothFinder {
        private       Vector m_devices = new Vector();

        public Vector find() {
            m_devices.removeAllElements();
            search();
            return m_devices;
        }

        public void deviceDiscovered(RemoteDevice remoteDevice, DeviceClass deviceClass) {
            m_devices.addElement(remoteDevice);
        }

        protected void startSearch() throws BluetoothStateException {
            m_localDevice.getDiscoveryAgent().startInquiry(0x9e8b33, this);
        }
    }

    private class BluetoothServiceFinder extends AbstractBluetoothFinder {
        protected final RemoteDevice m_remoteDevice;
        protected final UUID []      m_uuids;
        protected       Vector       m_services = new Vector();

        public BluetoothServiceFinder( final RemoteDevice remoteDevice, final UUID [] uuids) {
            m_remoteDevice = remoteDevice;
            m_uuids        = uuids;
        }

        public Vector find() {
            m_services.removeAllElements();
            search();
            return m_services;
        }

        public void servicesDiscovered(int p, ServiceRecord[] serviceRecords) {
            //BlueDict.logMessage("Services found: " + serviceRecords.length);
            for (int i = 0; i < serviceRecords.length; i++) {
                m_services.addElement( serviceRecords[i]);
            }
        }

        protected void startSearch() throws BluetoothStateException {
            m_localDevice.getDiscoveryAgent().searchServices(null, new UUID[] { UUID_SERIAL}, m_remoteDevice, this);
        }
    }
    
    private BluetoothManager() throws BluetoothStateException {
        initializeStack();
    }

    public LocalDevice getLocalDevice() {
        return m_localDevice;
    }
    
    public RemoteDevice findRemoteDevice(final String friendlyName) {
        BluetoothDeviceFinder finder = new BluetoothDeviceFinder();
        Vector devicesFound = finder.find();
        for (int i = devicesFound.size() - 1; i >= 0; i--) {
            RemoteDevice remote = (RemoteDevice) devicesFound.elementAt(i);
            try {
                String fName = remote.getFriendlyName(false);
                if ( fName != null && fName.equals(friendlyName)) {
                    return remote;
                }
            } catch( IOException e) {
                // nothing to do
            }
        }
        return null;
    }

    public RemoteDevice [] findRemoteDevices() {
        BluetoothDeviceFinder finder = new BluetoothDeviceFinder();
        Vector list = finder.find();
        RemoteDevice [] devices = new RemoteDevice[list.size()];
        list.copyInto(devices);
        return devices;
    }
    
    public ServiceRecord [] findServices( final RemoteDevice device, UUID [] uuids) {
        BluetoothServiceFinder finder = new BluetoothServiceFinder(device, uuids);
        Vector list = finder.find();
        ServiceRecord [] services = new ServiceRecord[list.size()];
        list.copyInto(services);
        return services;
    }

    public String selectService(UUID uuid) throws BluetoothStateException {
        return m_localDevice.getDiscoveryAgent().selectService( uuid, ServiceRecord.NOAUTHENTICATE_NOENCRYPT,  false);
    }
    
    /**
     * Helper method for safe retrieval of remote device friendly name.
     */
    public static String getName(RemoteDevice remote) {
        try {
            return remote.getFriendlyName(false);
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }
    
    public static String getURL(ServiceRecord service) {
        return service.getConnectionURL(ServiceRecord.NOAUTHENTICATE_NOENCRYPT, false);        
    }
    
    public static String getDescription(String name, String address) {
        StringBuffer sb = new StringBuffer();
        
        if (name != null) {
            sb.append(name);
        }
        
        if (address != null) {
            if (sb.length() == 0) {
                sb.append(address);
            } else {
                sb.append('[');
                sb.append(address);
                sb.append(']');
            }
        }
        return sb.toString();
    }
    
    private void initializeStack() throws BluetoothStateException {
        m_localDevice = LocalDevice.getLocalDevice();
        m_localDevice.setDiscoverable(DiscoveryAgent.GIAC);
    }
}
