/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.j2ee.websphere6;

import java.io.*;
import java.util.*;
import java.lang.reflect.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.enterprise.deploy.model.*;
import javax.enterprise.deploy.model.exceptions.*;
import javax.enterprise.deploy.shared.*;

import org.openide.*;
import org.netbeans.modules.j2ee.deployment.devmodules.spi.*;

import org.netbeans.modules.j2ee.websphere6.util.WSDebug;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;

/**
 * Wrapper for j2eeserver's <code>DeployableObject</code> implementation.
 * This class serves as a wrapper for j2eeserver's <code>DeployableObject</code>
 * implementation that fixes its incompartibility with the JSR-88 specification.
 * The j2eeserver's implementation does not implement the 
 * <code>getEntry()</code> and <code>entries()</code> methods, while these are
 * used intesively by the servers' classes.
 * 
 * @author Kirill Sorokin
 */
public class WSDeployableObject implements DeployableObject {
    
    /**
     * The original j2eeserver's <code>DeployableObject</code> implementation,
     * all operations except <code>getEntry()</code> and <code>entries</code>
     * are delegated to it
     */
    DeployableObject deployableObject;
    
    /**
     * The module provider that we fetch from the deployable object in order to
     * implement the <code>getEntry()</code> and <code>entries</code> methods
     */
    J2eeModuleProvider provider;
    
    /**
     * Map of <code>DDBeanRoot</code>s associated with this deployable object
     */
    Map ddBeanRoots = new HashMap();
    
    /**
     * Creates a new instance of <code>WSDeployableObject</code>.
     * 
     * @param deployableObject the original j2eeserver's 
     *      <code>DeployableObject</code> implementation
     */
    public WSDeployableObject(DeployableObject deployableObject) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "WSDeployableObject(" +         // NOI18N
                    deployableObject + ")");                           // NOI18N
        
        // save the supplied deployable object
        this.deployableObject = deployableObject;
        
        // fetch the J2eeModuleProvider handle
        try {
            Method method = deployableObject.getClass().
                    getMethod("getProvider", new Class[0]);            // NOI18N
            this.provider = (J2eeModuleProvider) method.
                    invoke(deployableObject, new Object[0]);
        } catch (IllegalAccessException e) {
            Logger.getLogger("global").log(Level.WARNING, null, e);
        } catch (NoSuchMethodException e) {
            Logger.getLogger("global").log(Level.WARNING, null, e);
        } catch (InvocationTargetException e) {
            Logger.getLogger("global").log(Level.WARNING, null, e);
        }
    }
    
    /**
     * Finds a plugin's wrapper for the supplied j2eeserver's 
     * <code>DDBEanRoot</code> implemetation.
     * 
     * @param bean j2eeserver's implementation that is used as a key during 
     *      search
     */
    public DDBeanRoot findDDBeanRoot(DDBeanRoot bean) {
        // get all registered wrappers
        Collection values = ddBeanRoots.values();
        
        // iterate over the collection and check whether any of them is the
        // wrapper for the supplied key
        for (Iterator iterator = values.iterator(); iterator.hasNext();) {
            WSDDBeanRoot wsBean = (WSDDBeanRoot) iterator.next();
            
            // if the wrapper's origin is the key - return it
            if (wsBean.getOrigin().equals(bean)) {
                return wsBean;
            }
        }
        
        // if no wrappers found return null
        return null;
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object. Returns a wrapper for the result value.
     */
    public DDBeanRoot getDDBeanRoot(String str) throws FileNotFoundException, 
            DDBeanCreateException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getDDBeanRoot(" + str + ")");  // NOI18N
        
        // if there is no such wrapper registered already - create a new one 
        // basing on the return value from the j2eeserver's deployable object
        if (ddBeanRoots.get(str) == null) {
            ddBeanRoots.put(str, new WSDDBeanRoot(deployableObject.
                    getDDBeanRoot(str), this));
        }
        
        // return the registered wrapper for this string
        return (DDBeanRoot) ddBeanRoots.get(str);
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object. Returns a wrapper for the result value.
     */
    public DDBeanRoot getDDBeanRoot() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getDDBeanRoot()");             // NOI18N
        
        // if there is no wrapper for the blank string registered already - 
        // create a new one basing on the return value from the j2eeserver's 
        // deployable object
        if (ddBeanRoots.get("") == null) {                             // NOI18N
            ddBeanRoots.put("", new WSDDBeanRoot(deployableObject.     // NOI18N
                    getDDBeanRoot(), this));
        }
        
        // return the registered wrapper for the blank string
        return (DDBeanRoot) ddBeanRoots.get("");                       // NOI18N
    }

    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public String[] getText(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getText(" + str + ")");        // NOI18N
        
        return deployableObject.getText(str);
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public Class getClassFromScope(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getClassFromScope(" + str +    // NOI18N
                    ")");                                              // NOI18N
        
        return deployableObject.getClassFromScope(str);
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public DDBean[] getChildBean(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getChildBean(" + str + ")");   // NOI18N
        
        return deployableObject.getChildBean(str);
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public ModuleType getType() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getType()");                   // NOI18N
        
        return deployableObject.getType();
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public String getModuleDTDVersion() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getModuleDTDVersion()");       // NOI18N
        
        return deployableObject.getDDBeanRoot().getDDBeanRootVersion();
    }

    /**
     * Returns an <code>Enumeration</code> that contains all the source files 
     * that are be included in the deployable object. While not being 
     * completely to the spec which requires the compiled files to be included
     * here, it solves the issue of the method not being implemented, as the 
     * servers are most interested in configuration files which are not affected 
     * by compilation.
     * 
     * @return an <code>Enumeartion</code> with all the source files of the 
     *      project
     */
    public Enumeration entries() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "entries()");                   // NOI18N
        
        // return a new Enumeration built by WSEntries object
        return new WSEntries(provider).getEntries();
    }
    
    /**
     * Returns an open <code>InputStream</code> to a named deployable object 
     * entry.
     * 
     * @param str the name of the entry
     * @return an <code>InputStream</code> to the named entry, null if the 
     *      entry cannot be found
     */
    public InputStream getEntry(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getEntry(" + str + ")");       // NOI18N
        
        // construct a new WSEntries object and request the entry
        InputStream in = new WSEntries(provider).getEntry(str);
        
        // return the input stream
        return in;
    }
    
    /**
     * Returns a File handle for the named deployment object entry.
     * 
     * @param str the name of the entry
     * @return a File handle for the named entry, null if the entry cannot be 
     *      found    
     */
    public File getEntryFile(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getEntryFile(" + str + ")");   // NOI18N
        
        // construct a new WSEntries object and request the entry
        for (FileObject fo: provider.getSourceRoots()) {
            File file = new File(FileUtil.toFile(fo), str);
            
            if (file.exists()) {
                if (WSDebug.isEnabled()) // debug output
                    WSDebug.notify(getClass(), " -- " + file);   // NOI18N
                
                return file;
            }
        }
        
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), " -- null");   // NOI18N
        return null;
    }
    
    /**
     * A class that is responsible for performing all operation on deployable 
     * object entries, i.e. returning an <code>Enumeration</code> with all the
     * files of the project and returning a named entry.
     * 
     * @author Kirill Sorokin
     */
    private static class WSEntries {
        
        /**
         * Handle for the <code>J2eeModuleProvider</code> of the parent
         * deployable object
         */
        private J2eeModuleProvider provider;
        
        /**
         * Creates a new instance of <code>WSEntries</code>.
         * 
         * @param provider the J2eeModuleProvider of the parent deployable
         *      object
         */
        public WSEntries (J2eeModuleProvider provider) {
            this.provider = provider;
        }
        
        /**
         * Returns an input stream to the named entry
         * 
         * @param str entry the entry name
         * @return an InputStream to the entry's file
         */
        public InputStream getEntry(String entry) {
            JarFile jar = null;
            try {
                File archive = FileUtil.toFile(provider.getJ2eeModule().getArchive());
                if (WSDebug.isEnabled())
                    WSDebug.notify(WSEntries.class, archive.getPath());
                
                jar = new JarFile(archive);
                
                return jar.getInputStream(jar.getEntry(entry));
            } catch (IOException e) {
                Logger.getLogger("global").log(Level.WARNING, null, e);
            } 
//            finally {
//                if (jar != null) {
//                    try {
//                        jar.close();
//                    } catch (IOException e) {
//                        ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, e);
//                    }
//                }
//            }
            
            return null;
        }
        
        /**
         * Returns an Enumeration with all the entries of the parent deployable 
         * object
         *
         * @return Enumeration with the entries
         */
        public Enumeration getEntries() {
            Vector entries = new Vector();
            
            JarFile jar = null;
            try {
                File archive = FileUtil.toFile(provider.getJ2eeModule().getArchive());
                
                if (WSDebug.isEnabled())
                    WSDebug.notify(WSEntries.class, archive.getPath());
                
                jar = new JarFile(archive);
                Enumeration<JarEntry> jarEntries = jar.entries();
                
                while (jarEntries.hasMoreElements()) {
                    String name = jarEntries.nextElement().getName();
                    
                    if (!name.endsWith("/")) {
                        if (WSDebug.isEnabled())
                            WSDebug.notify(WSEntries.class, " -- " + name);
                        entries.add(name);
                    }
                }
                
                return entries.elements();
            } catch (Exception ex) {
                WSDebug.notify(ex);
            } finally {
                if (jar != null) {
                    try {
                        jar.close();
                    } catch (IOException e) {
                        Logger.getLogger("global").log(Level.WARNING, null, e);
                    }
                }
            }
            
            return null;
        }
    }
}