// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/cg_array_access.cpp,v 1.5 2001/12/08 08:55:31 zying1 Exp $
//


#include "defines.h"
#include "jit_intf.h"
#include "jit_runtime_support.h"
#include <iostream.h>
#include <stdarg.h>
#include "code_emitter.h"
#include "operand.h"
#include "register_manager.h"
#include "stack.h"
#include "profiling.h"
#include "lazy_code_selector.h"
#include "cg_array_access.h"

#ifndef NO_BOUNDS_CHECKING
#include "bounds_checking.h"
#endif // NO_BOUNDS_CHECKING

#include "register_allocator.h"
#include "fp_compatibility.h"


static int nearest_encompassing_try_block_index(       // 1-based
                                                Method_Handle method_handle, unsigned bc_index) {
    unsigned cEH = method_get_num_handlers(method_handle);
    unsigned found = (unsigned)-1, Beg = 0, End = (unsigned)-1;
    for(unsigned e=0; e<cEH; e++) {
        unsigned tryBegOffsPtr, tryEndOffsPtr, handlerOffsPtr, handlerTypePtr;
        method_get_handler_info(method_handle,e, &tryBegOffsPtr, &tryEndOffsPtr,
                                      &handlerOffsPtr, &handlerTypePtr);
        if (bc_index>=tryBegOffsPtr && bc_index<=tryEndOffsPtr)
            if (Beg < tryBegOffsPtr) {
                found = e;
                Beg = tryBegOffsPtr;
            }
            if (End > tryEndOffsPtr) {
                found = e;
                End = tryEndOffsPtr;
            }
    }
    return (found==(unsigned)-1)? 0 : found+1+cEH;
}

static void emit_array_index(Code_Emitter& emitter,Stack& stack,
                             Operand*& array, Operand*& index) {
    Reg_Operand *dst;
    //
    // keep array in a reg.
    // it is safe to overwrite array's caller-save register if array is a mem operand
    //
    if (array->is_mem()) {
        array->free_opnd(&stack.reg_manager);
        dst = stack.reg_manager.get_reg();
        //
        // a[idx[i]] = c;
        // array: var,   index: array,    src: var
        //
        if (!dst) {
            if(index->is_mem()) {
                // free up index
                index->free_opnd(&stack.reg_manager);
                Reg_Operand *idx = stack.reg_manager.get_reg();
                index->emit_mov_to_reg(emitter,&idx->opnd);
                index = idx;
            }
            dst = stack.reg_manager.get_reg();
            assert(dst);
        }
        array->emit_mov_to_reg(emitter,&dst->opnd);
        array = dst;
    }
    //
    // load index into a reg if index is a mem operand
    //
    if (index->is_mem()) {
        index->free_opnd(&stack.reg_manager);
        Reg_Operand *idx = stack.reg_manager.get_reg();
        index->emit_mov_to_reg(emitter,&idx->opnd);
        index = idx;
    }
    stack.maintain_precise_exception();
}

static void emit_array_index_for_store64(Code_Emitter& emitter,Stack& stack,
                                         Operand*& array,Operand*& index,
                                         Reg_Operand *callee, 
                                         Stack_Operand *spill_loc) {
    if (spill_loc) {
		//
		// keep array in a reg.
		// it is safe to overwrite array's caller-save register if array is a mem operand
		//
		if (array->is_mem() || array->is_scratch_reg()) {
			array->free_opnd(&stack.reg_manager); 
			array->emit_mov_to_reg(emitter,&callee->opnd);
			array = callee;
		}
		//
		// load index into a reg if index is a mem operand
		//
		if (index->is_mem()) {
			index->free_opnd(&stack.reg_manager);
			Reg_Operand *idx;
			if (array == callee) // callee is used by array
				idx = stack.reg_manager.get_reg();
			else
				idx = callee;
			index->emit_mov_to_reg(emitter,&idx->opnd);
			index = idx;
		}
	} else
		emit_array_index(emitter,stack,array,index);
}

#ifndef NO_BOUNDS_CHECKING
//
// return 1, if the array bound checking can be eliminated
// return 0, otherwise
//
static int eliminate_bounds_checking(Operand *array,
                                     Operand *index,
                                     Stack& stack, 
                                     Jit_Method_Info *method_info,
                                     Bounds_Checking& bounds,
                                     Register_Allocator *regalloc,
                                     const unsigned char *where) {
    // 
	// eliminate bounds checking if possible
	// 
	if (index->kind != Operand::Imm) return 0;
	int val = ((Imm_Operand*)index)->imm_opnd.value;
	//
	// find variable no
	//
	unsigned var_no = -1;
	X86_Reg_No reg_no = n_reg;
	if (array->kind == Operand::Var)
		var_no = ((Mem_Var_Operand*)array)->var_no;
	else if (array->is_callee_reg())
		var_no = 
        regalloc->var_in_reg(((Reg_Operand*)array)->opnd.reg_no(),where)
        ;
	else if (var_no == -1 && array->is_reg())
		reg_no = ((Reg_Operand*)array)->opnd.reg_no();
	//
	// check if we have already check bound that is <= bounds.array_bounds(var_no)
	//
	if(var_no != -1 && val >= 0) {
		if (val <= bounds.array_bound(var_no)) {
			return 1;
		} else
			bounds.record(var_no,val);
	} else if (reg_no != n_reg && val >= 0 && 
		stack.reg_manager.is_newarray_in_reg(reg_no) &&
		val < bounds.scratch_bound(reg_no)) {
		return 1;
	}
	return 0;
}
#endif // NO_BOUNDS_CHECKING

static void make_runtime_throw_record(unsigned code_offset,
                                      unsigned try_index,
                                      Jit_Method_Info *method_info,
                                      Mem_Manager &mem_manager) {
	struct Runtime_Throw_Info *rec =
		(struct Runtime_Throw_Info *)mem_manager.alloc(sizeof(Runtime_Throw_Info));
	rec->offset = code_offset;
	rec->try_index = try_index;
	rec->next = method_info->runtime_throws;
	method_info->runtime_throws = rec;
}

static void emit_array_bound_checking(Mem_Manager& mem_manager,
                                      Code_Emitter& emitter,Operand *array,
                                      Operand *index,
                                      Code_Patch *& code_patch_list,
                                      Method_Handle method_handle,
                                      unsigned bc_index,
                                      Jit_Method_Info *method_info) {

	X86_Reg_No base_no = ((Reg_Operand*)array)->opnd.reg_no();
	int length_offset = array_length_offset();
//YZW	
	//M_Base_Opnd length(base_no,4);
	M_Base_Opnd length(base_no, length_offset);
//YZW
	if (index->is_reg())
		emitter.emit_alu(cmp_opc,&length,&((Reg_Operand*)index)->opnd);
	else
		emitter.emit_alu(cmp_opc,&length,&((Imm_Operand*)index)->imm_opnd);
	//
	// emit a branch with 32-bit offset that is later patched
    //
	//     jbe  L1  //+10
    //     push nearest_encompassing_try_index
    //     call RngChkFail
    // L1:
    //
    unsigned try_index = nearest_encompassing_try_block_index(method_handle,bc_index);
    unsigned precall_IP = emitter.get_offset();
    make_runtime_throw_record(emitter.get_offset(), try_index, method_info, mem_manager);
    // MOVE_RUNTIME_THROWS
	emitter.emit_branch32(cc_le,&Imm_Opnd(0),0);
    //make_esp_record(emitter.get_offset(),1,method_info,mem_manager);
    method_info->cs_info[method_info->cnt].ret_IP = (unsigned)emitter.get_offset();
    method_info->cs_info[method_info->cnt].precall_IP = precall_IP;
    method_info->cs_info[method_info->cnt].outarg_bv = 0;
    method_info->cs_info[method_info->cnt].m_handle = NULL;
    method_info->cs_info[method_info->cnt].returns_ref = 0;
    method_info->cs_info[method_info->cnt++].num_out_args = 0;
#if 1
    // Don't create a call_site_info for this exception throw.  The problem is
    // that the precall_IP is set incorrectly, so for GC, the ORP ends up
    // setting a breakpoint at a bogus point.  --JMS
    method_info->cnt --;
#endif // 1
	//
	// For statistics, inserting instruments
	//
	if(inner_statistics){
		assert(emitter.prof_rec) ;
		inner_bb_instrumenting_code(emitter,
			(unsigned*)&((PROF_COUNTER*)&((unsigned short*)&emitter.prof_rec->back_edge[emitter.prof_rec->n_back_edge])[emitter.prof_rec->n_back_edge])[emitter.inner_bb_cnt_offset++]);
	}
	//
	// For statistics
	//
	b_inner_counter = true ;
}

//-----------------------------------------------------------------------------
// array load/store
//-----------------------------------------------------------------------------
//
// emit_array_load: generate code for {i,l,f,d,a,b,c,s}aload bytecodes
//
void emit_array_load(Code_Emitter& emitter, Stack& stack,
                     Mem_Manager& mem_manager,char ty,
                     Code_Patch *& code_patch_list,
                     Method_Handle method_handle,
                     unsigned bc_index,
                     Jit_Method_Info *method_info,
                     Register_Allocator *regalloc
#ifndef NO_BOUNDS_CHECKING
                     , Bounds_Checking& bounds
#endif // NO_BOUNDS_CHECKING
                     ) {
    Operand *index = stack.pop();
    Operand *array = stack.pop();
    int check_bound = 1;
#ifndef NO_BOUNDS_CHECKING
    check_bound = !eliminate_bounds_checking(array,index,stack,method_info,bounds,
        regalloc,
        &method_info->code[bc_index]);
#endif // NO_BOUNDS_CHECKING
    //
    // pop index and array ref.  Make sure that array ref is in a reg.
    //
    emit_array_index(emitter,stack,array,index);
    //
    // array bound checking if necessary
    //
    if (check_bound) emit_array_bound_checking(mem_manager,emitter,array,index,
        code_patch_list,method_handle,bc_index,method_info);
#ifdef _CSE
    stack.update_cse(index,0);
    stack.update_cse(array,1);
#endif // _CSE
    //
    // access array --- if index is an imm, then we try to fold imm
    //
    Reg_Operand *dst;
    X86_Reg_No base_no = ((Reg_Operand*)array)->opnd.reg_no();
	int array_offset = get_array_offset((Java_Type)ty);
    if (index->kind == Operand::Imm) {
        int i = ((Imm_Operand*)index)->imm_opnd.value;
        switch (ty) {
        case 'B':	// boolean or 8-bit signed
            // movsx eax,[eax+ecx]
            // push eax
            array->free_opnd(&stack.reg_manager);
            dst = stack.reg_manager.get_reg();
//zying1     assert(array_offset == 8);
			emitter.emit_widen(&dst->opnd,&M_Base_Opnd(base_no,i+/*8*/array_offset),1,0); //yzw
            stack.push(dst);
            break;
        case 'C':	// 16-bit unsigned char
        case 'S':	// signed 16-bit short
            i = (i<<1) + /*8*/array_offset;
            array->free_opnd(&stack.reg_manager);
            dst = stack.reg_manager.get_reg();
            emitter.emit_widen(&dst->opnd,&M_Base_Opnd(base_no,i),(ty == 'S'),1);
            stack.push(dst);
            break;
        case 'F':	// single FP
        case 'I':	// 32-bit int
        case 'L':	// object reference
            {
                i = (i << 2) + /*8*/array_offset;
                Field_Operand *fld = new (mem_manager) Field_Operand(base_no,i);
                fld->base_is_not_null();
                stack.push(fld);
            }
            break;
        case 'J':	// 64-bit long
        case 'D':	// double FP
            {
                i = (i << 3) + /*8*/array_offset;
                Field_Operand *lo = new (mem_manager) Field_Operand(base_no,i);
                Field_Operand *hi = new (mem_manager) Field_Operand(base_no,i+4);
                lo->base_is_not_null();
                hi->base_is_not_null();
                stack.push64(lo,hi);
            }
            break;
        }
    } else {
        assert(index->is_reg());
        X86_Reg_No idx_no = ((Reg_Operand*)index)->opnd.reg_no();
        switch (ty) {
        case 'B':	// boolean or 8-bit signed
            // movsx eax,[eax+ecx]
            // push eax
            array->free_opnd(&stack.reg_manager); 
            index->free_opnd(&stack.reg_manager);
            dst = stack.reg_manager.get_reg();
            emitter.emit_widen(&dst->opnd,&M_Index_Opnd(base_no,idx_no,/*8*/array_offset,0),1,0); //yzw
            stack.push(dst);
            break;
        case 'C':	// 16-bit unsigned char
        case 'S':	// signed 16-bit short
            array->free_opnd(&stack.reg_manager); 
            index->free_opnd(&stack.reg_manager);
            dst = stack.reg_manager.get_reg();
            emitter.emit_widen(&dst->opnd,&M_Index_Opnd(base_no,idx_no,/*8*/array_offset,1),(ty == 'S'),1); // yzw
            stack.push(dst);
            break;
        case 'F':	// single FP
        case 'I':	// 32-bit int
        case 'L':	// object reference
            stack.push(new (mem_manager) Array_Operand(base_no,idx_no,/*8*/array_offset,2)); //yzw
            break;
        case 'J':	// 64-bit long
        case 'D':	// double FP
            stack.push64(new (mem_manager) Array_Operand(base_no,idx_no,/*8*/array_offset,3),
                new (mem_manager) Array_Operand(base_no,idx_no,/*12*/array_offset+4,3));
            break;
        }
    }
}

static void need_one_callee_for_store64(Stack& stack,
                                        Pre_Alloc_Operand_Pool& op_pool,
                                        X86_Reg_No& callee, Stack_Operand *&spill_loc,
                                        Operand *src_lo, Operand *src_hi,
                                        Operand *index,  Operand *array) {
    unsigned n_extra_reg = 0;
	// 
	// if src is mem, then we need one extra reg for mem to mem copy
	//
	if (src_lo->is_mem()) {
		n_extra_reg++;
		assert(src_lo->kind == src_hi->kind);
	}
	//
	// we load index/array into a reg.  If index/array has a scratch reg, 
	// we can use the scratch reg to keep the value of the index/array, 
	// e.g., mov  eax, [eax + 4].
	//
	if (index->is_mem() && !index->has_scratch_reg())
		n_extra_reg++;
	if (array->is_mem() && !array->has_scratch_reg())
		n_extra_reg++;

	unsigned reg_use_bv = 0;
	unsigned n_caller = 0;
	n_caller += index->reg_usage(reg_use_bv);
	n_caller += array->reg_usage(reg_use_bv);
	n_caller += src_lo->reg_usage(reg_use_bv);
	if (!src_hi->is_mem())
		n_caller += src_hi->reg_usage(reg_use_bv);
	//
	// determine if we need one extra callee reg or not
	// (3-n_caller) is the number of reg that reg_manager can free
	//
	if (n_extra_reg > (3 - n_caller)) {
		callee = stack.reg_manager.find_one_available_callee(reg_use_bv);
		//
		// find a spill location for the callee-save reg
		//
		if (array->kind != Operand::Stk)
			spill_loc = op_pool.nth_stack(stack.depth());
		else if (index->kind != Operand::Stk)
			spill_loc = op_pool.nth_stack(stack.depth()+1);
		else if (src_lo->kind != Operand::Stk)
			spill_loc = op_pool.nth_stack(stack.depth()+3);
		else spill_loc = NULL;
		assert(spill_loc);
	}
}
static void emit_array_store64(Mem_Manager& mem_manager,
                               Code_Emitter& emitter,Stack& stack,
							   char ty,
                               Pre_Alloc_Operand_Pool& op_pool,
                               Code_Patch *& code_patch_list,
                               Method_Handle method_handle,
                               unsigned bc_index,
                               Jit_Method_Info *method_info,
                               Register_Allocator *regalloc
#ifndef NO_BOUNDS_CHECKING
					    , Bounds_Checking& bounds
#endif // NO_BOUNDS_CHECKING
                        ) {
	Operand *src_lo, *src_hi;
	stack.pop64(src_lo, src_hi);
	Operand *index = stack.pop();
	Operand *array = stack.pop();
	int check_bound = 1;
#ifndef NO_BOUNDS_CHECKING
	check_bound = !eliminate_bounds_checking(array,index,stack,method_info,bounds,
        regalloc,
        &method_info->code[bc_index]);
#endif // NO_BOUNDS_CHECKING

	Stack_Operand *spill_loc = NULL;
	X86_Reg_No callee_r;
	need_one_callee_for_store64(stack,op_pool,callee_r,spill_loc,
								src_lo,src_hi,index,array);
	Reg_Operand callee(callee_r);
	if (spill_loc)
		callee.emit_mov_to_mem(emitter,&spill_loc->opnd);
	//
	// Make sure that array ref and index are in regs.
	//
	emit_array_index_for_store64(emitter,stack,array,index,&callee,spill_loc);
	X86_Reg_No base_no = ((Reg_Operand*)array)->opnd.reg_no();
	
	int array_offset = get_array_offset((Java_Type)ty);
	//
	// array bound checking if necessary
	//
	if (check_bound) emit_array_bound_checking(mem_manager,emitter,array,index,
							  code_patch_list,method_handle,bc_index,method_info);

#ifdef _CSE
	stack.update_cse(index,1);
	stack.update_cse(array,2);
#endif // _CSE
	if (src_lo->is_mem()) {// 32 bit
		Reg_Operand *reg = stack.reg_manager.get_reg();
		int field = 0;
		X86_Reg_No src_base_no = eax_reg;
		if (!reg) {
			assert(src_lo->has_scratch_reg());
			src_lo->free_opnd(&stack.reg_manager);
			Reg_Operand *base = stack.reg_manager.get_reg();
			assert(base);
			src_base_no = base->opnd.reg_no();
			((Mem_Operand*)src_lo)->emit_lea(emitter,&base->opnd);
			reg = stack.reg_manager.get_reg();
			assert(reg);
			field = 1;
		}
		Field_Operand lo32(src_base_no,0);
		Field_Operand hi32(src_base_no,4);
		if (field) {
			src_lo = &lo32;
			src_hi = &hi32;
		}
		if (index->kind == Operand::Imm) {
			int i = ((Imm_Operand*)index)->imm_opnd.value;
			//
			// store low 32 bit
			//
			src_lo->emit_mov_to_reg(emitter,&reg->opnd);
			reg->emit_mov_to_mem(emitter,&M_Base_Opnd(base_no,(i <<3) + /*8*/array_offset)); // yzw
			//
			// store high 32 bit
			//
			src_hi->emit_mov_to_reg(emitter,&reg->opnd);
			reg->emit_mov_to_mem(emitter,&M_Base_Opnd(base_no,(i <<3) + /*12*/array_offset+4)); // yzw
		} else {
			X86_Reg_No idx_no = ((Reg_Operand*)index)->opnd.reg_no();
			src_lo->emit_mov_to_reg(emitter,&reg->opnd);
			reg->emit_mov_to_mem(emitter,&M_Index_Opnd(base_no,idx_no,/*8*/array_offset,3)); // yzw
			src_hi->emit_mov_to_reg(emitter,&reg->opnd);
			reg->emit_mov_to_mem(emitter,&M_Index_Opnd(base_no,idx_no,/*12*/array_offset+4,3)); // yzw
		}
		reg->free_opnd(&stack.reg_manager);

	} else {
		if (index->kind == Operand::Imm) {
			int i = ((Imm_Operand*)index)->imm_opnd.value;
			src_lo->emit_mov_to_mem(emitter,&M_Base_Opnd(base_no,(i <<3) + array_offset)); //yzw
			src_hi->emit_mov_to_mem(emitter,&M_Base_Opnd(base_no,(i <<3) + array_offset+4)); //yzw
		} else {
			X86_Reg_No idx_no = ((Reg_Operand*)index)->opnd.reg_no();
			if (src_lo->kind == Operand::Fp && !stack.fp_strict_mode)
			    stack.fp_dec_cnt();
			src_lo->emit_mov_to_mem(emitter,&M_Index_Opnd(base_no,idx_no,array_offset,3)); //yzw
			src_hi->emit_mov_to_mem(emitter,&M_Index_Opnd(base_no,idx_no,array_offset+4,3)); //yzw
		}
	}
	// restore r if r is a callee-save reg
	if (spill_loc) 
		spill_loc->emit_mov_to_reg(emitter,&callee.opnd);

	array->free_opnd(&stack.reg_manager);
	index->free_opnd(&stack.reg_manager);
	src_lo->free_opnd(&stack.reg_manager);
	src_hi->free_opnd(&stack.reg_manager);
}

static void emit_array_store32(Mem_Manager& mem_manager,
                               Code_Emitter& emitter,Stack& stack,char ty,
                               Code_Patch *& code_patch_list,
                               Method_Handle method_handle,
                               unsigned bc_index,
                               Jit_Method_Info *method_info,
                               Register_Allocator *regalloc
#ifndef NO_BOUNDS_CHECKING
					    , Bounds_Checking& bounds
#endif // NO_BOUNDS_CHECKING
                        ) {
    Operand *src = stack.pop();
    Reg_Operand *reg;
    if (src->is_mem()) {// 32 bit
        src->free_opnd(&stack.reg_manager);
        reg = stack.reg_manager.get_reg();
        src->emit_mov_to_reg(emitter,&reg->opnd);
        src = reg;
#ifdef _CSE
        stack.update_cse(reg,0);
#endif // _CSE
    }
    //
    // pop index and array ref.  Make sure that array ref is in a reg.
    //
    Operand *index = stack.pop();
    Operand *array = stack.pop();
    int check_bound = 1;
#ifndef NO_BOUNDS_CHECKING
    check_bound = !eliminate_bounds_checking(array,index,stack,method_info,bounds,
        regalloc,
        &method_info->code[bc_index]);
#endif // NO_BOUNDS_CHECKING
    emit_array_index(emitter,stack,array,index);
    //
    // array bound checking if necessary
    //   (NOTE: Helper function we call for aastore does it all)
    //
    if (check_bound && ty!='L')
        emit_array_bound_checking(mem_manager,emitter,array,index,
        code_patch_list,method_handle,bc_index,method_info);
    
#ifdef _CSE
    stack.update_cse(index,1);
    stack.update_cse(array,2);
#endif // _CSE
    X86_Reg_No base_no = ((Reg_Operand*)array)->opnd.reg_no();
	int array_offset = get_array_offset((Java_Type)ty);

    if (index->kind == Operand::Imm) {
        int i = ((Imm_Operand*)index)->imm_opnd.value;
        switch (ty) {
        case 'B':	// boolean or 8-bit signed
            // mov8 [ebx+12],eax
            if (src->is_reg() && !((Reg_Operand *)src)->is_byte_reg()) {
                reg = stack.reg_manager.get_byte_reg();
                assert(reg);
                src->emit_mov_to_reg(emitter,&reg->opnd);
                src = reg;
            }
//            src->emit_mov_to_mem(emitter,&M_Base_Opnd(base_no,i+8),opnd_8);
            src->emit_mov_to_mem(emitter,&M_Base_Opnd(base_no,i+array_offset),opnd_8); //yzw

            break;
        case 'C':	// 16-bit unsigned char
        case 'S':	// signed 16-bit short
            src->emit_mov_to_mem(emitter,&M_Base_Opnd(base_no,(i<<1)+/*8*/array_offset),opnd_16);//yzw
            break;
        case 'F':	// single FP
        case 'I':	// 32-bit int
            if (src->kind == Operand::Fp && !stack.fp_strict_mode)
                stack.fp_dec_cnt();
            src->emit_mov_to_mem(emitter,&M_Base_Opnd(base_no,(i<<2)+/*8*/array_offset)); //yzw
            break;
        case 'L': {	// object reference
            void *addr = orp_get_rt_support_addr(ORP_RT_AASTORE);
            //
            // generate ArrStore(src,index,array);
            //
            stack.call_home(0);
            X86_Reg_No array_reg_no = ((Reg_Operand*)array)->opnd.reg_no();
            emitter.emit_push(&R_Opnd(array_reg_no));
            make_esp_record(emitter.get_offset(),1,method_info,mem_manager);
            emitter.emit_push(&Imm_Opnd(i));
            make_esp_record(emitter.get_offset(),2,method_info,mem_manager);
            src->emit_push(emitter);
            make_esp_record(emitter.get_offset(),3,method_info,mem_manager);
            unsigned patch_offset = emitter.get_offset()+1;
            method_info->cs_info[method_info->cnt].precall_IP = (unsigned)emitter.get_offset();
            method_info->cs_info[method_info->cnt].returns_ref = 0;
            emitter.emit_call((char*)addr);
            method_info->cs_info[method_info->cnt].ret_IP = (unsigned)emitter.get_offset();
            method_info->cs_info[method_info->cnt].outarg_bv = 0x5; // outargs 0 and 2 are refs
            method_info->cs_info[method_info->cnt].m_handle = NULL;
            method_info->cs_info[method_info->cnt++].num_out_args = 3;
            code_patch_list =
                new(mem_manager) Call_Patch(code_patch_list,patch_offset,(char*)addr);
            
                  }  break;
        default:
            assert(0);
        }
    } else {
        X86_Reg_No idx_no = ((Reg_Operand*)index)->opnd.reg_no();
        switch (ty) {
        case 'B':	// boolean or 8-bit signed
            // mov8 [ebx+ecx],eax
            if (src->is_reg() && !((Reg_Operand *)src)->is_byte_reg()) {
                reg = stack.reg_manager.get_byte_reg();
                assert(reg);
                src->emit_mov_to_reg(emitter,&reg->opnd);
                src = reg;
            }
            src->emit_mov_to_mem(emitter,&M_Index_Opnd(base_no,idx_no,/*8*/array_offset,0),opnd_8); //yzw
            break;
        case 'C':	// 16-bit unsigned char
        case 'S':	// signed 16-bit short
            src->emit_mov_to_mem(emitter,&M_Index_Opnd(base_no,idx_no,/*8*/array_offset,1),opnd_16); //yzw
            break;
        case 'F':	// single FP
        case 'I':	// 32-bit int
            if (src->kind == Operand::Fp && !stack.fp_strict_mode)
                stack.fp_dec_cnt();
            src->emit_mov_to_mem(emitter,&M_Index_Opnd(base_no,idx_no,/*8*/array_offset,2)); //yzw
            break;
        case 'L': {	// object reference
            void *addr = orp_get_rt_support_addr(ORP_RT_AASTORE);
            //
            // generate ArrStore(src,index,array);
            //
            stack.call_home(0);
            // 			stack.home_all();
            X86_Reg_No array_reg_no = ((Reg_Operand*)array)->opnd.reg_no();
            emitter.emit_push(&R_Opnd(array_reg_no));
            make_esp_record(emitter.get_offset(),1,method_info,mem_manager);
            emitter.emit_push(&R_Opnd(idx_no));
            make_esp_record(emitter.get_offset(),2,method_info,mem_manager);
            src->emit_push(emitter);
            make_esp_record(emitter.get_offset(),3,method_info,mem_manager);
            unsigned patch_offset = emitter.get_offset()+1;
            method_info->cs_info[method_info->cnt].precall_IP = (unsigned)emitter.get_offset();
            method_info->cs_info[method_info->cnt].returns_ref = 0;
            emitter.emit_call((char*)addr);
            method_info->cs_info[method_info->cnt].ret_IP = (unsigned)emitter.get_offset();
            method_info->cs_info[method_info->cnt].outarg_bv = 0x5; // outargs 0 and 2 are refs
            method_info->cs_info[method_info->cnt].m_handle = NULL;
            method_info->cs_info[method_info->cnt++].num_out_args = 3;
            code_patch_list =
                new(mem_manager) Call_Patch(code_patch_list,patch_offset,(char*)addr);
            
            }  break;
        default:
            assert(0);
        }
    }
    array->free_opnd(&stack.reg_manager);
    index->free_opnd(&stack.reg_manager);
    src->free_opnd(&stack.reg_manager);
}

//
// emit_array_store: generate code for {i,l,f,d,a,b,c,s}astore bytecodes
//
void emit_array_store(Mem_Manager& mem_manager,
                      Code_Emitter& emitter,Stack& stack,
					  Pre_Alloc_Operand_Pool& op_pool,char ty,
                      Code_Patch *& code_patch_list,
                      Method_Handle method_handle,
                      Register_Allocator *regalloc,
					  unsigned bc_index,
#ifndef NO_BOUNDS_CHECKING
					  Jit_Method_Info *method_info,
					  Bounds_Checking& bounds) {
	if (ty == 'D' || ty == 'J') {
		emit_array_store64(mem_manager,emitter,stack,ty,op_pool,
                        code_patch_list,method_handle,bc_index,method_info,
                        regalloc,
                        bounds);
    } else
		emit_array_store32(mem_manager,emitter,stack,ty,
                        code_patch_list,method_handle,bc_index,method_info,
                        regalloc,
                        bounds);
#else // NO_BOUNDS_CHECKING
					  Jit_Method_Info *method_info) {
	if (ty == 'D' || ty == 'J') {
		emit_array_store64(mem_manager,emitter,stack,ty,op_pool,
                        code_patch_list,method_handle,bc_index,method_info);
    } else
		emit_array_store32(mem_manager,emitter,stack,ty,
                        code_patch_list,method_handle,bc_index,method_info);
#endif // NO_BOUNDS_CHECKING
}
