// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/jni.cpp,v 1.9 2002/01/09 03:45:25 gwu2 Exp $
//


#include "platform.h"
#include "Class.h"
#include "Class_File_Loader.h"
#include "Class_Loader.h"
#include "environment.h"
#include "orp_utils.h"
#include "orp_threads.h"
#include "orp_synch.h"
#include "root_set_enum.h"
#include "exceptions.h"
#include "ini.h"

#ifndef OBJECT_LOCK_V2
#include "find_natives.h"
#else
#include "find_natives_olv2.h"
#endif


#include "jni.h"
#include "jni_utils.h"
#include "jni_direct.h"

#include "jit_runtime_support.h"

#include "jvmdi_clean.h"

#include "../../base_natives/gnu_classpath/include/gnu_classpath_jni_utils.h"

const struct JNINativeInterface_ jni_vtable = {
            
    NULL,
    NULL,
    NULL,
    NULL,
    GetVersion,
            
    DefineClass,
    FindClass,
    NULL,
    NULL,
    NULL,
    GetSuperclass,
    IsAssignableFrom,
    NULL,
            
    Throw,
    ThrowNew,
    ExceptionOccurred,
    ExceptionDescribe,
    ExceptionClear,
    FatalError,
    NULL,
    NULL,
            
    NewGlobalRef,
    DeleteGlobalRef,
    DeleteLocalRef,
    IsSameObject,
    NULL,
    NULL,
            
    AllocObject,
    NewObject,
    NewObjectV,
    NewObjectA,
            
    GetObjectClass,
    IsInstanceOf,
            
    GetMethodID,
            
    CallObjectMethod,
    CallObjectMethodV,
    CallObjectMethodA,
    CallBooleanMethod,
    CallBooleanMethodV,
    CallBooleanMethodA,
    CallByteMethod,
    CallByteMethodV,
    CallByteMethodA,
    CallCharMethod,
    CallCharMethodV,
    CallCharMethodA,
    CallShortMethod,
    CallShortMethodV,
    CallShortMethodA,
    CallIntMethod,
    CallIntMethodV,
    CallIntMethodA,
    CallLongMethod,
    CallLongMethodV,
    CallLongMethodA,
    CallFloatMethod,
    CallFloatMethodV,
    CallFloatMethodA,
    CallDoubleMethod,
    CallDoubleMethodV,
    CallDoubleMethodA,
    CallVoidMethod,
    CallVoidMethodV,
    CallVoidMethodA,
            
    CallNonvirtualObjectMethod,
    CallNonvirtualObjectMethodV,
    CallNonvirtualObjectMethodA,
    CallNonvirtualBooleanMethod,
    CallNonvirtualBooleanMethodV,
    CallNonvirtualBooleanMethodA,
    CallNonvirtualByteMethod,
    CallNonvirtualByteMethodV,
    CallNonvirtualByteMethodA,
    CallNonvirtualCharMethod,
    CallNonvirtualCharMethodV,
    CallNonvirtualCharMethodA,
    CallNonvirtualShortMethod,
    CallNonvirtualShortMethodV,
    CallNonvirtualShortMethodA,
    CallNonvirtualIntMethod,
    CallNonvirtualIntMethodV,
    CallNonvirtualIntMethodA,
    CallNonvirtualLongMethod,
    CallNonvirtualLongMethodV,
    CallNonvirtualLongMethodA,
    CallNonvirtualFloatMethod,
    CallNonvirtualFloatMethodV,
    CallNonvirtualFloatMethodA,
    CallNonvirtualDoubleMethod,
    CallNonvirtualDoubleMethodV,
    CallNonvirtualDoubleMethodA,
    CallNonvirtualVoidMethod,
    CallNonvirtualVoidMethodV,
    CallNonvirtualVoidMethodA,
            
    GetFieldID,
            
    GetObjectField,
    GetBooleanField,
    GetByteField,
    GetCharField,
    GetShortField,
    GetIntField,
    GetLongField,
    GetFloatField,
    GetDoubleField,
    SetObjectField,
    SetBooleanField,
    SetByteField,
    SetCharField,
    SetShortField,
    SetIntField,
    SetLongField,
    SetFloatField,
    SetDoubleField,
            
    GetStaticMethodID,
            
    CallStaticObjectMethod,
    CallStaticObjectMethodV,
    CallStaticObjectMethodA,
    CallStaticBooleanMethod,
    CallStaticBooleanMethodV,
    CallStaticBooleanMethodA,
    CallStaticByteMethod,
    CallStaticByteMethodV,
    CallStaticByteMethodA,
    CallStaticCharMethod,
    CallStaticCharMethodV,
    CallStaticCharMethodA,
    CallStaticShortMethod,
    CallStaticShortMethodV,
    CallStaticShortMethodA,
    CallStaticIntMethod,
    CallStaticIntMethodV,
    CallStaticIntMethodA,
    CallStaticLongMethod,
    CallStaticLongMethodV,
    CallStaticLongMethodA,
    CallStaticFloatMethod,
    CallStaticFloatMethodV,
    CallStaticFloatMethodA,
    CallStaticDoubleMethod,
    CallStaticDoubleMethodV,
    CallStaticDoubleMethodA,
    CallStaticVoidMethod,
    CallStaticVoidMethodV,
    CallStaticVoidMethodA,
            
    GetStaticFieldID,
            
    GetStaticObjectField,
    GetStaticBooleanField,
    GetStaticByteField,
    GetStaticCharField,
    GetStaticShortField,
    GetStaticIntField,
    GetStaticLongField,
    GetStaticFloatField,
    GetStaticDoubleField,
            
    SetStaticObjectField,
    SetStaticBooleanField,
    SetStaticByteField,
    SetStaticCharField,
    SetStaticShortField,
    SetStaticIntField,
    SetStaticLongField,
    SetStaticFloatField,
    SetStaticDoubleField,
            
    NewString,
    GetStringLength,
    GetStringChars,
    ReleaseStringChars,
            
    NewStringUTF,
    GetStringUTFLength,
    GetStringUTFChars,
    ReleaseStringUTFChars,
            
    GetArrayLength,
            
    NewObjectArray,
    GetObjectArrayElement,
    SetObjectArrayElement,
            
    NewBooleanArray,
    NewByteArray,
    NewCharArray,
    NewShortArray,
    NewIntArray,
    NewLongArray,
    NewFloatArray,
    NewDoubleArray,
            
    GetBooleanArrayElements,
    GetByteArrayElements,
    GetCharArrayElements,
    GetShortArrayElements,
    GetIntArrayElements,
    GetLongArrayElements,
    GetFloatArrayElements,
    GetDoubleArrayElements,
            
    ReleaseBooleanArrayElements,
    ReleaseByteArrayElements,
    ReleaseCharArrayElements,
    ReleaseShortArrayElements,
    ReleaseIntArrayElements,
    ReleaseLongArrayElements,
    ReleaseFloatArrayElements,
    ReleaseDoubleArrayElements,
            
    GetBooleanArrayRegion,
    GetByteArrayRegion,
    GetCharArrayRegion,
    GetShortArrayRegion,
    GetIntArrayRegion,
    GetLongArrayRegion,
    GetFloatArrayRegion,
    GetDoubleArrayRegion,
    SetBooleanArrayRegion,
    SetByteArrayRegion,
    SetCharArrayRegion,
    SetShortArrayRegion,
    SetIntArrayRegion,
    SetLongArrayRegion,
    SetFloatArrayRegion,
    SetDoubleArrayRegion,

    RegisterNatives,
    UnregisterNatives,
            
    MonitorEnter,
    MonitorExit,
            
    GetJavaVM,
};
            
        

static JNIEnv_ jni_env = { &jni_vtable, (void *)0x1234abcd };


//const struct JNINativeInterface_ *jni_native_intf = &jni_vtable;
const struct JNIEnv_ *jni_native_intf = &jni_env;



void jni_init()
{
} //jni_init



jint JNICALL GetVersion(JNIEnv *env)
{
    return 0x10001;
} //GetVersion



jclass JNICALL DefineClass(JNIEnv *env,
                           const char *name,
                           jobject loader,
                           const jbyte *buf,
                           jsize len)
{
    Class_Loader *cl;
    if (loader)
        cl = class_loader_lookup(loader);
    else
        cl = NULL;

    Loader_Result ld_result;
    Loader_Exception ld_exc;
    
    if(name){
		//Replace '.' with '/'
		char *p = (char*)name;
		while(*p){
			if(*p == '.')*p = '/';
			p++;
		}
	}

    orp_disable_gc();       //---------------------------------v

    Class *clss = load_class_from_memory((uint8 *)buf,
                                         0,
                                         len,
                                         name,
                                         cl,
                                         ORP_Global_State::loader_env,
                                         &ld_result,
                                         &ld_exc);

    orp_enable_gc();        //---------------------------------^

    if(clss != NULL) {
        class_prepare(clss);
    }

    if(clss) {
        Object_Handle new_handle = orp_create_local_object_handle();
        new_handle->java_reference = (Java_java_lang_Object *)clss;
        return new_handle;
    } else {
        // The JNI spec says that we should throw one of: ClassFormatError,
        // ClassCircularityError or OutOfMemoryError.  We don't throw
        // OutOfMemoryError (we assume that we can always allocate more
        // memory for a class.
        jclass clazz = 0;
        if(ld_result == LD_OK) {
            switch(ld_exc) {
            case LD_ClassCircularityError:
                clazz = FindClass(env, "java/lang/ClassCircularityError");
                break;
            case LD_ClassFormatError:
            default:
                clazz = FindClass(env, "java/lang/ClassFormatError");
                break;
            }
        } else {
            clazz = FindClass(env, "java/lang/ClassFormatError");
        }
        assert(clazz);
        const char *prefix = "DefineClass failed";
        char *msg;
		if(name){
			int len = strlen(prefix) + strlen(name) + 5;
			msg = (char *)malloc(len);
			sprintf(msg, "%s: '%s'", prefix, name);			
		}else{//DefineClass might have no name set
			int len = strlen(prefix) + 25;
			msg = (char *)malloc(len);
			sprintf(msg, "%s: 'no name'", prefix);			
		}
        jint ok = ThrowNew(env, clazz, msg);
        free(msg);
        env->DeleteLocalRef(clazz);
        assert(ok == 0);
        return 0;
    }
} //DefineClass



jclass JNICALL FindClass(JNIEnv *env,
                         const char *name)
{
    String *s = ORP_Global_State::loader_env->string_pool.lookup(name);
    Loader_Exception ld_exc;
    Class *clss =
        class_load_verify_prepare_from_jni(ORP_Global_State::loader_env, s, &ld_exc);
    if(clss) {
        Object_Handle new_handle = orp_create_local_object_handle();
        new_handle->java_reference = (Java_java_lang_Object *)clss;
        return new_handle;
    } else {
        // The JNI spec says that we should throw one of: ClassFormatError,
        // ClassCircularityError, NoClassDefFoundError or OutOfMemoryError.
        // We don't throw OutOfMemoryError (we assume that we can always
        // allocate more memory for a class.
        jclass clazz;
        switch(ld_exc) {
        case LD_NoClassDefFoundError:
            clazz = FindClass(env, "java/lang/NoClassDefFoundError");
            break;
        case LD_ClassCircularityError:
            clazz = FindClass(env, "java/lang/ClassCircularityError");
            break;
        case LD_IncompatibleClassChangeError:
            clazz = FindClass(env, "java/lang/IncompatibleClassChangeError");
            break;
        case LD_ClassFormatError:
        default:
            clazz = FindClass(env, "java/lang/ClassFormatError");
            break;
        }
        const char *prefix = "FindClass failed";
        int len = strlen(prefix) + strlen(name) + 5;
        char *msg = (char *)malloc(len);
        sprintf(msg, "%s: '%s'", prefix, name);
        jint ok = ThrowNew(env, clazz, msg);
        free(msg);
        env->DeleteLocalRef(clazz);
        assert(ok == 0);
        return 0;
    }
} //FindClass



jclass JNICALL GetSuperclass(JNIEnv *env, jclass clazz)
{
    Class *clss = (Class *)((Object_Handle)clazz)->java_reference;
    if(clss) {
        if(class_is_interface(clss)) {
            return 0;
        } else {
            Class *super_class = clss->super_class;
            if(super_class) {
                Object_Handle new_handle = orp_create_local_object_handle();
                new_handle->java_reference = (Java_java_lang_Object *)super_class;
                return new_handle;
            } else {
                return 0;
            }
        }
    } else {
        return 0;
    }
} //GetSuperclass



jboolean JNICALL IsAssignableFrom(JNIEnv *env,
                                  jclass clazz1,
                                  jclass clazz2)
{
    Class *clss1 = (Class *)((Object_Handle)clazz1)->java_reference;
    Class *clss2 = (Class *)((Object_Handle)clazz2)->java_reference;

    Boolean isAssignable = orp_instanceof_class(clss1, clss2);

    if(isAssignable) {
        return JNI_TRUE;
    } else {
        return JNI_FALSE;
    }
} //IsAssignableFrom



jint JNICALL Throw(JNIEnv *env, jthrowable obj)
{
    if(obj) {
        orp_disable_gc();       //---------------------------------v

        Object_Handle h = (Object_Handle)obj;
        set_current_thread_exception(h->java_reference);

        orp_enable_gc();        //---------------------------------^
        return 0;
    } else {
        return -1;
    }
} // Throw



jint JNICALL ThrowNew(JNIEnv *env, jclass clazz, const char *message)
{
    jstring str = (jstring)0;
	if(message) {
		str = NewStringUTF(env, message);
        if(!str) {
            return -1;
        }
	}

    jmethodID init_id = GetMethodID(env, clazz, "<init>", "(Ljava/lang/String;)V");
    jvalue args[1];
    args[0].l = str;
    jobject obj = NewObjectA(env, clazz, init_id, args);
    env->DeleteLocalRef(str);
    if(obj && !ExceptionOccurred(env)) {
        orp_disable_gc();       //---------------------------------v

        Object_Handle h = (Object_Handle)obj;
        set_current_thread_exception(h->java_reference);

        orp_enable_gc();        //---------------------------------^
        return 0;
    } else {
        return -1;
    }
} //ThrowNew



jint JNICALL ThrowNew_Quick(JNIEnv *env, const char *classname, const char *message)
{ 
    //??jclass exclazz = FindClass(env, classname);
    // wgs: I wonder if context class loader is always correct here, if not, must always
	//  let current class loader be transferred as a parameter
    jclass exclazz = FindClassWithContextClassLoader(classname);
    if(!exclazz) {
        return -1;
    }

    jint result = ThrowNew(env, exclazz, message);
    env->DeleteLocalRef(exclazz);
    return result;
} // ThrowNew_Quick



jthrowable JNICALL ExceptionOccurred(JNIEnv *env)
{
    if(get_current_thread_exception()) {
        orp_disable_gc();       //---------------------------------v

        Object_Handle new_handle = orp_create_local_object_handle();
        new_handle->java_reference = (Java_java_lang_Object *)get_current_thread_exception();

        orp_enable_gc();        //---------------------------------^

        return new_handle;
    } else {
        return 0;
    }
} //ExceptionOccurred



void JNICALL ExceptionDescribe(JNIEnv *env)
{
    if(get_current_thread_exception()) {
        orp_disable_gc();       //---------------------------------v

        Java_java_lang_Throwable *exc =
            (Java_java_lang_Throwable *)get_current_thread_exception();
        const char *name = exc->vt->clss->name->bytes;
        cerr << name << ":" << endl;
        print_stack_trace(stderr, exc);

        orp_enable_gc();        //---------------------------------^
    }
} //ExceptionDescribe



void JNICALL ExceptionClear(JNIEnv *env)
{
    clear_current_thread_exception();
} //ExceptionClear



void JNICALL FatalError(JNIEnv *env, const char *msg)
{
    orp_cout << "\nA fatal error in a JNI native function:\n" << msg << endl;
    orp_exit(109);
} //FatalError



////////////////////////////////////////////////////////////////////////////////
// Move those lines to appropriate files!


// To be moved to root_set_enum.cpp
Object_Handle global_object_handles = 0;


////////////////////////////////////////////////////////////////////////////////



#ifdef _DEBUG

static void orp_print_global_object_handles()
{
    int num = 0;
    Object_Handle last = 0;
    printf("Printing global handles (traversing forward):\n");
    Object_Handle h;
    for(h = global_object_handles; h; h = h->next, num++) {
        printf("%5d: [0x%08x] = 0x%08x\n", num, h, h->java_reference);
        last = h;
    }
    printf("       num=%3d          (traversing backward):\n", num);
    num = 0;
    for(h = last; h; h = h->prev, num++) {
        printf("%5d: [0x%08x] = 0x%08x\n", num, h, h->java_reference);
    }
    printf("       num=%3d          \nDone.\n", num);
} //orp_print_global_object_handles



static void orp_print_local_object_handles()
{
    int num = 0;
    Object_Handle last = 0;
    printf("Printing local handles (traversing forward):\n");
    J2N_Saved_State *ljf = get_orp_last_java_frame();
    assert(ljf);
    Object_Handle h;
    for(h = ljf->local_object_handles; h; h = h->next, num++) {
        printf("%5d: [0x%08x] = 0x%08x\n", num, h, h->java_reference);
        last = h;
    }
    printf("       num=%3d          (traversing backward):\n", num);
    num = 0;
    for(h = last; h; h = h->prev, num++) {
        printf("%5d: [0x%08x] = 0x%08x\n", num, h, h->java_reference);
    }
    printf("       num=%3d          \n", num);
} //orp_print_local_object_handles

#endif



Object_Handle orp_create_global_object_handle()
{
    // Allocate the handle.
    Object_Handle h = orp_allocate_object_handle();

    p_thread_lock->_lock_enum();

    // Insert it at the beginning of the list of global handles.
    h->next = global_object_handles;
    global_object_handles = h;
    if(h->next) {
        h->next->prev = h;
    }

    p_thread_lock->_unlock_enum();

    return h;
} //orp_create_global_object_handle



void orp_delete_global_object_handle(Object_Handle handle)
{
    p_thread_lock->_lock_enum();
    orp_disable_gc();       //---------------------------------v

    for(Object_Handle h = global_object_handles; h; h = h->next) {
        if(h == handle) {
            Object_Handle next = h->next;
            Object_Handle prev = h->prev;
            if(prev) {
                // We are not at the head of the list.
                prev->next = next;
            } else {
                // We are at the head of the list.
                global_object_handles = next;
            }
            if(next) {
                // We are not at the end of the list.
                next->prev = prev;
            }

            // Mark the handle as removed
            h->java_reference = 0;
            break;
        }
    }

    orp_enable_gc();        //---------------------------------^
    p_thread_lock->_unlock_enum();

    orp_free_object_handle(handle);
} //orp_delete_global_object_handle



//
// We don't need a lock, because local handles are accessed either
// from the owner thread or from another thread for enumeration.
// Since GC must be enabled for enumeration, disabling GC in this
// function achieves, in effect, a mutex.
//
void orp_delete_local_object_handle(Object_Handle handle)
{
#ifdef _IA64_
    // (mjc 20000727) To be implemented as part of the ljf cleanup.
    return;
#endif
    J2N_Saved_State *ljf = get_orp_last_java_frame();
    assert(ljf);

    orp_disable_gc();       //---------------------------------v

    for(Object_Handle h = ljf->local_object_handles; h; h = h->next) {
        if(h == handle) {
            Object_Handle next = h->next;
            Object_Handle prev = h->prev;
            if(prev) {
                // We are not at the head of the list.
                prev->next = next;
            } else {
                // We are at the head of the list.
                ljf->local_object_handles = next;
            }
            if(next) {
                // We are not at the end of the list.
                next->prev = prev;
            }

            // Mark the handle as removed
            h->java_reference = 0;
            break;
        }
    }

    orp_enable_gc();        //---------------------------------^

    orp_free_object_handle(handle);
} //orp_delete_local_object_handle



jobject JNICALL NewGlobalRef(JNIEnv *env, jobject obj)
{
    if(!obj) {
        return 0;
    }

    Object_Handle new_handle = orp_create_global_object_handle();
    Object_Handle old_handle = (Object_Handle)obj;

    orp_disable_gc();       //---------------------------------v

    new_handle->java_reference = old_handle->java_reference;

    orp_enable_gc();        //---------------------------------^

    return new_handle;
} //NewGlobalRef



void JNICALL DeleteGlobalRef(JNIEnv *env, jobject globalRef)
{
    if (globalRef)
        orp_delete_global_object_handle((Object_Handle)globalRef);
} //DeleteGlobalRef



void JNICALL DeleteLocalRef(JNIEnv *env, jobject localRef)
{
    if (localRef)
        orp_delete_local_object_handle((Object_Handle)localRef);
} //DeleteLocalRef



jboolean JNICALL IsSameObject(JNIEnv *env,
                              jobject ref1,
                              jobject ref2)
{
    if(ref1 == ref2) {
        return JNI_TRUE;
    }

    if(!(ref1 && ref2)) {
        // One reference is null and the other is not.
        return JNI_FALSE;
    }

    Object_Handle h1 = (Object_Handle)ref1;
    Object_Handle h2 = (Object_Handle)ref2;

    orp_disable_gc();       //---------------------------------v

    Java_java_lang_Object *java_ref1 = h1->java_reference;
    Java_java_lang_Object *java_ref2 = h2->java_reference;
    jboolean result = (java_ref1 == java_ref2) ? JNI_TRUE : JNI_FALSE;

    orp_enable_gc();        //---------------------------------^

    return result;
} //IsSameObject



jobject JNICALL AllocObject(JNIEnv *env,
                            jclass clazz)
{
    assert(clazz);
    Object_Handle h = (Object_Handle)clazz;
    Class *clss = (Class *)h->java_reference;

    if(class_is_interface(clss) || class_is_abstract(clss)) {
        // Can't instantiate interfaces and abtract classes.
        ThrowNew_Quick(env, "java/lang/InstantiationException", clss->name->bytes);
        return 0;
    }

    orp_disable_gc();       //---------------------------------v
    Java_java_lang_Object *new_obj = (Java_java_lang_Object *)class_alloc_new_object(clss);
    Object_Handle new_handle = orp_create_local_object_handle();
    new_handle->java_reference = (Java_java_lang_Object *)new_obj;
    orp_enable_gc();        //---------------------------------^

    return new_handle;
} //AllocObject



jobject JNICALL NewObject(JNIEnv *env,
                          jclass clazz,
                          jmethodID methodID,
                          ...)
{
    va_list args;
    va_start(args, methodID);
    return NewObjectV(env, clazz, methodID, args);
} //NewObject



jobject JNICALL NewObjectV(JNIEnv *env,
                           jclass clazz,
                           jmethodID methodID,
                           va_list args)
{
    jvalue *jvalue_args = get_jvalue_arg_array((Method *)methodID, args);
    jobject result = NewObjectA(env, clazz, methodID, jvalue_args);
    orp_free_gc_safe(jvalue_args);
    return result;
} //NewObjectV



jobject JNICALL NewObjectA(JNIEnv *env,
                           jclass clazz,
                           jmethodID methodID,
                           jvalue *args)
{
    // Allocate the object.
    jobject new_handle = AllocObject(env, clazz);

    if(!new_handle) {
        // Couldn't allocate the object.
        return 0;
    }

    // Invoke the constructor.
    // toss CallNonvirtualVoidMethodA(env, new_handle, clazz, methodID, args);
    CallVoidMethodA(env, new_handle, methodID, args);

    return new_handle;
} //NewObjectA



jclass JNICALL GetObjectClass(JNIEnv *env,
                              jobject obj)
{
    // The spec guarantees that the reference is not null.
    assert(obj);

    Object_Handle h = (Object_Handle)obj;

    orp_disable_gc();       //---------------------------------v

    Class *clss = h->java_reference->vt->clss;

    orp_enable_gc();        //---------------------------------^

    Object_Handle new_handle = orp_create_local_object_handle();
    new_handle->java_reference = (Java_java_lang_Object *)clss;

    return new_handle;
} //GetObjectClass



jboolean JNICALL IsInstanceOf(JNIEnv *env,
                              jobject obj,
                              jclass clazz)
{
    if(!obj) {
        return JNI_TRUE;
    }

    jclass obj_class = GetObjectClass(env, obj);
    Object_Handle h = (Object_Handle)clazz;
    Object_Handle obj_class_h = (Object_Handle)obj_class;
    Class *clss = (Class *)h->java_reference;
    Class *obj_clss = (Class *)obj_class_h->java_reference;

	Boolean isInstance = orp_instanceof_class(obj_clss, clss);

    if(isInstance) {
        return JNI_TRUE;
    } else {
        return JNI_FALSE;
    }
} //IsInstanceOf



jstring JNICALL NewString(JNIEnv *env,
                          const jchar *unicodeChars,
                          jsize length)
{
    jcharArray java_array = NewCharArray(env, length);
    jboolean isCopy;
    jchar *c_array = GetCharArrayElements(env, java_array, &isCopy);
    memcpy(c_array, unicodeChars, sizeof(jchar) * length);
    ReleaseCharArrayElements(env, java_array, c_array, 0);

    jclass jls = FindClass(env, "java/lang/String");
    jmethodID init = GetMethodID(env, jls, "<init>", "([C)V");
    jvalue args[1];
    args[0].l = java_array;
    jobject str = NewObjectA(env, jls, init, args);
    // use non-OO form to enable calls when jenv is not available.
    //env->DeleteLocalRef(java_array);
	DeleteLocalRef(env, java_array);
    //env->DeleteLocalRef(jls);
	DeleteLocalRef(env, jls);
    return str;
} //NewString



jsize JNICALL GetStringLength(JNIEnv *env,
                              jstring string)
{
    if(!string)
        return 0;
    jfieldID count_id = GetFieldID_Quick(env, "java/lang/String", "count", "I");
    jint count = GetIntField(env, string, count_id);
    return count;
} //GetStringLength



const jchar *JNICALL GetStringChars(JNIEnv *env,
                                    jstring string,
                                    jboolean *isCopy)
{
    assert(string);

	jint count;
	jint offset;
	jcharArray char_array;

	get_string_specifics(env, string, &count, &offset, &char_array);
	
	assert(count >= 0);
	assert(char_array);
	assert(offset >= 0);

    jchar *primitive_array = (jchar *)orp_malloc_gc_safe(sizeof(jchar) * count);
    Object_Handle char_array_handle = (Object_Handle)char_array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfChar *java_array = (JavaArrayOfChar *)char_array_handle->java_reference;
    memcpy(primitive_array, &(java_array->body[offset]), sizeof(jchar) * count);

    orp_enable_gc();        //---------------------------------^

    if(isCopy) {
        *isCopy = JNI_TRUE;
    }
    return primitive_array;
} //GetStringChars



void JNICALL ReleaseStringChars(JNIEnv *env,
                                jstring string,
                                const jchar *chars)
{
    orp_free_gc_safe((void *)chars);
} //ReleaseStringChars



jstring JNICALL NewStringUTF(JNIEnv *env,
                             const char *bytes)
{
    unsigned len = get_unicode_length_of_utf8(bytes);
    jchar *unicodeChars = (jchar *)malloc(len * sizeof(jchar));
    assert(unicodeChars);
    unpack_utf8(unicodeChars, bytes);
    jstring result = NewString(env, unicodeChars, len);
    free(unicodeChars);
    return result;
} //NewStringUTF



jsize JNICALL GetStringUTFLength(JNIEnv *env,
                                 jstring string)
{
    if (string == NULL)
        return 0;
  
	jint count;
	jint offset;
	jcharArray char_array;

	get_string_specifics(env, string, &count, &offset, &char_array);
	
	assert(count >= 0);
	assert(char_array);
	assert(offset >= 0);

    Object_Handle char_array_handle = (Object_Handle)char_array;

    jsize len = 0;
    orp_disable_gc();       //---------------------------------v

    JavaArrayOfChar *java_array = (JavaArrayOfChar *)char_array_handle->java_reference;
    len = get_utf8_length_of_unicode(&(java_array->body[offset]), count);

    orp_enable_gc();        //---------------------------------^

    return len;
} //GetStringUTFLength



const char *JNICALL GetStringUTFChars(JNIEnv *env,
                                      jstring string,
                                      jboolean *isCopy)
{
    if(!string)
        return 0;
    jsize unicode_length = GetStringLength(env, string);
    const jchar *jch = GetStringChars(env, string, isCopy);
    jsize utf_length = GetStringUTFLength(env, string);
    char *utf_array = (char *)orp_malloc_gc_safe(utf_length + 1);
    pack_utf8(utf_array, jch, unicode_length);
    ReleaseStringChars(env, string, jch);
    return utf_array;
} //GetStringUTFChars



void JNICALL ReleaseStringUTFChars(JNIEnv *env,
                                   jstring string,
                                   const char *utf)
{
    orp_free_gc_safe((void *)utf);
} //ReleaseStringUTFChars



jint JNICALL RegisterNatives(JNIEnv *env,
                             jclass clazz,
                             const JNINativeMethod *methods,
                             jint nMethods)
{
    Global_Env *ge = ORP_Global_State::loader_env;
    for(jint i = 0; i < nMethods; i++) {
        String *name = ge->string_pool.lookup(methods[i].name);
        String *sig  = ge->string_pool.lookup(methods[i].signature);
        void *f = methods[i].fnPtr;
        Class *clss = (Class *)((Object_Handle)clazz)->java_reference;
        register_native_method(clss, name, sig, f);
    }
    return 0;
} //RegisterNatives



jint JNICALL UnregisterNatives(JNIEnv *env, jclass clazz)
{
    Class *clss = (Class *)((Object_Handle)clazz)->java_reference;
    unregister_native_methods(clss);
    return 0;
} //UnregisterNatives



jint JNICALL MonitorEnter(JNIEnv *env, jobject obj)
{
#ifdef _IA64_
    assert(0);
#endif

    Object_Handle handle = (Object_Handle)obj;

    orp_disable_gc();       //---------------------------------v

    Java_java_lang_Object *object = (Java_java_lang_Object *)handle->java_reference;

    void (__stdcall *p_mon_enter_code) (Java_java_lang_Object *);
    p_mon_enter_code = 
      ( void (__stdcall *)(Java_java_lang_Object *) )orp_get_rt_support_addr(ORP_RT_MONITOR_ENTER);

    p_mon_enter_code(object);

    orp_enable_gc();        //---------------------------------^

    return 0;
} //MonitorEnter



jint JNICALL MonitorExit(JNIEnv *env, jobject obj)
{
#ifdef _IA64_
    assert(0);
#endif

    Object_Handle handle = (Object_Handle)obj;

    orp_disable_gc();       //---------------------------------v

    Java_java_lang_Object *object = (Java_java_lang_Object *)handle->java_reference;


    void (__stdcall *p_mon_exit_code) (Java_java_lang_Object *);
    p_mon_exit_code = 
      ( void (__stdcall *)(Java_java_lang_Object *) )orp_get_rt_support_addr(ORP_RT_MONITOR_EXIT);

    p_mon_exit_code(object);

    orp_enable_gc();        //---------------------------------^

    return 0;
} //MonitorExit



jint JNICALL GetJavaVM(JNIEnv *env, JavaVM **vm)
{
    assert(0);
    return 0;
} //GetJavaVM



