// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/root_set_enum.cpp,v 1.12 2001/12/29 09:29:22 xli18 Exp $
//



#include "platform.h"
#include <assert.h>
#include <iostream.h>

#include "object_layout.h"
#include "environment.h"
#include "orp_types.h"
#include "jit_runtime_support.h"
#include "method_lookup.h"
#include "stack_manipulation.h"
#include "orp_threads.h"
#include "orp_utils.h"
#include "jit_intf.h"
#include "jit_intf_cpp.h"
#include "nogc.h"
#include "compile.h"
#include "orp_stats.h"
#include "finalize.h"
#include "Class_Loader.h"

#include "root_set_enum.h"
#include "enum_trampoline.h"
#include "exceptions.h"

#ifdef ORP_POSIX
#include "platform2.h"
#endif

#ifdef OBJECT_LOCK_V2
#include "mon_enter_exit_olv2.h"
#else
#include "mon_enter_exit.h"
#endif

extern Object_Handle global_object_handles;

#include "GlobalClassIterator.h"

//////////////////////////////////////////////////////////////////////////
// begin GC native interface


// For internal ORP use only.
#ifdef USE_GC_DLL
ORPExport
#endif
Boolean orp_is_gc_disabled(void *thread_id)
{
    gc_enable_disable_state endis = ((ORP_thread *)thread_id)->gc_enabled_status;

    if (endis == disabled) {
        return true;
    }
    return false;
} //orp_is_gc_disabled


bool orp_get_gc_enabled_state()
{
    gc_enable_disable_state endis = p_TLS_orpthread->gc_enabled_status;
    if (endis == disabled)
        return 0;
    return 1;
} //orp_get_gc_enabled_state


// What are the rules for when something can and can't be enabled or disabled.
//
// 1. Only the current thread (p_TLS_orpthread) can disable and enable GC, The GC
//    thread can however change an enable to an enable_will_block.
// 2. You can not enable GC from within JITTED code.
// 3. You can only enable if you are disabled.
// 4. You can atomically go from disabled to enabled_will_block for use by the
//    gc hijack code. 
// 5. You can enable if ljf is 0 and
//    there is a legal throw_context available to p_TLS_orpthread.
//

bool print_bug_not_done = true;
bool orp_enable_gc() 
{
    ORP_thread *p_thr = p_TLS_orpthread;
    void *the_ljf;
#ifdef USE_IA64_JIT
    the_ljf = p_thr->saved_ar_bsp;
#else
    the_ljf = p_thr->last_java_frame;
#endif
    if (the_ljf == 0) { 
        // If "ljf" is zero, the thread had better be "birthing" or "dying".
        if (!((p_thr->app_status == thread_is_dying) || 
              (p_thr->app_status == thread_is_birthing)) ||
              (p_thr->throw_context != NULL)) {
            // We could be compiling main so there won't be an ljf
            if (print_bug_not_done) {
                orp_cout << "*** BUG *** - orp_enable_gc: p_thr->last_java_frame is 0 and p_thr->app_status is " \
                    << p_thr->app_status; 
                if (p_thr->app_status == thread_is_running) {
                    orp_cout << " ("<< p_thr->app_status << " means thread_is_running) " << endl;
                } else {
                    orp_cout << endl;
                }
                print_bug_not_done = false;
            }
        }
	}
    // If "ljf" is zero, the thread had better be "birthing" or "dying".								     
//	  		    assert((p_thr->app_status == thread_is_dying) || 
//                        (p_thr->app_status == thread_is_birthing));
    gc_enable_disable_state endis = 
            gc_if_disabled_then_enable(&p_thr->gc_enabled_status);

    if (endis == disabled) {
        return 0;
    } else {
        return 1;
    }

} //orp_enable_gc


// The legal transitions are
// enter v ------------------------v
//       |                         |
// enabled_will_block -(block)-> enabled --> disabled
//      ^                          v
//      |---GC thread action  -----|
//
bool orp_disable_gc()
{
    ORP_thread *p_thr = p_TLS_orpthread;

    gc_enable_disable_state endis = bogus;

    while (1) {

        endis = gc_if_enabled_then_disable(&p_thr->gc_enabled_status);    
        if (endis == enabled) { 
            
            if (p_thr->last_java_frame == 0) {
               if (!((p_thr->app_status == thread_is_dying) || 
                    (p_thr->app_status == thread_is_birthing))) {
                   if (print_bug_not_done) {
                       orp_cout << "BUG - orp_disable_gc: p_thr->last_java_frame is 0 and p_thr->app_status is " \
                             << p_thr->app_status << endl;
                       print_bug_not_done = false;
                   }
                }
                // If "ljf" is zero, the thread had better be "birthing" or "dying".								     
//	  		    assert((p_thr->app_status == thread_is_dying) || 
//                        (p_thr->app_status == thread_is_birthing));
	        }
            return true;

        }
        if (endis == disabled)
            return false;

        endis = gc_if_EWB_then_enable(&p_thr->gc_enabled_status);
        // IF THE THREAD BREAKS HERE, THE CURRENT GC FINISHES AND THE NEXT GC STARTS
        // THEN THE GC WILL CHANGE THE STATE BACK TO ENABLE_WILL_BLOCK, IT WILL THEN
        // GATE THE THREAD THROUGH THE WAIT BELOW AND THEN THE THREAD WILL HANG 
        // WAITING FOR ANOTHER SIGNAL......
        // I THINK THIS MEANS THAT as far as the wait below is concerned it needs to 
        // atomically move from enabled_waiting to disabled without going through
        // the enabled state.
        // THE GC THREAD WILL move the enumerated thread 
        // FROM enabled to enabled_will_block 
        // FROM enabled_waiting to disabled and 
        // FROM enabled_will_block to enabled.
        // The enumeration thread will move 
        // FROM enabled to disabled
        // FROM enabled_will_block to enabled_waiting
        //
        if (endis == enabled_will_block)
        {
            DWORD wstat = WaitForSingleObject(p_thr->gc_resume_event_handle, INFINITE);
            assert(wstat != WAIT_FAILED);
            // Clearly this thread can be resumed the state of the other threads
            // can be pretty much anything making these next two assert meaningless.
            // I did hit them running pBob.
            //  assert(p_the_safepoint_control_thread == 0);  // toss if hit -- diagnostic only
			//  assert(p_thr->which_trap == x_nothing);  // toss if hit -- diagnostic only
        }
    }
} //orp_disable_gc

// For internal ORP use only.
Boolean orp_is_gc_enabled(void *thread_id)
{
    gc_enable_disable_state endis = ((ORP_thread *)thread_id)->gc_enabled_status;

    if (endis == disabled)
        return 0;
    return 1;
} //orp_is_gc_enabled


void *orp_malloc_gc_safe(unsigned size)
{
    bool prev_gc_state = orp_disable_gc();
    p_TLS_orpthread->locks_held_add(LH_MALLOC_FREE);
    void *a = gc_malloc_fixed_data_C(size);
    p_TLS_orpthread->locks_held_remove(LH_MALLOC_FREE);
    if(prev_gc_state) {
        orp_enable_gc();
    }
    return a;
} //orp_malloc_gc_safe



void orp_free_gc_safe(void *a)
{
    bool prev_gc_state = orp_disable_gc();
    p_TLS_orpthread->locks_held_add(LH_MALLOC_FREE);
    gc_free_fixed_data_C(a);
    p_TLS_orpthread->locks_held_remove(LH_MALLOC_FREE);
    if(prev_gc_state) {
        orp_enable_gc();
    }
} //orp_free_gc_safe



//
// Insert a frame into the list of frames for current thread.
//
// No synchronization needed, because no other thread may want to update this
// data structure.
//
// However another thread may be reading this list while it is being updated.
// Therefore the list update operation is atomic.
//
void orp_push_gc_frame(GC_Frame *frame, void *gc_block, int gc_block_len)
{
    frame->gc_block = gc_block;
    frame->gc_block_len = gc_block_len;
    frame->next = p_TLS_orpthread->gc_frames;

    // Atomically modify the list.
    p_TLS_orpthread->gc_frames = frame;
} //orp_push_gc_frame



//
// Remove a frame from the list of frames for current thread.
//
// No synchronization needed, because no other thread may want to update this
// data structure.
//
// Possible race condition if the current thread is in gc_enabled state
// and another thread will read the list while it's being updated.
//
void orp_pop_gc_frame(GC_Frame *frame)
{
    if(p_TLS_orpthread->gc_frames == frame) {
        // Atomically update the list.
        p_TLS_orpthread->gc_frames = frame->next;
    } else {
#if 1
        cout << "**** Trying to pop frame " << (void *)frame
            << " top frame is " << (void *)p_TLS_orpthread->gc_frames
            << endl;
#endif
        // If the "push" and "pop" fucntions are properly nested, we should never
        // take this branch.
        for(GC_Frame *f = p_TLS_orpthread->gc_frames; f; f = f->next) {
#if 1
            cout << "  f=" << (void *)f << endl;
#endif
            if(f->next == frame) {
                f->next = frame->next;
                cout << "popped" << endl;
                return;
            }
        }
        // For debugging only.  In the release version, we ignore this condition.
        assert(0);
    }
} //orp_pop_gc_frame



void orp_enumerate_references_in_gc_frames(ORP_thread *thread)
{
    for(GC_Frame *f = thread->gc_frames; f; f = f->next) {
        void **refs = (void **)(f->gc_block);
        void **end_of_refs = (void **)(((char *)refs) + f->gc_block_len);
        assert((f->gc_block_len % sizeof(void *)) == 0);
        while(refs < end_of_refs) {
            orp_enumerate_root_reference(0, refs);
            refs++;
        }
    }
} //orp_enumerate_references_in_gc_frames



// end GC native interface
//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
//////// begin object handles


Object_Handle orp_allocate_object_handle()
{
    Object_Handle h =
        (Object_Handle)gc_malloc_fixed_data_C(sizeof(Object_Handle_Struct));
    memset(h, 0, sizeof(Object_Handle_Struct));
    return h;
} //orp_allocate_object_handle



Object_Handle orp_create_local_object_handle()
{
#ifdef USE_IA64_JIT
    Object_Handle orp_create_local_object_handle_ia64();
    return orp_create_local_object_handle_ia64();
#else
    // Allocate the handle.
    Object_Handle h = orp_allocate_object_handle();

    // Insert it at the beginning of the list of local handles.
    J2N_Saved_State *ljf = get_orp_last_java_frame();
    assert(ljf);
    h->next = ljf->local_object_handles;
    ljf->local_object_handles = h;
    if(h->next) {
        h->next->prev = h;
    }

    return h;
#endif
} //orp_create_local_object_handle


void orp_free_object_handle(Object_Handle h)
{
    if(!h->allocated_on_the_stack) {
        gc_free_fixed_data_C(h);
    } else {
        // We shouldn't call this function for stack-allocated handles
        assert(0);
    }
} //orp_free_object_handle



void orp_enumerate_object_handles(ORP_thread *thread)
{
#ifdef USE_IA64_JIT
    // (mjc 2000-07-10) use orp_enumerate_object_handles_ia64 instead
    assert(0);
#else
    J2N_Saved_State *ljf = (J2N_Saved_State *)thread->get_last_java_frame();
    while(ljf) {
        for(Object_Handle h = ljf->local_object_handles; h; h = h->next) {
            orp_enumerate_root_reference(0, (void **)&(h->java_reference));
        }
        ljf = (J2N_Saved_State *)ljf->prev_ljf;
    }
#endif
} //orp_enumerate_object_handles



//////// end object handles
//////////////////////////////////////////////////////////////////////////



void orp_enumerate_interned_strings()
{
    unsigned cookie = 0;
    ORP_Global_State::loader_env->string_pool.lock_pool();
    String *ps = ORP_Global_State::loader_env->string_pool.get_first_string_intern(&cookie);
    while(ps) {
        void *s = ps->intern;
        orp_enumerate_root_reference(0, (void **)&(ps->intern));
        ps = ORP_Global_State::loader_env->string_pool.get_next_string_intern(ps, &cookie);
    }
    ORP_Global_State::loader_env->string_pool.unlock_pool();
} //orp_enumerate_interned_strings

//
// Enumerate all classes and their fields.
//
void orp_enumerate_static_fields()
{
#if 1
	Class **ppc;
	//GlobalClassIterator::first();
	GlobalClassIterator iterator;
	iterator.first();
    //while(ppc = GlobalClassIterator::next()) {
	while(ppc = iterator.next()) {
		Class *c = *ppc;
        // First enumerate the class
        orp_enumerate_root_reference(0, (void **)ppc);
        unsigned n_fields = c->n_fields;
        if((c->state == ST_Initializing) ||
           (c->state == ST_Initialized)) {
            // Class has been prepared, so we can iterate over all its fields.
            for(unsigned i = 0; i < n_fields; i++) {
                Field *f = &c->fields[i];
                if(f->is_static()) {
                    char sig0 = f->get_signature()->descriptor->bytes[0];
                    if(sig0 == 'L' || sig0 == '[') {
                        // The field is static and it is a reference.
                        orp_enumerate_root_reference(0, (void **)f->get_address());
                    }
                }
            }
        }
    }
#else //only iterating ORP_Global_State::loader_env->class_table
    Class **ptrC = ORP_Global_State::loader_env->class_table.get_first_class();
    while(ptrC) {
        // First enumerate the class
        orp_enumerate_root_reference(0, (void **)ptrC);
        Class *c = *ptrC;
		if(!strcmp(c->name->bytes, "java/lang/ClassLoader"))
			int kk = 0;
        unsigned n_fields = c->n_fields;
        if((c->state == ST_Initializing) ||
           (c->state == ST_Initialized)) {
            // Class has been prepared, so we can iterate over all its fields.
#if 0 //def NEW_DATA_ALLOC
            if(c->static_fields_boolean) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_boolean);
            }
            if(c->static_fields_byte) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_byte);
            }
            if(c->static_fields_char) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_char);
            }
            if(c->static_fields_short) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_short);
            }
            if(c->static_fields_int) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_int);
            }
            if(c->static_fields_long) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_long);
            }
            if(c->static_fields_float) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_float);
            }
            if(c->static_fields_double) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_double);
            }
            if(c->static_fields_ref) {
                orp_enumerate_root_reference(0, (void **)&c->static_fields_ref);
            }
#else
            for(unsigned i = 0; i < n_fields; i++) {
                Field *f = &c->fields[i];
                if(f->is_static()) {
                    char sig0 = f->get_signature()->descriptor->bytes[0];
                    if(sig0 == 'L' || sig0 == '[') {
                        // The field is static and it is a reference.
                        orp_enumerate_root_reference(0, (void **)f->get_address());
#if 0                         
                        gc_trace_slot ((void **)f->get_address(), 
                                        *((void **)f->get_address()), 
                                        "Enumerating a static slot.");
#endif 

                    }
                }
            }
#endif
        }

        ptrC = ORP_Global_State::loader_env->class_table.get_next_class(c);
    }
#endif
} //orp_enumerate_static_fields

//
// This is the main function used to enumerate Java references by the ORP and
// the JITs.  It is part of the JIT-ORP interface.
//
void orp_enumerate_root_reference(GC_Enumeration_Handle gc_handle, void **ref)
{
    gc_add_root_set_entry((Java_java_lang_Object **)ref);
} //orp_enumerate_root_reference

//
// This is the main function used to enumerate interior pointers by the JITS.
// It is part of the JIT-ORP interface

void orp_enumerate_root_interior_pointer(GC_Enumeration_Handle gc_handle, void **slot,
                                         int offset)
{
    gc_add_root_set_entry_interior_pointer(slot, offset);
} //orp_enumerate_root_interior_pointer

void orp_enumerate_jni_global_handles()
{
    for(Object_Handle h = global_object_handles; h; h = h->next) {
        orp_enumerate_root_reference(0, (void **)&(h->java_reference));
    } 
}


void orp_enumerate_root_set_global_refs()
{
    ////////////////////////////////////////
    ///// First enumerate strong pointers

    // Static fields of all classes
    orp_enumerate_static_fields();

    orp_enumerate_objects_to_be_finalized();

    orp_enumerate_jni_global_handles();

    ////////////////////////////////////////
    //// Now enumerate weak pointers
    orp_enumerate_interned_strings();

#ifdef OBJECT_LOCK_V2
    extern void orp_enumerate_root_set_mon_arrays();
	orp_enumerate_root_set_mon_arrays();

#ifdef LAZY_LOCK
	if(!bMultithreaded){
		for(unsigned i=0; i<(num_lazylock-(int)lazylist)/(sizeof(POINTER_SIZE_INT)/sizeof(BYTE)); i++)
			orp_enumerate_root_reference(0, (void **)&(lazylist[i]));
	}
#endif //LAZY_LOCK

#endif

    Class_Loader::gc_enumerate();
} //orp_enumerate_root_set_global_refs



//
// Enumerate references associated with a thread which are not stored on
// the thread's stack.
//
void orp_enumerate_root_set_single_thread_not_on_stack(ORP_thread *thread)
{
    orp_enumerate_root_reference(0, (void **)&(thread->p_java_lang_thread));
    orp_enumerate_root_reference(0, (void **)&(thread->p_current_object));
    orp_enumerate_root_reference(0, (void **)&(thread->p_stop_object));
    orp_enumerate_root_reference(0, (void **)&(thread->p_exception_object));

    orp_enumerate_references_in_gc_frames(thread);
#ifdef MONITOR_STO
#ifndef USE_IA64_JIT
	if(!bMultithreaded){
		for(int i=0; i<(num_lazylock-(int)lazylist)/4; i++)
			orp_enumerate_root_reference(0, (void **)&(lazylist[i]));
	}
#endif //USE_IA64_JIT
#endif //MONITOR_STO

} //orp_enumerate_root_set_single_thread_not_on_stack
