// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/include/step_generation.h,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

#ifndef _step_generation_H_
#define _step_generation_H_

//
// Generations that comprise multiple steps (but no nurseries).
// Step Generations inherit from Generation.
//
//                Generation
//                    |
//                    |
//       ---------------------------
//       |                         |
//       |                         |
//    Step_Generation              |
//       |                         |
//       |                  Train_Generation
//       |
//  Step_Plus_Nursery_Generation
//

#include "generation.h"
#include "step.h"
// #include "mrl_gc_v1.h"
#include "gc_space.h"
#include "gc_consts.h"
#include "object_list.h"
#include "gc_hooks.h"
//#include "car.h"
//#include "train.h"

class Card_Table;
class Nursery;
class Train_Generation;
class Gc_Fast_Hooks;
class Gc_Plan;

class Step_Generation : public Generation {

public:

	Step_Generation(unsigned long gen_num,
                    Gc_Fast_Hooks *p_gc_hooks,
                    Gc_Plan       *p_gc_plan,
		            Gc_Interface  *p_container,
					Generation    *p_superior,
                    Card_Table    *p_card_table,
		            Block_Store   *p_block_store);

	virtual ~Step_Generation() {
		for (unsigned int step_index = 0; 
		                  step_index < _number_of_steps; 
						  step_index++) {
			delete _p_step[step_index];
		}
#ifndef PER_CAR_BARRIERS
		delete _p_old_to_young_remembered_set;
#endif

        delete _p_finalizable_object_list;
	}

	//
	// Every generation needs to define a policy for handling objects
	// found during cheney scans. Step Generations (and NS Generations)
	// simply send eviction notices to all reachable objects.
	//
	virtual void cheney_scan_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection);

//#ifdef GC_OFFSET_LOGIC
   	//
	// Every generation needs to define a policy for handling objects
	// found during cheney scans. Step Generations (and NS Generations)
	// simply send eviction notices to all reachable objects.
	//
	virtual void cheney_scan_array_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection);
//#endif // GC_OFFSET_LOGIC

    void enumerate_reference_queues ();
    virtual void enqueue_soft_ref (java_lang_ref_Reference *a_reference);
	virtual void enqueue_weak_ref (java_lang_ref_Reference *a_reference);
	virtual void enqueue_phantom_ref (java_lang_ref_Reference *a_reference);

	//
	// The GC interface gives each generation an opportunity to
	// clean up after a collection of that generation.
	//
	virtual void cleanup();
	
    virtual Remembered_Set *
		p_reclaim_generation(Remembered_Set *p_root_set,
                             Remembered_Set *p_weak_refs,
							 bool *p_is_train_alive);

    virtual
    Java_java_lang_Object *p_import_object_from_yos(Java_java_lang_Object **pp_obj) {
        orp_cout << " How can I call p_import_object_from_yos into the step generation?? " << endl;
        assert (0); // Assert and debug....
        return _p_step[0]->p_scavenge_object(pp_obj);
    }

    virtual bool is_train_generation() {
        return false;
    }

    virtual bool is_step_generation() {
        return true;
    }

    virtual bool is_step_plus_nursery_generation() {
        return false;
    }

#if (GC_DEBUG>3)
	//
	// Routine to display the contents of the generation.
	//
	virtual void inspect(unsigned int level);
#endif // _DEBUG

#if (GC_DEBUG>2)
	//
	// Step Generations don't have nurseries.
	//
	virtual Nursery *p_cycle_nursery(Nursery *p_nursery,
		                             bool returnNullOnFail) {
		orp_exit(1); // illegal for this generation type.
		return NULL;
	}
#endif

#ifndef PER_CAR_BARRIERS
	//
	// The GC Interface is asking us to selectively purge some write
	// barrier entries, most probably because some car just got reclaimed.
	//
	virtual void selective_flush_write_barrier(Car *p_car);
#endif

	//
	// Used by our GC Interface container to link me to my older gen.
	//
	void set_mature_generation(Train_Generation *p_gen) {
		_p_mature_generation = p_gen;
	}

	//
	// A contained space is asking me to tenure this object.
	// I pass the request to to the mature generation.
	//
    virtual Java_java_lang_Object *p_tenure_object(Java_java_lang_Object **pp_obj_ref) { 
		assert(_p_superior_generation); 
		//
		// Save away the present location of the object.
		//
		Java_java_lang_Object *p_old_obj = *pp_obj_ref; 

        gc_trace (*pp_obj_ref, "tenuring object in step_generation.h ln 193."); 
        
		//
		// Ask our superior generation to take the object. The returned
		// address reflects the object's new location.
		//
		Java_java_lang_Object *p_new_obj =
			_p_superior_generation->p_import_object_from_yos(pp_obj_ref); 

        gc_trace (p_new_obj, "New Object just tenured from step_generation.h ln 195.");

        // Always tenure into the train generation.
        assert ((Train_Generation *)_p_superior_generation == _p_mature_generation); 

        // Updated of the remembered set is done during the cheney scan.

#if (GC_DEBUG>2)
		object_tenure_hook(p_old_obj, p_new_obj);
#endif 
		return p_new_obj;
}

    //
    // A LOS or Nursery is telling us that a new finalizable object
    // has been allocated.
    //
    void record_finalizable_object(Java_java_lang_Object *p_obj) {
        _p_finalizable_object_list->add_entry(p_obj);
    }

    //
    // At shutdown time, may need to run all finalizers.
    //
    virtual void run_all_finalizers();



#ifndef PER_CAR_BARRIERS
    //
    // The GC interface is processing (hybrid) write barriers and
    // has found an entry that may be of interest to this generation.
    //
    // Interesting is a pointer that is in the older generation that
    // refers to an object in this generation.
    // 
    virtual void add_entry_to_generation_write_barriers(Java_java_lang_Object **pp_obj_ref,
                                                        Java_java_lang_Object *p_obj_ref) {

        // Is is NULL.
        if (p_obj_ref == NULL) {
            assert (0); // used to just return
        }

        // Make sure that pp_obj_ref pointer to what it should.
        assert (*pp_obj_ref == p_obj_ref);

        // Is it a pointer from the youngest (this step) generation?
        if (is_address_in_my_generation (pp_obj_ref)) {
            return; // Not interesting.
        }

        if (is_address_in_my_generation (p_obj_ref)) {
            // This object is in this generation but the slot isn't, remember it.
            assert (p_global_bs->is_address_in_heap (pp_obj_ref)); 
            gc_trace ((void *)p_obj_ref, "Adding ref to old_to_young_remembered_set."); 
            _p_old_to_young_remembered_set->add_entry (pp_obj_ref);
            return;
        }
        // We have a slot and an object both in MOS, this should have called
        // the MOS add_entry_to_generation_write_barriers();
        assert (0);
    }


#endif

#ifndef PER_CAR_BARRIERS
    //
    // This is a faster, non-virtual version. Retire the other
    // if this works well.
    // 
    inline void update_old_to_young_write_barrier(Java_java_lang_Object **pp_obj_ref,
                                                  Java_java_lang_Object *p_obj_ref) {
        _p_old_to_young_remembered_set->add_entry(pp_obj_ref);
    }
#endif

	//
	// An entry in the young-to-old write barrier just became obsolete,
	// probably because an object was tenured into mature space.
	//
	virtual void purge_write_barrier_remembered_set(Java_java_lang_Object **pp_obj_ref) {
		_p_old_to_young_remembered_set->delete_entry(pp_obj_ref);
	}

	Remembered_Set *p_old_to_young_remembered_set() {
		return _p_old_to_young_remembered_set;
	}

    bool cheney_scan_pending ();
    bool scan_cheney_generation (bool doing_mos_collection);

#if (GC_DEBUG>3)
	//
	// Routine to verify the consistency of all the spaces of this generation.
	//
	virtual bool verify_generation();
#endif // _DEBUG

protected:
    Step_Generation();
    //
    // Add a new step into this generation.
    //    
	void _add_step(unsigned long step_index);

	//
	// While supplying cheney scan policy, it was discovered that the object
	// being scanned is in a car in the mature generation.
	//
	void _update_a_slot_in_car(Java_java_lang_Object **pp_child_ref, bool doing_mos_collection);

	//
	// When supplying cheney scan policy, the object being scanned was discovered
	// to be in our generation (i.e. a step or nursery).
	//
	void _update_a_slot_in_step_or_nursery(Java_java_lang_Object **pp_child_ref, 
                                           bool doing_mos_collection);
	//
	// This queue holds all the reference object whose referent (as returned by get)
	// are in this step generation. Since all steps in this generation are collected
	// at the same time we only need one set of queues. In MOS the queues are assocated with
	// each car.
	//
    java_lang_ref_Reference *_p_weak_object_list;

    java_lang_ref_Reference *_p_soft_object_list;

    java_lang_ref_Reference *_p_phantom_object_list;

	//
	// Evict an object and update the reference if it hasn't already been evicted.
	// Otherwise just update the reference.
	//
	Java_java_lang_Object *
	_p_evict_object_or_update_forwarding(Java_java_lang_Object **pp_obj_ref, bool doing_mos_collection);

	bool _is_object_in_mature_generation(Java_java_lang_Object *p_obj);

	bool _is_reference_in_mature_generation(Java_java_lang_Object **pp_obj_ref);

    void update_remembered_sets_as_needed (Java_java_lang_Object **pp_obj_ref, bool doing_mos_collection); 
    //
    // This object set maintains a list of objects in young space
    // that are candidates for finalization.
    //
    Object_List *_p_finalizable_object_list;

    //
    // We have finished a minor collection. Beforp we return we need to
    // scan the spent nurseries, from spaces and LOS for unreachable 
    // objects that have just become candidates for finalization.
    //
    void _locate_and_process_finalizable_objects(bool doing_mos_collection);

	//
	// A pointer to the older generation
	//
	Train_Generation *_p_mature_generation;
	
	//			
	// We detected that an object got tenured into the mature space.
	// Therefore we are notifying the mature generation that the
	// inter-car write barriers may need updating.
	//
	void _notify_mature_generation_possible_car_write_barrier_updates(Java_java_lang_Object **pp_obj_ref);
    //
    // Currently the number of steps is fixed at compile time (at 
    // run time with .plan files.) But that could change with dynamic
    // adaptive tenuring, where we might add or remove steps based
    // on observed object demographics.
    //
    unsigned long _number_of_steps;

    //
    // "Tenure reclaim" policy: default is to age all objects before
    // tenuring.
    //
    void _slow_tenure_reclaim(Remembered_Set *p_orp_live_references);

	//
	// Support routine for use by p_reclaim_generation. This routine
	// is first given the ORP live references, then the old-to-young
	// write barrier. For each, it cycles through and issues 
	// eviction notices to the containers of the objects.
	//
	void _reclaim_generation(Remembered_Set *p_root_set);

	//
	// Beforp doing a reclamation, give the contained steps 
	// an opportunity to set up their cheney spaces.
	//
	void _set_up_cheney_spaces();

    Step *_p_step[MAXIMUM_STEPS];
    //
    // This flag is used when collecting this generation to record if
    // a pointer has been found into the focus train.
    //
	bool _train_is_live;
	//
	// This helper routine is used when reclaiming a generation 
	// or doing a cheney scan to check if a pointer into mature 
	// space is into the focus car or train. If so, it is duely recorded.
	//
	void _update_into_mature_generation_ref(Java_java_lang_Object **pp_obj_ref, bool doing_mos_collection);
    
    //
    // After reclaiming this space, we update weak references.
    //

	void _update_phantom_references(bool doing_mos_collection);
	void _update_soft_references(bool low_on_memory, bool doing_mos_collection);
    void _update_weak_references(bool doing_mos_collection);
};

#endif // _step_generation_H_


