// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/mrl_gc_v1.cpp,v 1.4 2001/08/24 09:43:37 xli18 Exp $
//

 
#include "gc_for_orp.h"
#include "finalize.h"
 
#include "mrl_gc_v1.h"
#include "nursery_step_gen.h"
#include "train_generation.h"
 
#include "los.h"
#include "gc_hooks.h"
#include "gc_consts.h"
#include "gc_plan.h"
#include "gc_globals.h"

#ifdef ORP_POSIX
#include "linux_api.h"
#endif

#ifdef ORP_NT
#include "win32_api.h"
#endif

extern bool garbage_collector_is_initialized;
extern LARGE_INTEGER performance_frequency;
extern LARGE_INTEGER gc_start_time;
extern LARGE_INTEGER gc_end_time;
extern double elapsed_time;
//
// This global is defined in gc_interface.cpp, and is used to
// determine if an object is large, in routine gc_malloc_3.
// This may need to be cleaned up when we speed up allocation.
//
extern unsigned los_threshold_bytes;

extern Step_Plus_Nursery_Generation *p_young_gen;

class ORP_Control;

extern Block_Store *p_global_bs;

//extern Directory *p_global_gc_dir; obsolete: we squished this into the bs.

//
// A debug time pointer to the Card Table manager.
// THIS IS ALSO IN GC_GLOBALS.H, BUT DUE TO A VC++ BUG,
// THIS DUPLICATE PROTOTYPE IS INCLUDED HERE. (SEE ALSO
// GC_INTERFACE.CPP.
//
extern Card_Table *p_global_card_table_manager;

//
// This is a fast verification guard to ensure that the
// JIT is not enumerating at an inappropriate time.
// This could result in stale references.
//
bool _collection_in_progress;

    // There are two Remembered Sets that need to be maintained for
    // the train algorithm. Since the other algorithms are just subsets
    // of the train algorithm these sets also work for them.
    Remembered_Set *_p_refs_to_young_object_space;
    Remembered_Set *_p_refs_to_focus_car;

    // If there is a reference to the train it is noted here. This will
    // Keep the train alive.
    bool ref_to_focus_train_exists;

    Car *_p_focus_car;
    Train *_p_focus_train;

    //
    // The young generation has nurseries and steps.
    //
	Step_Plus_Nursery_Generation *_p_young_generation;

Mrl_Gc_V1::Mrl_Gc_V1(ORP_Control *p_orp_control,
					 char *p_plan_file_name)
: Gc_Interface(p_orp_control)
{

#if (GC_DEBUG>2)
	//
	// At debug time (fast verification), this goes on when
	// the GC stop-the-world kicks in.
	//
	_collection_in_progress = false;
#endif

    //
    // This is the interface object for communicating back to
    // the ORP.
    //
	_p_orp_control = p_orp_control;
 
    //
    // Set up the plan object which defines GC parameters,
    // and set up the right level of hooks, depending on
    // the debug level specified by the build.
    //
    _initialize_plan_and_hooks(p_plan_file_name);

	//
	// Make this a global for use by gc_malloc_3 in gc_interface.cpp
	// till we figure out the optimal code path for allocation.
	//
    // UNNECESSARY: INIT_GC HAS DONE THIS ALREADY. <<<<CHECK>>>>
#ifdef GC_COPY_V2
	los_threshold_bytes = _p_gc_plan->los_threshold_bytes();
#endif // GC_COPY_V2
#ifdef GC_GEN_V3
	los_threshold_bytes = _p_gc_plan->los_threshold_bytes();
#endif // GC_GEN_V3
    
    // Initialization (perhaps later the ORP should set defaults):
	if (initial_heap_size_bytes == 0) {
		initial_heap_size_bytes = _p_gc_plan->initial_heap_size_bytes();
	}

	if (final_heap_size_bytes == 0) {
		final_heap_size_bytes = _p_gc_plan->final_heap_size_bytes();
	}

	current_heap_size_bytes = initial_heap_size_bytes;


    unsigned int sub_block_size_bytes = _p_gc_plan->sub_block_size_bytes();

    unsigned long card_size_bytes  = _p_gc_plan->card_size_bytes(); 

    POINTER_SIZE_INT heap_base;

    _p_block_store  = 
        new Block_Store(_p_gc_hooks,
                        _p_gc_plan,
                        initial_heap_size_bytes,
		                final_heap_size_bytes,
		                sub_block_size_bytes);    

	heap_base = (POINTER_SIZE_INT)(_p_block_store->get_heap_base());
    //
    // Give the write barrier code global access to the heap base,
    // so that it can use this for card marking without needing
    // to make a function call.
    //
    global_heap_base = heap_base;

	p_global_bs     = _p_block_store;

#ifdef GC_GEN_V3
        p_global_card_table_manager =
            _p_card_table   = 
             new Card_Table(_p_gc_hooks,
                            _p_gc_plan,
                            final_heap_size_bytes,   // reserve the space for now
                            initial_heap_size_bytes, // the amount to use.
                            card_size_bytes,
                            _p_block_store,
					        heap_base);
#endif // GC_GEN_V3
 
    //
    // This is the remembered set used to store all the
    // weak references enumerated by the ORP.
    //
    _p_weak_references = new Remembered_Set();

    // These are the root sets that are required for each and every
    // collection. They are empty at the start of the collection, filled
    // by the orp the MOS remembered sets, or the collection of YOS.
    // They are emptied at the end of the collection.

    // A combination of the orp roots and MOS to YOS pointers.
    _p_refs_to_young_object_space = new Remembered_Set();
    // Starting with the _p_container_write_barrier for this car this
    // is augmented by the orp roots as well as the YOS to MOS pointers.
    // It is scanned for entries from outside the train is none are found
    // from YOS to MOS or from ORP to the focus train.
    // This can hold refs from both inside and outside of the train.
    _p_refs_to_focus_car = new Remembered_Set();

    // This is created as we go through the _p_refs_to_focus_car set
    // It is empty at the start of a collection prior to enumerating
    // the roots. It is augmented by looping through the non-focus cars in
    // the focus train.
//    _p_refs_to_focus_car_from_focus_train = new Remembered_Set();
    ref_to_focus_train_exists = false;

    // For the time being, create two generations:

	_number_of_generations = 2;

    // For GC_COPY_V2   we use a single younger generation with
    //                  there is a train but we do not tenure into it.
    //                  assert are set to ensure that no objects ever reside
    //                  in the train. 
    //                  There is also a fixed collector

    // For GC_GEN_V3    we have a nursery step generation and two trains.
    //                  Each train contains a single car that is large enough
    //                  to hold all object in the MOS generation. The youngest 
    //                  train spends it life empty except during a collection when
    //                  it exists only to receive object from the oldest
    //                  train. Since the oldest train will evacuate all its objects
    //                  during a GC_GEN_V3 collection it will be eliminated and
    //                  the youngest train will become the oldest train. 
    //
    //                  INVARIANT - we have only 2 trains and at the start and
    //                  end of each collection the youngest train is empty.
    //                  INVARIANT - a single car can hold all objects in MOS.
    //                  INVARIANT - no train has 2 cars.
    //                  
    //                  We tenure objects into the oldest train. 
    //                  For now the strategy is to have N steps and collect
    //                  MOS every N collections. This means that every live object
    //                  is moved at most N + M/N times where N is the number of steps
    //                  and M is the number of collection since the birth of the object.
    //
    //                  We also have a Fixed collector associated with the older space.
    //
    // For GC_TRAIN_V5
    //                  TBD - once I have done some tuning...
    //

    // Use the train algorithm here.
    _p_mature_generation = 
        new Train_Generation(1,
                             _p_gc_hooks,
                             _p_gc_plan,
		                     this,
							 NULL,
                             _p_card_table,
		                     _p_block_store);
    _p_young_generation = 
		new Step_Plus_Nursery_Generation(YOS_GEN_NUMBER,
                                         _p_gc_hooks,
                                         _p_gc_plan,
		                                 this,
#ifdef GC_COPY_V2 
                                         // There is not superior generation in the 
                                         // simple copy situation.
                                         NULL,
#else
										 _p_mature_generation,
#endif // GC_COPY_V2.
                                         _p_card_table,
		                                 _p_block_store);

    // This car and train are the focus of this collection.

    _p_focus_car = NULL;
    _p_focus_train = NULL;

	//
	// Cross-link to point each generation to the other:
	//
	_p_mature_generation->
		set_young_generation(_p_young_generation);

	_p_young_generation->
		set_mature_generation(_p_mature_generation);

    //
	// For the time being, keep all large objects in the youngest
	// generation and avoid tenuring them. The down side of this 
	// approach is that we will always scan these objects.
	//
	_p_los_container = new LOS_Container(this,
                                         _p_gc_hooks,
                                         _p_gc_plan,
		                                 _p_young_generation,
										 _p_card_table,
		                                 _p_block_store);
    //
    // This global is made available for simplicity during
    // allocation, since there are a number of confusing allocation
    // paths due to different available options during development.
    // Later on this needs to be removed, and the container that
    // allocates the finalizable object should send a message
    // directly to its containing generation.
    //
    p_young_gen = _p_young_generation;

	//
	// The block store needs to know about generations to qualify pointers.
	//
	_p_block_store->set_young_generation(_p_young_generation);
	_p_block_store->set_mature_generation(_p_mature_generation);
	_p_block_store->set_los_container(_p_los_container);

	_p_los_container->set_generation(p_young_gen);

	//
	// Finally register the GC as being fully initialized.
	//
	garbage_collector_is_initialized = true;
}

Mrl_Gc_V1::~Mrl_Gc_V1()
{
#if (GC_DEBUG>3)
    orp_cout << "Deleting Mrl_Gc_V1" << endl;
#endif
    //
    // Make sure the block store is deleted after all the
    // generations and containers (which still need use of
    // this block store for releasing blocks.)
    //
//	delete _p_root_set;
    delete _p_weak_references;
    delete _p_refs_to_young_object_space;
    delete _p_refs_to_focus_car;
//    delete _p_refs_to_focus_car_from_focus_train;

	delete _p_orp_control;
	delete _p_mature_generation;
//    if (_p_focus_car) {
//        orp_cout << _p_focus_car << endl;
//        delete _p_focus_car;
//        delete _p_focus_train;
//    }
	delete _p_young_generation;
	delete _p_los_container;
    //delete _p_fixed_object_store;
#ifdef GC_GEN_V3
	delete _p_card_table;
#endif // GC_GEN_V3
	delete _p_block_store;
}

//
// This is the last thing done by reclaim_heap, i.e. at
// the end of an incremental reclamation. This gives all
// generations a chance to clean up, then signals the 
// ORP to restart.
//
void
Mrl_Gc_V1::_clean_up_and_resume_orp()
{
    //
    // After all clean ups, verify the generations.
    //
#if (GC_DEBUG>3)
	_p_young_generation->verify_generation();

    _p_mature_generation->verify_generation();

#endif // _DEBUG

#if (GC_DEBUG>3) // debugging stuff
    //cout << "Dumping stats of LOS" << endl;
//    _p_large_object_store->verify_space();

    //cout << "Dumping stats of FOS" << endl;
    //_p_fixed_object_store->verify_space();
#endif

	_p_orp_control->resume_mutators();

	return;
}

// The thread is telling us that the nursery is exhausted.
// Add this to the used list and give the thread a new one.
// If all nurseries are exhausted, trigger a collection.
Nursery *
Mrl_Gc_V1::p_cycle_nursery(Nursery *p_nursery,
			               bool returnNullOnFail) 
{
    // This thread should not have nursery associated with it.
//    assert (orp_get_nursery() == NULL);
    assert (returnNullOnFail == false); // assert and debug...
    Nursery *result = _p_young_generation->p_cycle_nursery(p_nursery, 
	                                            returnNullOnFail);
    return result;
}


Remembered_Set *
Mrl_Gc_V1::_execute_minor_collection(Remembered_Set *_p_refs_to_young_object_space,
                                     Remembered_Set *p_weak_refs,
									 bool *p_is_train_alive) 
{
	// Pass the ORP live references to the youngest generation
    // to do a minor collection, and get
    // the young-to-old roots back.

    // RLH-TRAIN this YOS collection will also collect the
    // focus car if p_focus_car is not NULL. This means that
    // the p_reclaim_gernation for the mature space is only used
    // when we do a complete collection. The full colleciton is "to be done."
	Remembered_Set *p_young_to_old_references = 
        _p_young_generation->
            p_reclaim_generation(_p_refs_to_young_object_space,
                                 p_weak_refs,
								 p_is_train_alive);
#if (GC_DEBUG>2)
	assert(no_pending_cheney_scans());
#endif // _DEBUG
    // We won't be cleaning up for now since the MOS  collection may find some YOS pointers
//    orp_cout << "CLEANING UP FOR ONLY WHEN no focus_car..." << endl;
//    if (p_focus_car == NULL) {
    	//
	    // Give the generations a chance to clean up.
	    //
	    _p_young_generation->cleanup();

        // _p_mature_generation->cleanup();
	    //
	    // Give the Large Object Store a chance to clean up.
	    // // The sweep is done here... or we could do it while allocating if we
        // want better (hopefully) scalability and locality.
	    _p_los_container->cleanup();
        //
        // This alternate LOS is used in VERSION1 for avoiding
        // fragmentation problems (despite coalescing).
        //
        //_p_fixed_object_store->cleanup();
//    } 
    return p_young_to_old_references;
}

Car *
Mrl_Gc_V1::p_get_focus_car() {
    return _p_focus_car;
}

Train *
Mrl_Gc_V1::p_get_focus_train() {
    return _p_focus_train;
}

//
// Do one unit of incremental collection of mature space.
//
void
Mrl_Gc_V1::_execute_incremental_collection_of_mature_space(Remembered_Set *all_refs_to_focus_car,
														   Remembered_Set *p_weak_refs,
                                                           bool train_is_alive)
{
#ifdef GC_COPY_V2
    // Check to make sure nothing is in these generations.
    return;
#endif // GC_COPY_V2
	_p_mature_generation->
            p_reclaim_generation(all_refs_to_focus_car,
                                 p_weak_refs,
			                     train_is_alive);
}

void
Mrl_Gc_V1::free(Java_java_lang_Object *p_obj)
{
	// hint from the mutator: p_obj is toast.
}

void
Mrl_Gc_V1::freeze()
{
	// the mutator is informing us to cease and desist
	// from any barrier collection, probably because
	// some mutator activity (such as RNI native code)
	// has rendered moving of objects unsafe.
}

void
Mrl_Gc_V1::enumerate_weak_reference_queues ()
{
 
	_p_young_generation->enumerate_reference_queues ();
	_p_mature_generation->enumerate_reference_queues ();

}

//
// Just before doing a stop-the-world collection, we go to
// the ORP asking for all live references. In the current 
// scheme, rather than return the root set, the ORP makes
// a call back into the GC for each live reference. This is
// the interface routine gc_add_root_set_entry in gc\gc_interface.cpp
//
void
Mrl_Gc_V1::_get_orp_live_references()
{
#if (GC_DEBUG>2)
	//
	// Open a window during which the JIT can legitimately
	// enumerate live references. Fast verification for
	// debugging the JIT.
	//
	_collection_in_progress = true;
#endif


    enumeration_start_hook(_p_refs_to_young_object_space,
                           _p_refs_to_focus_car, 
                           _p_weak_references);

    //
	// First ask the mutator for the live references:
    //
#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - Empty _p_refs_to_young_object_space and _p_refs_to_focus_car." << endl;
#endif

    _p_refs_to_young_object_space->empty_all();
    _p_weak_references->empty_all();
    _p_refs_to_focus_car->empty_all();

    // Perhaps this is not where this should be done...
	//    _p_refs_to_focus_car_from_focus_train->empty_all(); 
 
    // This fills the _p_refs_to_young_object_space 
    // and _p_refs_to_focus_car.
    _p_orp_control->enumerate_live_references();
 
    // There are a reference queues associated with the step_generation and the
    // cars that need to be enumerated.
    enumerate_weak_reference_queues();

    enumeration_end_hook(_p_refs_to_young_object_space,
                         _p_refs_to_focus_car,
                         _p_weak_references);

#if (GC_DEBUG>2)
	//
	// Close the window during which the JIT can enumerate.
	//
	_collection_in_progress = false;
#endif
}

//
// This private method is called during the creation of the
// Mrl_Gc_V1 object in order to set up the hooks and 
// create the plan object.
//
void
Mrl_Gc_V1::_initialize_plan_and_hooks(char *p_plan_file_name)
{
    //
    // Create a plan object. This process will check if the
    // user has a .plan file in the right place. If so, the
    // contents of that file override the default settings
    // in the plan.
    //  
	if (p_plan_file_name) {
		_p_gc_plan = new Gc_Plan(p_plan_file_name);
	} else {
		_p_gc_plan = new Gc_Plan();
	}
}

#if (GC_DEBUG>3)
//
// Debug time routine for inspecting the state of the collector.
//
void 
Mrl_Gc_V1::inspect(unsigned int level)
{
	_p_young_generation->inspect(level);
	_p_mature_generation->inspect(level);
}
#endif // _DEBUG


//
// Verify that there are no pending cheney scans in any generation.
//
bool 
Mrl_Gc_V1::no_pending_cheney_scans() 
{
	if (_p_young_generation->cheney_scan_pending()) {
		return false;
	}

	if (_p_mature_generation->cheney_scan_pending()) {
        return false;
	}

	return true;
}

#if 1
//
// This code is called only at the end of the GC, and by software convention, no more calls
// are made to gc_add_root_set_entry_interior_pointer during this GC. 
// This code loops through the interior pointers updating each of them, one at a time.
// It then resets the interior pointer table so that it is empty in preperation for the 
// next GC.
// In the case of SAPPHIRE this routine is called for each thread whenever we update the
// root set for that particular thread. 
void fixup_interior_pointers ()
{
    void **slot;
    interior_pointer_table->rewind();
    while (interior_pointer_table->available()) {
        slot = interior_pointer_table->get_slot();
        *slot = (void *)((POINTER_SIZE_INT)interior_pointer_table->get_base() + 
                    (POINTER_SIZE_INT)interior_pointer_table->get_offset());
        interior_pointer_table->next();
    }
    // We are done with this round reset it for the next GC.
    interior_pointer_table->reset();
}
#endif // was GC_INTERIOR_POINTERS

//
//  reclaim heap is called when there are no longer any nurseries left to
//  allocate objects into or when there is no space in LOS to place an object.
//
//  One of several things can happen at this point. Traditionally this would
// indicate a garbage collection of at least the YOS and the LOS. It could mean that
// a single car is collected. Finally we are adding a new twist called a "declined
// collection."
//
//  A "declined collection" is a "logical" collection of the nurseries where all the objects
//  in the nurseries that are filled are moved en masse into the step generation. The gain
//  is that since the objects are not actually copied to new addresses and no objects are
//  being evicted then we do not have to stop threads or scan the cards. The downside is that
//  we do not collect any objects. All we have succeeded in doing is moving objects that are
//  live at this time from the nurseries to the step.
//
//  Motivation: Several programs create large data structures at the beginning of
//      execution and these objects live for a long time. If we would copy them from
//      the nurseries into a step and then into MOS they would have been copied twice.
//      If however we use the empty space in the steps to stash the objects into so they
//      can age a little before they are moved into MOS then we end up only copying the
//      objects once and having to scan stacks and heaps less.
//      Of course this has the cost of doubling the space needed for nurseries but if the
//      space is not needed for anything else it is a win. Obviously the space isn't
//      being used during the first collection.
// 
// Argument size:
//  The size argument is of interest only in the case of the LOS container wanting more
//  space. If the size is 0 then we just need nurseries for small objects. If the size
//  is >0 then it indicates how much space we need.
//

// The convention here is that the caller holds and will relinquish the gc_lock.

int _yos_collections_since_mos_collection = 0;

int _calls_since_yos_collection = 0;

volatile bool gc_just_completed = false;
volatile bool mos_collection_was_needed = true;
volatile int number_of_blocks_freed_last_yos = 0;
volatile int number_of_blocks_freed_next_to_last_yos = 0;
volatile int number_of_blocks_freed_third_to_last_yos = 0;
volatile int ave_blocks_recovered_by_yos = 0;

void
Mrl_Gc_V1::reclaim_heap(unsigned int size, bool forced)   // This guy needs to know why he was called was it
                            // an the LOS that ran out of space or was it a nursery.
                            // a true forced indicates a forced gc.
{

#ifdef GC_SAPPHIRE
    p_sapphire_lock->_lock();
#endif // GC_SAPPHIRE
    gc_start_hook(this);
    global_gc_count++; // Also set at end of reclaim_heap.
	//
    // Since the enumeration needs to have these initialized do it before
    // we get the live references from the orp.
    //
    _p_focus_car = _p_mature_generation->p_nominate_focus_car();
	_p_focus_train = _p_mature_generation->p_nominate_focus_train();

#ifdef GC_SAPPHIRE
	extern void sapphire_driver();

    unsigned long sapphire_number_of_spent_nurseries = _p_young_generation->get_number_of_spent_nurseries();
    unsigned long sapphire_number_of_nurseries = _p_young_generation->get_number_of_nurseries();
//    orp_cout << "Number_of_spent_nurseries " << sapphire_number_of_spent_nurseries << endl;
//    orp_cout << "Number_of_nurseries " << sapphire_number_of_nurseries << endl;
//    orp_cout << "s";
    sapphire_driver();
    p_sapphire_lock->_unlock();
//    orp_cout << "Heap reclaimed";
    gc_end_hook(this);
      return;
#endif // GC_SAPPHIRE


#if (GC_DEBUG>0)
	assert(no_pending_cheney_scans());
#endif
    //
	// Do a minor collection. Pass pointers to the oldest car in
	// the oldest train so that the younger generation can be more focused in
	// the young-to-old pointers that are returned. (We only need
	// pointers into the oldest, ie focus_car, along with a boolean
	// telling us if there are any pointers into the oldest train.)

    ref_to_focus_train_exists = false;

    //
    // This is a simple heuristic check to see when we need to collect the older generation.
    // For a simple generational collector available free space as a function of the
    // preset nursery size seems to work well. For the train algorithm the decision to
    // collect a car will be triggered at a lower level since the work required and the
    // amount we expect to collect will be much lower than with a simple generational 
    // collector.
    //
    if (stats_gc) {
        orp_cout << "_yos_collections_since_mos_collection is " 
            << _yos_collections_since_mos_collection << endl;
    }
    _calls_since_yos_collection += 1;
    
    if (stats_gc) {
        orp_cout << "_calls_since_yos_collection is " 
            << _calls_since_yos_collection << endl;
    }
    
    //    orp_cout << "Free blocks at this collection is " 
    //             << p_global_bs->get_free_sub_blocks() << endl;

    //
    // Both final_heap_size_bytes and initial_heap_size_bytes are
    // available for use here. Perhaps we could come up with a heuristic
    // that would allow us to do deferred collection until we have eaten up
    // 80 % of the current available space and then do a YOS collection knowing that
    // we will be able to extend if need be.
    // Is there a current heap size available??
    //
    // We could keep track of the number of deferred collection since the
    // last collection and limit it to something reasonable so we don't expect a
    // pause of more than say 50 milliseconds. How much can I copy in 50 milliseconds.
    //
    // Keeping track of the amount recovered in past collections might give us a feel
    // for how long we can go without doing a new collection.
    //

    unsigned long number_of_large_nurseries = _p_gc_plan->number_of_large_nurseries ();
    unsigned long blocks_in_large_nursery = _p_gc_plan->large_nursery_size_sub_blocks();
    unsigned long blocks_in_large_nurseries = number_of_large_nurseries * blocks_in_large_nursery;
    unsigned long free_blocks = p_global_bs->get_free_sub_blocks ();
    unsigned long total_blocks_allocated = p_global_bs->get_total_current_sub_blocks();
    unsigned long total_blocks_in_use = total_blocks_allocated - free_blocks;
    unsigned long number_of_spent_nurseries = _p_young_generation->get_number_of_spent_nurseries();
    unsigned long number_of_nurseries = _p_young_generation->get_number_of_nurseries();
    // This is the limit of the number of blocks that can be allocated.
    unsigned long maximum_number_of_blocks = p_global_bs->get_final_heap_size_blocks();
    unsigned long number_of_threads = thread_gc_number_of_threads() + 1;    

//    orp_cout << "------ At quarter way point" << endl;
//    gc_end_hook (this);
//    gc_start_hook (this);

    // 2.5 X seems to work well for small heap sizes. This could be a function of
    // something a little better. But it is very application specific.
    // RESEARCH OPPORTUNITY....

    bool do_mos_collection = false;

    // Allow for an overdraft since we can extend the space if the overdraft is to much. 
    bool do_yos_collection =  
        (total_blocks_in_use > (free_blocks * 3)); 

//    orp_cout << " Force MOS Collection" << endl;

//    do_mos_collection = true;
//    do_yos_collection = true;

    if (stats_gc) {
        orp_cout << " do_yos_collection is " << do_yos_collection << endl;
        orp_cout << " do_mos_collection is " << do_mos_collection << endl;
    }

    if (_calls_since_yos_collection > 16) {
        do_yos_collection = true;
    }

    if (_yos_collections_since_mos_collection > 12) {
        do_mos_collection = true;
        do_yos_collection = true;
    }

    //
    // Figure out if there is enough free blocks to ensure a full generational collection.
    // This is a safety issue.
    // We need also to take into account the number of blocks in nurseries since
    // we may well be moving these blocks into the step data structure, thus reducing 
    // the number of blocks that will be available at the next collection.

    long int blocks_to_play_with = maximum_number_of_blocks - (total_blocks_in_use * 2);

    // The first 16 nurseries are accounted for by blocks_in_large_nurseries. The other
    // nurseries are only one block each.

    blocks_to_play_with = blocks_to_play_with 
        - number_of_spent_nurseries + blocks_in_large_nurseries ;

    blocks_to_play_with = blocks_to_play_with + ave_blocks_recovered_by_yos;

    bool full_collection_safe = blocks_to_play_with > 0;

    if (!full_collection_safe) {
        // In order to be safe if we are running low on space do a collection.
        if (mos_collection_was_needed) {
            // last time we did an MOS it was needed so assume it is needed this time.
            do_mos_collection = true;
        }
        do_yos_collection = true;
        if (stats_gc) {
           orp_cout << "       Out of blocks to play with, MOS collection is true." << endl;
        }
    }

    if (forced) {
        do_yos_collection = true;
        if (verbose_gc) {
            orp_cout << "     -- GC requested by Java program." << endl;
        }
        if (_p_focus_car != NULL) {
            do_mos_collection = true;
            do_yos_collection = true;
        } else {
            if (stats_gc) {
                orp_cout << "    -- GC notes that no focus car exists so MOS collection is not done." << endl;
            }
        }
    }
    if (stats_gc) {
        orp_cout << "        Maximum_number_of_blocks = " << maximum_number_of_blocks << endl;
        orp_cout << "        Blocks in use = " << total_blocks_in_use << endl;
        orp_cout << "        Blocks to play with = " << (void *)blocks_to_play_with << endl;
        orp_cout << "        Focus car is " << _p_focus_car << endl;
        orp_cout << "        Do MOS collection is " << do_mos_collection << endl;
        orp_cout << "        Free blocks = " << free_blocks << endl;
        orp_cout << "        Number of spend nurseries = " << number_of_spent_nurseries << endl;
        orp_cout << "        Number of nurseries = " << number_of_nurseries << endl;
        orp_cout << "        Number of threads = " << number_of_threads << endl;
    }
    
    if (verbose_gc) {
        orp_cout << "<gc enter>" << endl;
    }

    // At this time check to see if we really want to do a collection or whether
    // we can get by with less drastic action. 
    if (stats_gc) {
        orp_cout << " do_yos_collection is " << do_yos_collection << endl;
        orp_cout << " do_mos_collection is " << do_mos_collection << endl;
    }

    if (number_of_nurseries < (number_of_threads * 2)) {
        if (verbose_gc) {
            orp_cout << "<-- Adding nurseries to accomodate additional threads >" << endl;
        }
        _p_young_generation->add_nurseries();
        gc_end_hook(this);
        return;
    }

    if (!do_mos_collection && !do_yos_collection) {
        // There seem to be enough extra blocks laying around that we can put off
        // a collection. Check to see if we were called due to LOS running out of
        // space.
        
        if (stats_gc) {
            orp_cout << " Avoiding collection if possible, size is . " << size << endl ;
        }

        if (size > 0) {
            
            unsigned int size_in_blocks = 
                (unsigned int)(size /  _p_gc_plan->sub_block_size_bytes()) + 1;
            // MORE MAGIC TUNING NUMBERS THAT NEED TO BE ELEVATED.
            if (((size_in_blocks) + (blocks_in_large_nurseries * 3)) <  free_blocks) {
                if (stats_gc) {
                    // Yes we need some LOS space. See if we have enough extra blocks
                    // without having to resort to a collection.
                   orp_cout << " -- Avoiding collection by extending pinned areas. " << endl;
                   orp_cout << " Current free blocks before extending: " << p_global_bs->get_free_sub_blocks () << endl;
                }
                _p_los_container->extend_los (size);
                if (verbose_gc) {
                    orp_cout << " Blocks Extend =  " << p_global_bs->get_free_sub_blocks () << " >"<< endl;
                }

                gc_end_hook(this);

                return;
            }

            do_yos_collection = true;
            if (stats_gc) {
                orp_cout << " We decided not to extend los. " << endl;

            }
        } else {

            // GC_FALSE_COLLECTION
            // Decline the collection if it can just move the blocks associated 
            // with nurseries into steps.

            // We have lots of free space, so much that we won't even do a collection.
            // Instead we will just move what is in the nursery into the step logically.
            // We will then refill the nursery with free blocks. We will do this
            // without stopping the threads and enumerating their stacks.
 
            // We have a lot of space but are there a lot of spent nurseries. We
            // could be here because we have a lot of threads and not enough nurseries.
            
            if ((number_of_spent_nurseries * 4) < (number_of_nurseries) ) {
                if (stats_gc) {
                    orp_cout << " -- Deferring collection by allocating new nurseries." << endl;
                    orp_cout << "Number of spend nurseries = " << number_of_spent_nurseries << endl;
                    orp_cout << "Number of nurseries = " << number_of_nurseries << endl;
                }
                if (verbose_gc) {
                    orp_cout << "<GC deferred>" << endl;
                }
                //
                // We have space but no spent nurseries so just add nurseries and
                // return.
                //
                // Add nurseries if we can and less than 25% of the nurseries have been
                // spent.
                //
                _p_young_generation->add_nurseries ();

                gc_end_hook(this);
                return;
            } else {

                // Do the false collection by moving the spent_nurseries into the steps.
                // This will also reallocate the correct number of blocks to the steps.

                if (stats_gc) {
                    orp_cout << " -- Deferring collection by moving nurseries into step." << endl;
                }   
                bool success = _p_young_generation->move_spent_nurseries_into_step();
    
                if (success) {
                    
                    if (verbose_gc) {
                        orp_cout << "<GC deferred>" << endl;
                    }
                    // There was enough space so we moved the nurseries.
                    gc_end_hook(this);

                    return;
                } else {                
                    if (stats_gc) {
                        orp_cout << "******* Unable to move_spent_nurseries into a step ******" << endl;
                    }
                }

                //
                // Something went wrong so fall through and try to do a full collection.
                // Perhaps there were not enough contiguous blocks to refill all the 
                // nurseries.
            }
        }
    }
    
    if (stats_gc) {
        orp_cout << " do_yos_collection is " << do_yos_collection << endl;
        orp_cout << " do_mos_collection is " << do_mos_collection << endl;
    }

    // Reset the number of calls since we did a YOS collection

    _calls_since_yos_collection = 0;

    // Only collect the older generation every MUMBLE collections.
    // For now pound away.
    if (!do_mos_collection) {
        _yos_collections_since_mos_collection += 1;
        if (_yos_collections_since_mos_collection < 8) {
            _p_focus_car = NULL;
            _p_focus_train = NULL;
            if (stats_gc) {
                orp_cout << " -- Collecting YOS." << endl;
            }
        } else {
            if (_p_focus_car) {
                _yos_collections_since_mos_collection = 0;
                if (stats_gc) {
                    orp_cout << " -- Doing MOS collection." << endl;
                } 
            } else {
                if (stats_gc) {
                    orp_cout << " -- Doing YOS collection." << endl;
                }
            }
        }
    } else {
        if (_p_focus_car) {
            _yos_collections_since_mos_collection = 0;
            if (stats_gc) {
                orp_cout << " -- Doing MOS collection." << endl;
            }
        } else {
            // We wanted to do an MOS collection but MOS was empty.
            // Wait another round to do one now.
            _yos_collections_since_mos_collection = 0;
            if (stats_gc) {
                orp_cout << " -- Doing YOS collection." << endl;

            }
        }
    }

#if (GC_DEBUG>1)
    if (_p_focus_train && !_p_focus_car) {
        // We have what amounts to an empty train.
        assert (_p_focus_train->get_number_of_cars() == 0);
    }
#endif

    // Since any previous versions are obsolete we clear out of the root sets
    // needed in _get_orp_live_references.

    //
    // Ask the ORP for all live references. The ORP will
    // make a series of callbacks into the GC for each
    // live reference. (See gc\gc_interface.cpp, function
    // gc_add_root_set_entry). This results in _p_refs_to_young_object_space and
    // _p_refs_to_focus_car being filled in. 
    //                          
    //
    _get_orp_live_references();

    // Now that we have all the references from the orp into the YOS 
    // in _p_refs_to_young_object_space we need to add all the references
    // from MOS into YOS which are currently in _p_old_to_young_rememberd_set
    // into _p_refs_to_young_object_space. Since all the objects in YOS will
    // be moved we will will empty _p_refs_to_young_object_space and then
    // refill it because we will need it for our next GC. In addition to
    // the population of _p_refs_to_young_object_space done during this
    // collection we will need to augment it with the refs stored in MOS by
    // the mutator before the next collection. This could be done using a card 
    // table or directly using a remembered set add.
   
#ifdef GC_GEN_V3
    //
    // Also for the GC_TRAIN_V5 we need to do this.
    //
    // If we are doing card marking, this routine will scan the dirty cards
    // to discover interesting references (from mature to young space, or from
    // younger to older car.) This routine is currently called here (synchronously
    // during a stop-the-world collection.) However it can (and probably should) be
    // run concurrently with the mutator.
    //
    _p_card_table->scan_cards(_p_mature_generation,
                              _p_young_generation);
#endif // GC_GEN_V3

#ifdef GC_GEN_V3

    //
    // Add refs in _p_old_to_young_remembered_set into _p_refs_to_young_object_space.
    //

//    orp_cout << "Adding old_to_young_write_barrier."<< endl;

//    Remembered_Set *p_saved_old_to_young_write_barrier = 
//        _p_young_generation->get_p_old_to_young_remembered_set()->clone();

    // Scan the entire mature area and add whatever you find to 
    // the _p_refs_to_young_object_space.
//    _p_card_table->scan_mature_generation(_p_mature_generation, _p_young_generation);

        // MERGE THESE SO WE ARE SURE WE DON'T MISS ANY REFERENCES OF OBJECTS
        // THAT MIGHT BE PROMOTED.
        // Some object will be promoted that were pointed to from the stack/orp
        // roots. We need to figure out where these references are. This is not
        // the quickest way to do it, We could collect both the focus car and
        // the YOS at the same time and promote directly into the MOS into
        // someother train than the oldest one with the focus car. But for now...
        // We do it now before we merge the old to young remset entries since these
        // will be uninteresting since they are objects in MOS.
 
    if (_p_focus_car) {
        _p_refs_to_focus_car->merge(_p_refs_to_young_object_space); 

#ifdef GC_TRAIN_TRACE
        orp_cout << "Train trace - _p_focus_car is " << _p_focus_car << endl;
        orp_cout << "Train trace - _p_focus_train is " << _p_focus_train << endl;
#endif // GC_TRAIN_TRACE

    }

    
    // In an ideal world this extra scan would not be needed and we should
    // compare the two remembered sets to figure out which entries are missing
    // (if any are).

    //
    // Now merge the old to young references with the _p_refs_to_young_object_space.
    //
    _p_refs_to_young_object_space->merge 
        (_p_young_generation->get_p_old_to_young_remembered_set());

#if (GC_DEBUG>3)
    Remembered_Set *p_saved_root_set = _p_refs_to_young_object_space->p_clone(); // save this away
    _p_refs_to_young_object_space->verify_no_obsolete_references();
    p_saved_root_set->verify_no_obsolete_references();
    _p_refs_to_focus_car->verify_no_obsolete_references();
    _p_young_generation->verify_generation();
	_p_mature_generation->verify_generation();
#endif // GC_DEBUG>3

    // At this point _p_refs_to_young_object_space holds *all* pointers into
    // YOS. In addition no other remembered set has *any* pointers into YOS.

#if (GC_DEBUG>3)
    // To be done....
    // We need to verify this with a scan of MOS to see if we can find any pointers 
    // into YOS that do not exist in the _p_refs_to_young_object_space

    //    _p_mature_generation->scan_and_assert_membership(_p_refs_to_young_object_space);

#endif // GC_DEBUG>3

#endif // GC_GEN_V3

    // Now empty _p_old_to_young_remembered_set so it can be recreated by this
    // collection for use in the next collection. Since everything in YOS will
    // be rewritten we can rebuild the remembered set each time we
    // collect this area. This will be done as we process the _p_refs_to_young_object_space
    // collection. Each time we find a reference into YOS the object will be moved
    // if it hasn't already been moved and as we update the reference to the new
    // object location we will rebuild the _p_old_to_young_remembered_set.

    // If at some later time we collect the MOS area and move objects this newly
    // create write barrier will need to be recreated.

    // **************************************************************************************
    // GC_TRAIN_V5 will collect the focus car and will update pointer in the focus train
    // that point into YOS. If the car or train is reclaimed then the remembered set pointers
    // need to be removed.
    // **************************************************************************************
    //    orp_cout << "Emptying old to young remembered_set" << endl;
    _p_young_generation->get_p_old_to_young_remembered_set()->empty_all();

	//
	// This flag will be passed back from a minor reclamation, informing
	// us if there are pointers from the younger generation into the
	// oldest train.
	//

	bool train_is_alive = ref_to_focus_train_exists;

    if (_p_focus_car != NULL) {
//        orp_cout << "We have a focus car and are collecting a car." << endl;
        _p_mature_generation->set_executing_mature_generation_collection (true);
    } else {
//        orp_cout << "We don't have a focus car and aren't collecting a car." << endl;
        _p_mature_generation->set_executing_mature_generation_collection (false);
    }


    // I think this logic can keep trains alive for one round too long.
    // If the only pointers into the train are to the focus car then
    // the car will be evacuated and the train may well need to be reclaimed.
    // Certainly if it is the last car in the train.

    // Evict and scavange all objects with pointers from everywhere BUT the focus car if
    // one exists. Invariant - no slots in the focus car will point to "TO" space in
    // the YOS. When we collect MOS and the focus car objects will be evicted from the
    // focus car *before* they are scanned and their slots fill up with pointers to 
    // the "TO" space.

    // p_young_to_old_refs is ignored by the NEW_CHENEY_CODE.
    assert (do_yos_collection);

    Remembered_Set *p_young_to_old_refs =
        _execute_minor_collection(_p_refs_to_young_object_space,
                                  _p_weak_references,
                                  &train_is_alive);
    
#ifdef GC_TRAIN_V5
    Car *_p_focus_car_check = _p_mature_generation->p_nominate_focus_car();
    Train *_p_focus_train_check = _p_mature_generation->p_nominate_focus_train();

    if (_p_focus_train_check && !_p_focus_car_check) { // We have an empty train.
        //
        // Remove it. MOS collection will not happen since focus car is NULL.
        //
        assert (_p_focus_car == NULL);
#ifdef GC_TRAIN_TRACE
        orp_cout << "Train trace - Removing oldest train since it holds no cars." << endl;
#endif
        _p_mature_generation->free_empty_oldest_train();
    }
#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - end of minor collection." << endl;
#endif

#endif // GC_TRAIN_V5

        if (blocks_to_play_with < 0) {
            blocks_to_play_with = maximum_number_of_blocks - (total_blocks_in_use * 2);
            if (blocks_to_play_with > 0) {
                mos_collection_was_needed = false;
            } else {
                mos_collection_was_needed = true;
            }

        }

    //
    // Verify the generations, they should be stable at this point.
    //
    
	assert(no_pending_cheney_scans());

#ifndef GC_FIXED_V1

    //
    // Add the refs from MOS into the focus car to the refs from the ORP.
    // This is done after the YOS collection since tenuring could
    // add refs.
    //

    if (_p_focus_car) {
        // For GC_GEN_V3 this should be empty.

#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
        assert (_p_focus_car->get_car_remembered_set()->is_empty());
#endif // GC_GEN_V3
#endif // GC_TRAIN_V5

        // Add the cars remembered set so that all pointers into the car are now in 
        // a single remembered set. Including refs from the same train.

        _p_refs_to_focus_car->merge (_p_focus_car->get_car_remembered_set());
        // Add the young to old refs also.

        _p_refs_to_focus_car->merge (p_young_to_old_refs);
        p_young_to_old_refs->empty_all(); // This is rebuild at next GC.

    } else {
        assert (_p_refs_to_focus_car->is_empty());
    }

#if (GC_DEBUG>3)
    Remembered_Set *_p_saved_refs_to_focus_car = _p_refs_to_focus_car->p_clone();    
#endif // (GC_DEBUG>3)

    //
	// Now incrementally collect the older generation:
    //

    if (_p_focus_car) {

#if(GC_DEBUG>3)
        _p_saved_refs_to_focus_car->verify_no_obsolete_references();
        // Check to make sure there are no refs in the focus car!!!
#endif // GC_DEBUG>3
        //
        // If there was a focus car then the YOS and the ORP did their
        // part to make sure that all the refs into it are in _p_refs_to_focus_car
        // However, if there was not a focus car then that processing was
        // not done.
        //
#ifndef GC_TRAIN_V5

        //
        // Rebuild this write barrier. during the incremental collection of
        // MOS. This will work for our simple generational collector since
        // we will be collecting the entire MOS but once we move to the
        // train we will need to only remove entries that pointed to the 
        // focus car.
        //

        // When we delete a car we selectively flush objects out of this set.
        _p_young_generation->get_p_old_to_young_remembered_set()->empty_all();
#endif // GC_TRAIN_V5
#if 0 // (GC_DEBUG>0)
        // Make sure all the p_incoming_refs are valid.
        _p_refs_to_focus_car->rewind();
        Java_java_lang_Object **pp_obj_ref_quick_check;
        while (pp_obj_ref_quick_check = _p_refs_to_focus_car->next()) {
            if (*pp_obj_ref_quick_check == NULL) {
                assert (0);
            }
        }
#endif
#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - start of MOS collection." << endl;
#endif

        _p_mature_generation->set_executing_mature_generation_collection (true);
        
//        orp_cout << "Starting MOS collection" << endl;
        // Finalizable object that were discovered during the YOS collection are waiting to
        // be finalized and are on the Objects_To_Finalize queue. These objects need to be
        // enumerated or they will not be scanned. If they are not scanned then they could point
        // to the old copies of objects and when they are finalized these old objects would be
        // bogus.

        orp_enumerate_objects_to_be_finalized();

        _execute_incremental_collection_of_mature_space(_p_refs_to_focus_car,
                                                        _p_weak_references,
		                                                train_is_alive); 
//        orp_cout << "Train trace - end of MOS collection." << endl; 
    }

#if (GC_DEBUG > 0)
	assert(no_pending_cheney_scans());
#endif

#if (GC_DEBUG>3)
    // Is it possible that we have pointers in this set that point to 
    // the focus car and after we collect the focus car and delete it
    // this set will hold bad references.
    // _p_saved_refs_to_focus_car->verify_no_obsolete_references();
    delete _p_saved_refs_to_focus_car;
	_p_young_generation->verify_generation();
    _p_mature_generation->verify_generation();
    //   _p_refs_to_focus_car->verify_no_obsolete_references();
#endif // GC_DEBUG>3

#if (GC_DEBUG>0)
    // Verify that the old to young remembered set is valid.
    _p_young_generation->verify_old_to_young_write_barrier();
	assert(p_young_to_old_refs->is_empty());
	assert(no_pending_cheney_scans());
#endif // GC_DEBUG >3

#endif //  ndef GC_FIXED_V1

	//
	// Empty the remembered set.
	//

    p_young_to_old_refs->empty_all();

#if 1
    fixup_interior_pointers();
#endif // was GC_INTERIOR_POINTERS

#ifdef GC_PT_WB
    // Clear the page dirty bits
    PVOID heap_base = p_global_bs->get_heap_base();
    SIZE_T region_size = p_global_bs->get_heap_size_bytes();
    UINT foo = dll_ResetWriteWatch (heap_base, region_size);
#endif // GC_PT_WB

    // Verify generations if gc_debug is high enough.
    _clean_up_and_resume_orp();
 
    if (size > 0) {
        if (! _p_los_container->is_block_available(size)) {
            _p_los_container->extend_los (size);
        }
    } 

    if (_p_focus_car == NULL) {
        number_of_blocks_freed_third_to_last_yos = number_of_blocks_freed_next_to_last_yos;
        number_of_blocks_freed_next_to_last_yos = number_of_blocks_freed_last_yos;
        number_of_blocks_freed_last_yos = p_global_bs->get_free_sub_blocks () - free_blocks;
        // Now determine if we need to do a MOS next time around.
        ave_blocks_recovered_by_yos = (number_of_blocks_freed_third_to_last_yos + number_of_blocks_freed_next_to_last_yos +
        number_of_blocks_freed_last_yos) / 3;
    }
    if (verbose_gc || stats_gc) {

        number_of_large_nurseries = _p_gc_plan->number_of_large_nurseries ();
        blocks_in_large_nursery = _p_gc_plan->large_nursery_size_sub_blocks();
        blocks_in_large_nurseries = number_of_large_nurseries * blocks_in_large_nursery;    
        free_blocks = p_global_bs->get_free_sub_blocks ();
        total_blocks_allocated = p_global_bs->get_total_current_sub_blocks();
        total_blocks_in_use = total_blocks_allocated - free_blocks;
        number_of_spent_nurseries = _p_young_generation->get_number_of_spent_nurseries();
        number_of_nurseries = _p_young_generation->get_number_of_nurseries();
        // This is the limit of the number of blocks that can be allocated.
        maximum_number_of_blocks = p_global_bs->get_final_heap_size_blocks();
        if (stats_gc) {
            orp_cout << "       Collection is done here is the status." << endl;
            orp_cout << "        Maximum_number_of_blocks = " << maximum_number_of_blocks << endl;
            orp_cout << "        Blocks in use = " << total_blocks_in_use << endl;
            orp_cout << "        Free blocks = " << free_blocks << endl;
            orp_cout << "        Number of spend nurseries should be 0 = " << number_of_spent_nurseries << endl;
            orp_cout << "        Number of nurseries = " << number_of_nurseries << endl;
            orp_cout << "       End of status." << endl;
        }
        if (verbose_gc) {
            orp_cout << "<gc exit free blocks = " << free_blocks << ">" << endl;
        }
    }

    //
    // _after_stop_the_world_hook is called in the above 
    // cleanup routine *before* re-starting the ORP.
    //
    gc_end_hook(this);

    return;
}

void
Mrl_Gc_V1::run_all_finalizers()
{
    _p_young_generation->run_all_finalizers();
    _p_mature_generation->run_all_finalizers();
}

void
Mrl_Gc_V1::set_verbose_level(unsigned int level)
{
	_p_gc_plan->verbose_gc = level;
}

//
// ORP notification to GC to clean up.
//
void 
Mrl_Gc_V1::wrapup_gc() 
{
	//
	// Currently does nothing.
	//
}

	
//
// Take a spent nursery and store it away till 
// the next reclaim.
//
void 
Mrl_Gc_V1::store_spent_nursery(Nursery *p_nursery) 
{
	_p_young_generation->store_spent_nursery(p_nursery);
}


//
// The ORP is providing us with yet another live reference.
// We add it to our root set. This happens at the beginning
// of a stop-the-world phase, when we call into the ORP
// to enumerate all live references.
//
void Mrl_Gc_V1::gc_add_root_set_entry(Java_java_lang_Object **ref) 
{
	gc_trace ((void *)*ref, "gc_add_root_set_entry enumerates this pointer"); 
#ifdef GC_SAPPHIRE
	extern void sapphire_add_root_set_entry(Java_java_lang_Object **ref);
	// let sapphire deal with this.
	sapphire_add_root_set_entry(ref);
	return;
#endif // GC_SAPPHIRE

#if (GC_DEBUG>2)
		assert(_collection_in_progress);
#endif
//        orp_cout << "-";
        // Split the remembered set entry into one of four different
        // catagories.

        // Is is NULL.  The NULL reference is not interesting.
        if (*ref == NULL) {
            return;
        }
 
        // Is it a pointer into the youngest generation?
        if (_p_young_generation->is_address_in_my_generation (*ref)) { 
            gc_trace ((void *)*ref, "gc_add_root_set_entry enumerates this YOS pointer"); 
            _p_refs_to_young_object_space->add_entry ((Java_java_lang_Object **)ref);
            return;
        }
        
        if (!_p_focus_car) {  
            gc_trace ((void *)*ref, "gc_add_root_set_entry ignores this MOS pointer - no focus car"); 
            // We do not have a focus car so we can ignore this slot.
            // We are not doing a MOS collection this time through.
//            // We probable have an empty train. Check it out and then return.
//            assert(_p_focus_train);
//            assert(_p_focus_train->get_number_of_cars() == 0);
//            assert(ref_to_focus_train_exists == false);
            return;
        }

        // We we got this far we should have a focus car and focus train.
#if (GC_DEBUG>1)
        assert (_p_focus_car);
        assert (_p_focus_train);
#endif // GC_DEBUG>1
        // We know it isn't null or a pointer into YOS so it is a pointer into
        // MOS. Is it to the target car?

        if (_p_focus_car->is_address_in_this_car (*ref)) {
 
    		gc_trace ((void *)*ref, "gc_add_root_set_entry adds this ref to focus car"); 
            _p_refs_to_focus_car->add_entry ((Java_java_lang_Object **)ref);
            ref_to_focus_train_exists = true;
        } else {

#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
            assert (0); // We have only one train and one car so the above
                        // conditional had better have been true.
                        // RLH-TRAIN if we promote into the youngest train
                        // Then we will have two trains alive.
#endif
#endif // GC_TRAIN_V5
            // Is it into the target train?
            if (!ref_to_focus_train_exists) {
                if (_p_focus_train->is_my_address(*ref)) {
                    
 
            		gc_trace ((void *)*ref, "gc_add_root_set_entry in focus train but not focus car"); 

                    ref_to_focus_train_exists = true;
                    return;
                }
            }
            
 
    		gc_trace ((void *)*ref, "gc_add_root_set_entry in MOS but not focus train (or focus car)");
 
        }
//		_p_root_set->add_entry((Java_java_lang_Object **)ref);
}


// end file gc\mrl_gc_v1.cpp

