/*
**	MMU_V2.c 1998-02-01 Mikael Forselius <mikaelf@comenius.se>
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
**	Determine memory mappings
**
**	Inspiration and code from the OpenBSD effort.
*/

#include <QuickDraw.h>
#include <Gestalt.h>

#include "penguin_prototypes.h"
#include "MMU.h"
#include "MMU_V2.h"

// Defs
#define NBPG			(4096)
#define MMU_68851		0
#define MMU_68030		1
#define MMU_68040		2
#define MMU40_RES		0x00000001
#define MMU40_TTX		0x00000002

#define NUM_MEMCHUNKS	32

typedef struct MemChunk {
	u_long			lo;
	u_long			hi;
	u_long			logical;
	u_long			size;
} MemChunk;

typedef struct hw_dev {
	u_long		lmem_addr;
	char		*name;
	char		*note;
} hw_dev;

// Prototypes [mmu030/040.a | mmu68851.a]
extern void			Test030(...);

// Transparent translation regs
extern void			Get030_TTx(u_long *tt0, u_long *tt1);
extern void			Get040_xTTx(u_long *itt0, u_long *dtt0, u_long *itt1, u_long *dtt1);

// Translation control reg
extern u_long		Get020_TC(void);
extern u_long		Get030_TC(void);
extern u_long		Get040_TC(void);

// CRP and SRP for the 020/030, URP and SRP for the 040
extern u_long		Get030_xRP(u_long crp[2], u_long srp[2]);
extern u_long		Get040_xRP(u_long *urp, u_long *srp);

// Translated Page Table Entry routines
extern u_long		Get68851_PTE(u_long addr, u_long pte[2], u_short * psr);
extern u_long		Test030_TTx(u_long addr);
extern u_long		Get030_PTE(u_long addr, u_long pte[2], u_short * psr);
extern u_long		Get040_PTE(u_long addr);

// Prototypes [local]
static u_short		init_mmu_vars(void);
static u_long		logical_to_physical(u_long addr, u_long * phys, int fail);
static u_long		setup_ram_mappings(void);
static int			get_top_of_ram();
static u_long		get_rbv_size();

// Locals
static u_long		sMacOS_videoaddr = 0;	// convienient address...
static u_long		sMacOS_tc = 0;			// translation control
static u_long		sMacOS_ps = 0;			// page size (in bytes)
static u_long		sMacOS_mmu = 0;			// MMU type
static u_short		sMacOS_mmu_enable = 0;	// MMU enable setting

static u_long		sNumMemChunks;
static MemChunk		sMemChunks[NUM_MEMCHUNKS];

hw_dev	hw_dev_list[] = {
	{ 0x02AE,	"ROM",		"" },
	{ 0x01D4,	"VIA1",		"" },
	{ 0x0CEC,	"VIA2",		"(VIA2/RBV/OSS)" },
	{ 0x01D8,	"SCCRd",	"" },
	{ 0x01DC,	"SCCWr",	"" },
	{ 0x01E0,	"IWM",		"" },
	{ 0x0C00,	"SCSIBase",	"" },
	{ 0x0CC0,	"ASCBase",	"" },
	{ 0x0D18,	"PMgrBase",	"(PowerManager)" },
	{ -1,		"",			"" }
};

/*
 *	InitMMU_V2
 *
 *	Initialize MMU variables. Remember to call this routine before
 *	using any other routine in this file!
 *
 */
void
InitMMU_V2(void)
{
	if (!init_mmu_vars())
		Error("Unable to determine MMU mappings\nAborting...\n");

	// Setup RAM mappings
	setup_ram_mappings();
}

/*
 *	LogicalToPhysical_V2
 *
 *	NOTE: init_mmu_vars() *MUST* be called before any usage of this routine!!!
 *
 *	This is the global routine, will always fail if translation not possible
 *   
 */

u_long
LogicalToPhysical_V2(u_long addr, u_long *phys)
{
	return logical_to_physical(addr, phys, 1);
}


/*
 * MMUDeviceMappings
 */
void
ShowMMUDeviceMappings(void)
{
	hw_dev		*hwp;
	long		result;
	u_long		log_addr, phys_addr;

	cprintf("## MMU Device Mappings, Log -> Phys ##\n");

	if (!init_mmu_vars()) {
		cprintf("Unable to determine MMU mappings\nAborting...\n");
		return;
	}

	for(hwp = hw_dev_list; hwp->lmem_addr != -1; hwp++) {
		log_addr = *(unsigned long *)hwp->lmem_addr;
		if (log_addr != -1) {
			result = logical_to_physical(log_addr, &phys_addr, 0);
			if (result)
				cprintf("  %-10s: 0x%08lX -> 0x%08lX %s\n", hwp->name, log_addr, phys_addr, hwp->note);
			else
				cprintf("  %-10s: 0x%08lX -> *** logical_to_physical() failure\n", hwp->name, log_addr);
		} else
			cprintf("  %-10s: 0x%08lX -> 0x%08lX (not present)\n", hwp->name, (long)-1, (long)-1);
	}

	// Video mapping
	if (sMacOS_videoaddr != -1) {
		result = logical_to_physical(sMacOS_videoaddr, &phys_addr, 0);
		if (result) {
			cprintf("  Video     : 0x%08lX -> 0x%08lX\n", sMacOS_videoaddr, phys_addr);
		} else
			cprintf("  Video     : 0x%08lX -> *** logical_to_physical() failure\n", sMacOS_videoaddr);
	}

	cprintf("######################################\n");
}

/*
 *	init_mmu_vars
 *
 *	Initialize local variables
 *
 */
static u_short
init_mmu_vars(void)
{
	GDHandle		hGD;
	long			result;

	// Video address
	hGD = GetMainDevice();
	if (hGD != nil) {
		sMacOS_videoaddr = (long)(**((**hGD).gdPMap)).baseAddr;
	} else
		sMacOS_videoaddr = -1;

	// MMU
	Gestalt(gestaltMMUType, &result);
	switch(result) {
		case gestalt68851:
			sMacOS_mmu = MMU_68851;
			sMacOS_tc = Get020_TC();
			sMacOS_ps = 256 << ((sMacOS_tc >> 20) & 7);
			break;
		case gestalt68030MMU:
			sMacOS_mmu = MMU_68030;
			sMacOS_tc = Get030_TC();
			sMacOS_ps = 256 << ((sMacOS_tc >> 20) & 7);
			break;
		case gestalt68040MMU:
			sMacOS_mmu = MMU_68040;
			sMacOS_tc = Get040_TC();
			sMacOS_ps = (sMacOS_tc & 0x00004000) ? 8192 : 4096;
			break;
		default:
			cprintf("MMU not present or unsupprted hardware!\n");
			return 0;
	}

	// MMU enabled?
	if (sMacOS_mmu == MMU_68040) {
		sMacOS_mmu_enable = (sMacOS_tc & 0x00008000) != 0;
	} else {
		sMacOS_mmu_enable = (sMacOS_tc & 0x80000000) != 0;
	}

	if (!sMacOS_mmu_enable)
		cprintf("*** MMU not enabled, assuming logical=physical\n");

	return 1;
}

/*
 *	logical_to_physical
 *
 *	NOTE: init_mmu_vars() *MUST* be called before any usage of this routine!!!
 *
 *	This is the real work routine, will check if it should fail or not when
 *	translation not possible. Will return 0 (zero) if translation
 *	fails and "silent"/non-fail mode
 *   
 */

static u_long
logical_to_physical(u_long addr, u_long *phys, int fail)
{
	u_long		pte[2], ph, mask;
	u_short		psr;
	int			i, numbits;

	// Return physical == logical if MMU not enabled
	if (!sMacOS_mmu_enable) {
		*phys = addr;
		return 1;
	}

	if (sMacOS_mmu == MMU_68040) {
		ph = Get040_PTE(addr);
		if ((ph & MMU40_RES) == 0) {
			return 0;
		}
		if ((ph & MMU40_TTX) == 0) {
			mask = (sMacOS_tc & 0x4000) ? 0x00001fff : 0x00000fff;
			ph &= (~mask);
		} else {
			mask = 0xFFFFFFFF;
			ph = 0;
		}
	} else {
		if (sMacOS_mmu == MMU_68030) {
			// Check for transparent translation of addr
			if (Test030_TTx(addr)) {
				*phys = addr;
				return 1;
			}
			i = Get030_PTE(addr, pte, &psr);
		} else
			i = Get68851_PTE(addr, pte, &psr);

		switch (i) {
			case -1:		/* General failure */
				if (fail)
					ErrorNumNum("Failure while reading MMU tables (%ld), logical address 0x%08lX\n", pte[0], addr);
				return 0;
			case 0:			/* Short format DTE */
				ph = pte[0] & 0xFFFFFF00;
				break;
			case 1:			/* Long format DTE */
				ph = pte[1] & 0xFFFFFF00;
				break;
			default:
				if (fail)
					Error("LogicalToPhysical_V2(): bad, no MMU table translation");
				return 0;
		}

		/*
		 * We must now figure out how many levels down we went and
		 * mask the bits appropriately -- the returned value may only
		 * be the upper n bits, and we have to take the rest from addr.
		 */
		numbits = (sMacOS_tc >> 16) & 0x0F;
		psr &= 0x0007;		/* Number of levels we went */
		for (i = 0; i < psr; i++)
			numbits += (sMacOS_tc >> (12 - i * 4)) & 0x0F;

		/*
		 * We have to take the most significant "numbits" from
		 * the returned value "ph", and the rest from our addr.
		 * Assume that numbits != 0.
		 */
		mask = (1 << (32 - numbits)) - 1;
	}
	*phys = ph + (addr & mask);

	return 1;
}

/*
 *	setup_ram_mappings()
 *
 *	Find out how MacOS has mapped itself so we can do the same thing.
 *	Returns the address of logical 0 so that locore can map the kernel
 *	properly.
 *
 */
static u_long
setup_ram_mappings(void)
{
	int			i, failed;
	u_long		addr, lastpage, phys, page_size;
	MemChunk	*mcp;

	failed = 0;

	sNumMemChunks = 0;
	mcp = sMemChunks;
	for (i = 0; i < NUM_MEMCHUNKS; i++) {
		mcp->lo = 0;
		mcp->hi = 0;
		mcp->logical = -1;
		mcp->size = -1;
		++mcp;
	}

	lastpage = get_top_of_ram();
	addr = get_rbv_size();
	if (addr) {
		cprintf("RBV boot: %ld KB RAM allocated to video.\n", addr / 1024);
		lastpage -= addr;
	}
	page_size = sMacOS_ps;

	// Probe high -> low RAM addresses until we can get
	// a mapping without errors, this should solve
	// the problems with the Classic II

	addr = lastpage - page_size;
	while( (addr > 0) && (!logical_to_physical(addr, &phys, 0)) )
		addr -= page_size;
	lastpage = addr + page_size;

	mcp = sMemChunks;
	--mcp;
	for (addr = 0; addr < lastpage; addr += page_size) {

		if (logical_to_physical(addr, &phys, 0)) {

			if ( (sNumMemChunks > 0) && (phys == mcp->hi) ) {
				mcp->hi += page_size;
			} else {

				if (sNumMemChunks < (NUM_MEMCHUNKS-1)) {
					++sNumMemChunks;
					++mcp;
					mcp->lo = phys;
					mcp->hi = phys + page_size;
					mcp->logical = addr;
				} else {
					cprintf("*** Too many memory ranges!!! \n");
					failed = 1;
					break;
					
				}

			}

		}

	}

#if 0
	cprintf("  Amount: %ld MB in %ld Linux pages.\n", addr / (1024 * 1024), addr / NBPG);
#endif

	mcp = sMemChunks;
	for (i = 0; i < sNumMemChunks; i++) {

#if 0
		cprintf("    Logical 0x%08lX to 0x%08lX, size 0x%08lX\n",
				mcp->logical,
				mcp->logical + (mcp->hi - mcp->lo),
				mcp->hi - mcp->lo);
		cprintf("      at physical 0x%08lX to 0x%08lX\n", mcp->lo, mcp->hi);
#endif

		recordMemoryMapping(mcp->logical, mcp->lo, mcp->hi - mcp->lo, 0);

		++mcp;

	}

	if (failed)
		Error("*** setup_ram_mappings() failure - too many mappings\n");

	return 0;
}

/*
 *	It should be possible to probe for the top of RAM, but Apple has
 *	memory structured so that in at least some cases, it's possible
 *	for RAM to be aliased across all memory--or for it to appear that
 *	there is more RAM than there really is.
 */
static int
get_top_of_ram()
{
	long	ram_size;

	Gestalt(gestaltPhysicalRAMSize, &ram_size);
	return ram_size;
}

/*
 *	Verify if this is what I call a "rbv boot", i.e. IIci internal video.
 *	Return number of bytes the internal video has claimed for the display
 *	or zero to indicate a non-"rbv boot".
 */
static u_long
get_rbv_size()
{
	u_long		result, phys_addr, prev_phys_addr, addr;

	result = 0;

	if (logical_to_physical(0, &phys_addr, 0)) {
		if (phys_addr != 0) {
			// logical 0 != physical 0 - should be "rbv boot"
			if (logical_to_physical(sMacOS_videoaddr, &phys_addr, 0)) {
				if (phys_addr == 0) {
					// logical video address == physical 0 - must be "rbv boot"
					// calculate length of RAM allocated to video
					addr = sMacOS_videoaddr;
					prev_phys_addr = phys_addr;
					result = sMacOS_ps;
					while(1) {
						addr += sMacOS_ps;
						if (logical_to_physical(addr, &phys_addr, 0) && ((phys_addr - sMacOS_ps) == prev_phys_addr)) {
							result += sMacOS_ps;
							prev_phys_addr = phys_addr;
						} else
							break;
					}
				}
			}
		}
	}

	return result;
}

