/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

import java.util.*;
import java.awt.Point;
import javax.swing.tree.TreeModel;

/**
 * A simple implementation of TreeLayoutModel. This class lays out a
 * tree in the top-down fashion that diagrams of trees are usually
 * seen in. Nodes on the same level have some minimum space between
 * them. There is a constant space between levels.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */

public class DefaultTreeLayoutModel extends AbstractTreeLayoutModel {

	/** Minimum space between adjacent nodes */
	public static final int WIDTH_SPACE_TB = 30;

	/** Space between levels in the tree */
	public static final int HEIGHT_SPACE_TB = 50;

	public static final int WIDTH_SPACE_LR = 80;
	public static final int HEIGHT_SPACE_LR = 50;
	
	private ArrayList nodes;
	private ArrayList edges;

	/**
	 * Creates a new <code>DefaultTreeLayoutModel</code> instance.
	 */
	public DefaultTreeLayoutModel()
	{
		setTreeLayoutNodeClass(DefaultTreeLayoutNode.class);
	}

	/**
	 * Creates a new <code>DefaultTreeLayoutModel</code> instance
	 * using the given TreeModel m as the model to lay out.
	 *
	 * @param m a <code>TreeModel</code> value
	 */
	public DefaultTreeLayoutModel(TreeModel m)
	{
		this();

		setModel(m);
	}

	/**
	 * Does the layout model support the given orientation?
	 *
	 * @param o an <code>int</code> value
	 * @return true if yes, false if no
	 */
	public boolean supportsOrientation(int o)
	{
		switch (o)
		{
			case TreeDisplay.TOP:
			case TreeDisplay.BOTTOM:
			case TreeDisplay.RIGHT:
			case TreeDisplay.LEFT:
				return true;
			default:
				return false;
		}
	}

	/**
	 * Layout the tree.
	 */
	public void layout()
	{
		root = createTree(model);
		
		doLayout();
	}

	/**
	 * Get the node that contains the Point p.
	 *
	 * @param p a <code>Point</code> value
	 * @return a <code>TreeLayoutNode</code> value
	 */
	public TreeLayoutNode getNode(Point p)
	{
		return getNode(p.x, p.y);
	}

	/**
	 * Get the node that contains the point (x, y).
	 *
	 * @param x an <code>int</code> value
	 * @param y an <code>int</code> value
	 * @return a <code>TreeLayoutNode</code> value
	 */
	public TreeLayoutNode getNode(int x, int y)
	{
		Iterator it = nodes.iterator();

		while (it.hasNext())
		{
			TreeLayoutNode n = (TreeLayoutNode) it.next();
			if (n.contains(x, y))
				return n;
		}

		return null;
	}

	/**
	 * Get an Iterator that can be used for iterating over all nodes.
	 *
	 * @return an <code>Iterator</code> value
	 */
	public Iterator getNodes()
	{
		return nodes.iterator();
	}

	/**
	 * Get an Iterator that can be used for iterating over all edges.
	 *
	 * @return an <code>Iterator</code> value
	 */
	public Iterator getEdges()
	{
		return edges.iterator();
	}

	public TreeLayoutNode createNode(Object data)
	{
		TreeLayoutNode root;

		try {
			root = (TreeLayoutNode) getTreeLayoutNodeClass().newInstance();
		} catch (Exception ex) {
			root = new DefaultTreeLayoutNode();
		}

		root.setData(data);

		return root;
	}

	//
	// private methods
	//

	private TreeLayoutNode createTree(TreeModel model)
	{
		return createTree(model, model.getRoot());
	}

	private TreeLayoutNode createTree(TreeModel model, Object parent_obj)
	{
		TreeLayoutNode root, child;

		if (parent_obj == null) {
			return null;
		}

		root = createNode(parent_obj);
		if (root == null) return null;
		root.setTreeLayoutModel(this);

		for (int i = 0 ; i < model.getChildCount(parent_obj); i++) {
			child = createTree(model, model.getChild(parent_obj, i));
			root.addChild(child);
			child.setParent(root);
		}

		return root;
	}

	private void doLayout()
	{
		nodes = new ArrayList(40);
		edges = new ArrayList(40);

		switch (getOrientation())
		{
			case TreeDisplay.TOP:
				doLayout_vertical(true, root, 0, 0, 0);
				break;
			case TreeDisplay.BOTTOM:
				doLayout_vertical(false, root, 0, (root != null) ? root.getBoundingHeight() : 0, 0);
				break;
			case TreeDisplay.LEFT:
				doLayout_horizontal(true, root, 0, 0);
				break;
			case TreeDisplay.RIGHT:
				doLayout_horizontal(false, root, (root != null) ? root.getBoundingWidth() : 0, 0);
				break;
		}
	}

	private void doLayout_vertical(boolean root_at_top, TreeLayoutNode node, int x, int y, int shift)
	{
		if (node == null)
			return;

		int new_y;
		if (root_at_top) {
			new_y = y + node.getHeight() + HEIGHT_SPACE_TB;
		} else {
			y -= node.getHeight();
			new_y = y - HEIGHT_SPACE_TB;
		}
		
		int children = node.getChildCount();

		nodes.add(node);

		// 'shift' is how much the parent is shifted from the left margin
		// of this column of nodes.  This is a space that can potentially
		// be reused if we encounter children that is larger than the
		// parent or when we have more than one child

		if (children == 0)
		{
			node.setX(x);
			node.setY(y);
		}
		else if (children == 1)
		{
			// want to center parent over child
			
			int offset;
			TreeLayoutNode child = node.getChild(0);

			offset = (node.getWidth() - child.getWidth()) / 2;
			
			if (offset >= 0) {
				// Parent is wider, offset the child
				// (we handle the same size case here for convenience)
				doLayout_vertical(root_at_top, child, x + offset, new_y, shift + offset);
				// The parent may move as well if child had to move further,
				// so we use the child's final position, but we
				// must discount child's given offset
				node.setX(child.getX() - offset);
			} else {
				// Child is wider
				offset = -offset;
				if (offset <= shift) {
					// We have enough shift accumulated so we can just 
					// move the child to the left and leave the parent alone
					// (we handle the same size case here for convenience)
					doLayout_vertical(root_at_top,child, x - offset, new_y, shift - offset);
					// The parent may move as well if child had to move further,
					// so we use the child's final position, but we
					// must discount child's given offset
					node.setX(child.getX() + offset);
				} else {
					// We do not have enough shift accumulated, so we will
					// have to offset the parent a bit more
					// The child may stay where we would be
					// Note that all the shift has been consumed
					doLayout_vertical(root_at_top, child, x - shift, new_y, 0);
					// The parent may move more if child had to move further,
					// so we must start with the child's final position
					// and add what we couldn't accomodate on the child's move
					// must discount child's given offset
					node.setX(child.getX() + (offset - shift) + shift);
				}
			}

			node.setY(y);

			edges.add(new DefaultTreeLayoutEdge(node, child, node.getWidth() / 2, this));
		}
		else
		{
			TreeLayoutNode lchild = node.getChild(0);
			TreeLayoutNode rchild = node.getChild(children - 1);

			// for determining offset from parent's left side
			int nc = children + 1;
			int p = node.getWidth() / nc;
			int off = p;

			int l = x;
			// We may throw away some of the slack we have (shift) here because there
			// is no way of making use of it with just one pass	as we don't know if
			// we have enough nodes to make use of all we had accumulated.
			// But we can be sure we have at least two nodes, so that much is safe
			// to assume we can use.  Hopefully, this will be enough for 99.99999% of the cases
			l -= Math.min(shift, (lchild.getWidth() + rchild.getWidth() + WIDTH_SPACE_TB));
				
			for (int i = 0; i < children; i++)
			{
				edges.add(new DefaultTreeLayoutEdge(node, node.getChild(i), off, this));
				off += p;
				doLayout_vertical(root_at_top, node.getChild(i), l, new_y, 0);
				l += node.getChild(i).getBoundingWidth() + WIDTH_SPACE_TB;
			}

			node.setX(((rchild.getX() + rchild.getWidth() - lchild.getX() - node.getWidth()) / 2) + lchild.getX());
			node.setY(y);
		}
	}

	private void doLayout_horizontal(boolean root_at_left, TreeLayoutNode node, int x, int y)
	{
		if (node == null)
			return;

		int new_x;

		if (root_at_left) {
			new_x = x + node.getWidth() + WIDTH_SPACE_LR;
		} else {
			x -= node.getWidth();
			new_x = x - WIDTH_SPACE_LR;
		}
		
		int children = node.getChildCount();

		nodes.add(node);

		if (children == 0)
		{
			node.setX(x);
			node.setY(y);
		}
		else if (children == 1)
		{
			// want to center parent over child
			
			int offset;
			TreeLayoutNode child = node.getChild(0);

			offset = (node.getHeight() - child.getHeight()) / 2;
			doLayout_horizontal(root_at_left, child, new_x, y);

			node.setX(x);
			node.setY(child.getY() - offset);

			edges.add(new DefaultTreeLayoutEdge(node, child, node.getHeight() / 2, this));
		}
		else
		{
			TreeLayoutNode lchild = node.getChild(0);
			TreeLayoutNode rchild = node.getChild(children - 1);
			int l = y;

			// for determining offset from parent's left side
			int nc = children + 1;
			int p = node.getHeight() / nc;
			int off = p;
			
			for (int i = 0; i < children; i++)
			{
				edges.add(new DefaultTreeLayoutEdge(node, node.getChild(i), off, this));
				off += p;
				doLayout_horizontal(root_at_left, node.getChild(i), new_x, l);
				l += node.getChild(i).getBoundingHeight() + HEIGHT_SPACE_LR;
			}

			node.setX(x);
			node.setY(((rchild.getY() + rchild.getHeight() - lchild.getY() - node.getHeight()) / 2) + lchild.getY());
		}
	}

}// DefaultTreeLayoutModel
