;;; this is a translation to Snd (from CLM's prc-toolkit95.lisp)
;;;  of Perry Cook's Physical Modelling Toolkit.
;;;  substantially cleaned up 29-Mar-02 (6 years later...)

(use-modules (ice-9 optargs))
(use-modules (ice-9 format))
(define pi 3.141592653589793)


(define* (make-reed #:key (offset 0.6) (slope -0.8))
  (let ((roffset offset)
	(rslope slope))
    (lambda (sample)
      (min 1.0 (+ roffset (* rslope sample))))))

(define (reedtable r sample) (r sample))


(define* (make-bowtable #:key (offset 0.0) (slope 1.0))
  (let ((boffset offset)
	(bslope slope))
    (lambda (sample)
      (max 0.0 (- 1.0 (abs (* bslope (+ sample boffset))))))))

(define (bowtable b sample) (b sample))


(define (jettable sample) 
  (max -1.0 (min 1.0 (* sample (- (* sample sample) 1.0)))))


(define* (make-onezero #:key (gain 0.5) (zerocoeff 1.0) (input 0.0))
  (make-one-zero gain (* gain zerocoeff)))


(define* (make-onep #:key (polecoeff 0.9))
  (make-one-pole (- 1.0 polecoeff) (- polecoeff)))

(define (set-pole p val) 
  (set! (mus-b1 p) (- val))
  (set! (mus-a0 p) (- 1.0 val)))

(define (set-gain p val)
  (set! (mus-a0 p) (* (mus-a0 p) val)))


(define (lip-set-freq b freq)
  (set! (mus-b1 b) (* -2.0 0.999 (cos (/ (* pi 2 freq) (srate)))))
  (set! (mus-b2 b) (* 0.999 0.999))
  (set! (mus-a0 b) 0.02))

(define (lip b mouthsample boresample)
  (let ((temp (formant b (- mouthsample boresample))))
    (set! temp (min 1.0 (* temp temp)))
    (+ (* temp mouthsample) (* (- 1.0 temp) boresample))))


(define (make-dc-block)
  (let ((input 0.0)
	(output 0.0))
    (lambda (sample)
      (set! output (+ sample (- (* 0.99 output) input)))
      (set! input sample)
      output)))

(define (dc-block b sample) (b sample))


(define (make-delaya len lag) 
  (let ((outpoint 0)
	(lastin 0.0)
	(input (make-delay len))
	(output 0.0)
	(alpha 0.0)
	(coeff 0.0))
    (let ((outpointer (- 2.0 lag)))
      (if (< outpointer 0.0)
	  (do ()
	      ((>= outpointer 0.0))
	    (set! outpointer (+ outpointer len))))
      (set! outpoint (inexact->exact (floor outpointer)))
      (set! alpha (- outpointer outpoint))
      (set! coeff (/ (- 1.0 alpha) (+ 1.0 alpha)))
      (set! outpoint (- outpoint)))
    (lambda (sample)
      (delay input sample)
      (let ((temp (tap input outpoint)))
	(set! output (+ (* (- coeff) output) lastin (* temp coeff)))
	(set! lastin temp)
	output))))

(define (delaya d sample) (d sample))


(define (make-delayl len lag)
  (let ((outpoint 0)
	(input (make-delay len))
	(output 0.0)
	(omalpha 0.0)
	(alpha 0.0))
    (let ((outpointer (- 1 lag)))
      (if (< outpointer 0.0)
	  (do ()
	      ((>= outpointer 0.0))
	    (set! outpointer (+ outpointer len))))
      (set! outpoint (inexact->exact (floor outpointer)))
      (set! alpha (- outpointer outpoint))
      (set! omalpha (- 1.0 alpha))
      (set! outpoint (- outpoint)))
    (lambda (sample)
      (delay input sample)
      (set! output (+ (* (tap input (1- outpoint)) omalpha)
		      (* (tap input outpoint) alpha)))
      output)))

(define (delayl d sample) (d sample))


;;; now some example instruments

(define (plucky beg dur freq amplitude maxa)
  (let* ((lowestfreq 100.0)
	 (len (1+ (inexact->exact (floor (/ (srate) lowestfreq)))))
	 (delayline (make-delaya len (- (/ (srate) freq) 0.5)))
	 (filter (make-onezero))
	 (st (inexact->exact (floor (* (srate) beg))))
	 (durlen (inexact->exact (floor (* (srate) dur))))
	 (out-data (make-vct durlen))
	 (dout 0.0))
    (do ((i 0 (1+ i)))
	((= i len))
      (set! dout (delaya delayline (+ (* 0.99 dout)
				      (* maxa (- 1.0 (random 2.0)))))))
    (vct-map! out-data
	      (lambda ()
		(set! dout (delaya delayline (one-zero filter dout)))
		(* amplitude dout)))
    (mix-vct out-data st #f 0 #f)))


;;; freq is off in this one (in prc's original also)
(define (bow beg dur frq amplitude maxa)
  (let* ((lowestfreq 100.0)
	 (len (1+ (inexact->exact (floor (/ (srate) lowestfreq)))))
	 (ratio 0.8317)
	 (temp (- (/ (srate) frq) 4.0))
	 (neckdelay (make-delayl len (* temp ratio)))
	 (bridgedelay (make-delayl (inexact->exact (floor (/ len 2))) (* temp (- 1.0 ratio))))
	 (bowtab (make-bowtable :slope 3.0))
	 (filt (make-onep))
	 (rate .001)
	 (bowing #t)
	 (bowvelocity rate)
	 (maxvelocity maxa)
	 (attackrate rate)
	 (st (inexact->exact (floor (* (srate) beg))))
	 (durlen (inexact->exact (floor (* (srate) dur))))
	 (out-data (make-vct durlen))
	 (ctr 0)
	 (release (inexact->exact (floor (* .8 durlen))))
	 (bridgeout 0.0)
	 (neckout 0.0))
    (set-pole filt 0.6)
    (set-gain filt 0.3)
    (vct-map! out-data
	      (lambda ()
		(let* ((bridgerefl 0.0)
		       (nutrefl 0.0) 
		       (veldiff 0.0) 
		       (stringvel 0.0) 
		       (bowtemp 0.0))
		  (if bowing
		      (if (not (= maxvelocity bowvelocity))
			  (if (< bowvelocity maxvelocity)
			      (set! bowvelocity (+ bowvelocity attackrate))
			      (set! bowvelocity (- bowvelocity attackrate))))
		      (if (> bowvelocity 0.0) 
			  (set! bowvelocity (- bowvelocity attackrate))))
		  (set! bowtemp (* 0.3 bowvelocity))
		  (let ((filt-output (one-pole filt bridgeout)))
		    (set! bridgerefl (- filt-output))
		    (set! nutrefl (- neckout))
		    (set! stringvel (+ bridgerefl nutrefl))
		    (set! veldiff (- bowtemp stringvel))
		    (set! veldiff (* veldiff (bowtable bowtab veldiff)))
		    (set! neckout (delayl neckdelay (+ bridgerefl veldiff)))
		    (set! bridgeout (delayl bridgedelay (+ nutrefl veldiff)))
		    (let ((result (* amplitude 10.0 filt-output)))
		      (if (= ctr release)
			  (begin
			    (set! bowing #f)
			    (set! attackrate .0005)))
		      (set! ctr (+ ctr 1))
		      result)))))
    (mix-vct out-data st #f 0 #f)))


(define (brass beg dur freq amplitude maxa)
  (let* ((lowestfreq 100.0)
	 (len (1+ (inexact->exact (floor (/ (srate) lowestfreq)))))
	 (delayline (make-delaya len (+ 1.0 (/ (srate) freq))))
	 (lipfilter (make-formant))
	 (dcblocker (make-dc-block))
	 (blowing #t)
	 (rate .001)
	 (breathpressure 0.0)  ; 0.1 ?
	 (maxpressure maxa)
	 (attackrate rate)
	 (st (inexact->exact (floor (* (srate) beg))))
	 (durlen (inexact->exact (floor (* (srate) dur))))
	 (out-data (make-vct durlen))
	 (release (inexact->exact (floor (* .8 durlen))))
	 (ctr 0)
	 (dout 0.0))
    (lip-set-freq lipfilter freq)
    (vct-map! out-data
	      (lambda ()
		(if blowing
		    (if (not (= maxpressure breathpressure))
			(if (< breathpressure maxpressure)
			    (set! breathpressure (+ breathpressure attackrate))
			    (set! breathpressure (- breathpressure attackrate))))
		    (if (> breathpressure 0.0)
			(set! breathpressure (- breathpressure attackrate))))
		(set! dout (delaya delayline (dc-block dcblocker
						       (lip lipfilter
							    (* 0.3 breathpressure)
							    (* 0.9 dout)))))
		(let ((result (* amplitude dout)))
		  (if (= ctr release) 
		      (begin
			(set! blowing #f)
			(set! attackrate .0005)))
		  (set! ctr (+ ctr 1))
		  result)))
    (mix-vct out-data st #f 0 #f)))


(define (clarinet beg dur freq amplitude maxa)
  (let* ((lowestfreq 100.0)
	 (len (1+ (inexact->exact (floor (/ (srate) lowestfreq)))))
	 (delayline (make-delayl len (- (* 0.5 (/ (srate) freq)) 1.0)))
	 (rtable (make-reed :offset 0.7 :slope -0.3))
	 (filter (make-onezero))
	 (blowing #t)
	 (breathpressure 0.0) ; 0.1 ?
	 (rate .001)
	 (maxpressure maxa)
	 (attackrate rate)
	 (st (inexact->exact (floor (* (srate) beg))))
	 (durlen (inexact->exact (floor (* (srate) dur))))
	 (out-data (make-vct durlen))
	 (ctr 0)
	 (release (inexact->exact (floor (* .8 durlen))))
	 (dlyout 0.0))
    (vct-map! out-data
	      (lambda ()
		(let ((pressurediff 0.0))
		  (if blowing
		      (if (not (= maxpressure breathpressure))
			  (if (< breathpressure maxpressure)
			      (set! breathpressure (+ breathpressure attackrate))
			      (set! breathpressure (- breathpressure attackrate))))
		      (if (> breathpressure 0.0)
			  (set! breathpressure (- breathpressure attackrate))))
		  (set! pressurediff (- (one-zero filter (* -0.95 dlyout)) breathpressure))
		  (set! dlyout (delayl delayline 
				       (+ breathpressure 
					  (* pressurediff 
					     (reedtable rtable pressurediff)))))
		  (let ((result (* amplitude dlyout)))
		    (if (= ctr release)
			(begin
			  (set! blowing #f)
			  (set! attackrate .0005)))
		    (set! ctr (+ ctr 1))
		    result))))
    (mix-vct out-data st #f 0 #f)))


(define (flute beg dur freq amplitude maxa)
  (let* ((lowestfreq 100.0)
	 (len (1+ (inexact->exact (floor (/ (srate) lowestfreq)))))
	 (ratio 0.8)
	 (temp (- (/ (srate) freq) 5.0))
	 (jetdelay (make-delayl (inexact->exact (floor (/ len 2))) (* temp (- 1.0 ratio))))
	 (boredelay (make-delayl len (* ratio temp)))
	 (filter (make-onep))
	 (dcblocker (make-dc-block))
	 (jetrefl 0.6)
	 (endrefl 0.6)
	 (sinphase 0.0)
	 (blowing #t)
	 (rate .0005)
	 (breathpressure 0.0) ; 0.1 ?
	 (maxpressure maxa)
	 (attackrate rate)
	 (st (inexact->exact (floor (* (srate) beg))))
	 (durlen (inexact->exact (floor (* (srate) dur))))
	 (out-data (make-vct durlen))
	 (ctr 0)
	 (release (inexact->exact (floor (* .8 durlen))))
	 (boreout 0.0))
    (set-pole filter 0.8)
    (set-gain filter -1.0)
    (vct-map! out-data
	      (lambda ()
		(let ((randpressure (* 0.1 breathpressure (random 1.0)))
		      (temp 0.0) 
		      (pressurediff 0.0))
		  (set! sinphase (+ sinphase 0.0007))		;5 hz vibrato?
		  (if (> sinphase 6.28) (set! sinphase (- sinphase 6.28)))
		  (set! randpressure (+ randpressure (* 0.05 breathpressure (sin sinphase))))
		  (if blowing
		      (if (not (= maxpressure breathpressure))
			  (if (< breathpressure maxpressure)
			      (set! breathpressure (+ breathpressure attackrate))
			      (set! breathpressure (- breathpressure attackrate))))
		      (if (> breathpressure 0.0) 
			  (set! breathpressure (- breathpressure attackrate))))
		  (set! temp (dc-block dcblocker (one-pole filter boreout)))
		  (set! pressurediff (+ (jettable 
					 (delayl jetdelay 
						 (+ breathpressure 
						    (- randpressure (* jetrefl temp))))) 
					(* endrefl temp)))
		  (set! boreout (delayl boredelay pressurediff))
		  (let ((result (* 0.3 amplitude boreout)))
		    (if (= ctr release)
			(begin
			  (set! blowing #f)
			  (set! attackrate .0005)))
		    (set! ctr (+ ctr 1))
		    result))))
    (mix-vct out-data st #f 0 #f)))

(define (test-prc95)
  (plucky 0 .3 440 .2 1.0)
  (bow .5 .3 220 .2 1.0)
  (brass 1 .3 440 .2 1.0)
  (clarinet 1.5 .3 440 .2 1.0)
  (flute 2 .3 440 .2 1.0))

