/*
 * snes9express
 * rom.cpp
 * Licensed under the "Artistic" license
 * To view the details of the Artistic license, see the file
 * ./ARTISTIC, or go to http://www.opensource.org/artistic-license.html
 */

#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>
#include "rom.h"
#include "interface.h"

#ifdef S9X_INCLUDE_ZLIB
# include <zlib.h>
#endif

#include "pix/rom.xpm"
#include "pix/bnr.xpm"

extern s9x_Interface *Interface;

static char*ROMcharacters =
  "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
  "0123456789 !#$&*()_+-/,.?:"
  "abcdefghijklmnopqrstuvwxyz";

s9x_ROM::s9x_ROM(fr_Element*parent):
s9x_Notepage(parent, "ROM"),
RomBtn(this, new fr_Label(this, "ROMs", rom_xpm)),
File(this, Name, S9X_ROMENV, S9X_ROMDIR, false, rom_xpm),
MemoryMap(this, "Memory Map"),
Format(this, "Format"),
Bnr(this, PROG, bnr_xpm),
Selector(this)
{
   fr_MenuItem*MI;

   SetGridSize(3, 2, true);
   SetPadding(6, 3);
   Label = new fr_Label(this, Name, rom_xpm);

   RomBtn.SetTooltip("ROM Selector");
   RomBtn.AddListener(this);
   SetStretch(Grow, Fill);
   Pack(RomBtn);
   SetStretch(Fill, Fill);

   Pack(File, 5, 1);
        
   MI = new fr_MenuItem(&MemoryMap, "auto");
   MI = new fr_MenuItem(&MemoryMap, "Force Lo-ROM");
   MI->Args << fr_CaseInsensitive << "-fl" << "-lorom" << "-lr";
   MI->SetTooltip("Force Lo-ROM memory map");

   MI = new fr_MenuItem(&MemoryMap, "Force Hi-ROM");
   MI->Args << fr_CaseInsensitive << "-fh" << "-hirom" << "-hr";
   MI->SetTooltip("Force Hi-ROM memory map");

   Pack(MemoryMap, 2, 1, 4, 2);

   MI = new fr_MenuItem(&Format, "auto");
   MI = new fr_MenuItem(&Format, "Interleaved");
   MI->Args << fr_CaseInsensitive << "-i" << "-interleaved";
   MI->SetTooltip("Interleaved format");

   MI = new fr_MenuItem(&Format, "Interleaved 2");
   MI->Args << fr_CaseInsensitive << "-i2" << "-interleaved2";
   MI->SetTooltip("Interleaved 2 format.  This format can't be auto-deteced.  Many Super-FX games use this format.");

   Pack(Format, 4, 1, 6, 2);   
   Pack(Bnr, 0, 1, 2, 2);

   Selector.AddListener(this);
   AddListener(this);
	
   Frame();
}

void s9x_ROM::SetToDefaults() {
   MemoryMap.SetToDefault();
   Format.SetToDefault();
   File.SetToDefault();
}

void s9x_ROM::Set9xVersion(float version) {
}

void s9x_ROM::SiftArgs(fr_ArgList& L) {
   int a, p;
   char*A;
   
   MemoryMap.SiftArgs(L);
   Format.SiftArgs(L);
   
   p = L.CountArgs();
   for(a=0; a<p; a++) {
   	A = L[a];
      if(L.IsMarked(a))
	continue;
      if(fr_Exists(A)) {
         File.SetFileName(A);
         L.Mark(a);
      };
   };
}

void s9x_ROM::SetFileName(char*file) {
   File.SetFileName(file);
}

char*s9x_ROM::GetFileName() {
   return File.GetFileName();
}

void s9x_ROM::FilePopup() {
   File.FilePopup();
}

void s9x_ROM::CompileArgs(fr_ArgList& L) {
   MemoryMap.CompileArgs(L);
   Format.CompileArgs(L);
   File.CompileArgs(L);
}

void s9x_ROM::EventOccurred(fr_Event*e) {
   if(e->Is(RomBtn, fr_Click)) {
      Selector.SetVisibility(true);
   } else if(e->Is(Selector, fr_Destroy)) {
      Selector.SetVisibility(false);
   } else if(e->Is(this, fr_MenuClick)) {
      FilePopup();
   }
}

/* ############################# s9x_ROMdata ############################## */

s9x_ROMdata::s9x_ROMdata() {
   is_a_rom = false;
   Name[0] = 0;
   FileSize = HiROM = 0;
   FileName = 0;
}

s9x_ROMdata::~s9x_ROMdata() {
   if(FileName)
     delete[] FileName;
}

int s9x_ROMdata::LoadROM(char*dir, char*file) { 
   bool valid;
   int i;
   long BaseOffset;
#ifdef ZLIB_VERSION
# define romopen(f) (gzFile*)gzopen(f, "rb")
# define romseek gzseek
# define romread gzread
# define romclose gzclose
   gzFile *fptr;
#else
# define romopen(f) fopen(f, "rb")
# define romseek fseek
# define romread(f, b, s) fread(b, 1, s, f)
# define romclose fclose
   FILE *fptr;
#endif

   if((!dir)||(!file)||(file[0]=='.'))
     return -1;

   if(SetFile(dir, file)!=0)
     return -2;
   is_a_rom = false;
   
   fptr = romopen(FileName);
   if(!fptr)
     return -1;
	
   for(BaseOffset = 0x8100; BaseOffset <= 0x10100; BaseOffset += 0x8000) {
      if(romseek(fptr, BaseOffset + 0xC0, SEEK_SET)<0) {
	 romclose(fptr);
	 return -1;
      }
      if(romread(fptr, Name, sizeof(Name))!=sizeof(Name)) {
	 romclose(fptr);
	 return -1;
      };
      valid = false;
      if(
	 ((Name[21] >= '0' )&&(Name[21] <='9'))
	 ||((Name[23] >= 0x0A)&&(Name[23] <= 0x0F))
	 ||((Name[25] >= 0x01)&&(Name[25] <= 0x09))
	 ) {
	 valid = true;
	 for(i=0; i<21; i++)
	   if(!strchr(ROMcharacters, Name[i])) {
	      valid = false;
	      break;
	   };
      }
      if(BaseOffset==0x8100)
	HiROM = 0;
      else
	HiROM = 1;
      if(valid) break;
   };
   romclose(fptr);
   if(!valid)
     return -1;
   Name[21] = 0;
   g_strstrip(Name);

   is_a_rom = true;
   return 0;

# undef romopen
# undef romseek
# undef romread
# undef romclose
}

bool s9x_ROMdata::IsAROM() {
   return is_a_rom;
}

char*s9x_ROMdata::GetName() {
   return Name;
}

char*s9x_ROMdata::GetFile() {
   return FileName;
}

long s9x_ROMdata::GetSize() {
   return FileSize;
}

int s9x_ROMdata::SetFile(char*dir, char*file) {
   char *dot;
   struct stat statbuf;
	
   if((!dir)||(!file))
     return -1;
   if(FileName)
     delete[] FileName;
   try {
      FileName = new char[strlen(dir) + strlen(file) + 3];
   } catch(...) {
      FileName = (char*)0;
      return -2;
   };
   strcpy(FileName, dir);
   if(dir[strlen(dir)-1]!='/') strcat(FileName, "/");
   strcat(FileName, file);
   strncpy(Name, file, sizeof(Name));
   Name[sizeof(Name)-1] = 0;
   if((dot = strrchr(Name, '.')))
     dot[0] = 0;
   
   if(stat(FileName, &statbuf)!=0)
     return -1;
   FileSize = statbuf.st_size;

   is_a_rom = true;
   return 0;
}

/* ############################# s9x_ROMselector ########################## */

s9x_ROMselector::s9x_ROMselector(fr_Element*parent):
fr_Window(parent, "ROM Selector"),
RomIcon(this, "ROM", rom_xpm),
ROMlist(this, 1),
InfoBox(this, "ROM Information"),
LblFile(this, "File:"),
LblSize(this, "Size:"),
InfFile(this, ""),
InfSize(this, ""),
BtnOK(this, "Ok"),
BtnCancel(this, "Cancel"),
BtnBox(this)
{
   SetIcon("ROMs", rom_xpm);
   SetGridSize(1, 3, false);
   SetPadding(3, 3);
   Pad();
	
   Matched = -1;
   CountFiles = 0;
   ROMlist.AddListener(this);
   ROMlist.SetSize(96, 256);
   ROMlist.SetSortColumn(0);
   ROMlist.SetSortOnInsert(true);
   Pack(ROMlist);

   InfoBox.SetGridSize(2, 2, false);
   InfoBox.SetStretch(Shrink, Shrink);
   InfoBox.Pack(LblFile);
   InfoBox.Pack(InfFile);
   InfoBox.Pack(LblSize);
   InfoBox.Pack(InfSize);
   InfoBox.Frame();
   Pack(InfoBox);
	
   BtnOK.AddListener(this);
   BtnCancel.AddListener(this);
   BtnBox.AddButton(BtnOK, true);
   BtnBox.AddButton(BtnCancel);
   Pack(BtnBox);
}

void s9x_ROMselector::LoadDirectory(char *dir) {
   int dlen;
   char Mesg[128];
   struct dirent *DirEntry;
   DIR *d;
	
   if(!dir)
     return;
   if(dir[0] == 0)
     dir = ".";
   dlen = strlen(dir);
   if(!fr_DirExists(dir)) {
      sprintf(Mesg, "The directory \"%s\" does not exist.\n"
	      "You can change the ROM directory in the Preferences\n"
	      "window.",
	      dlen<80?dir:"[directory name too long]");
      fr_Mesg(Mesg);
      return;
   };
   d = opendir(dir);
   if(!d) {
      sprintf(Mesg, "Error reading directory \"%s\".\n"
	      "Perhaps you do not have read permission.",
	      dlen<80?dir:"[directory name too long]");
      fr_Mesg(Mesg);
      return;
   };
   for(DirEntry = readdir(d);  DirEntry;  DirEntry = readdir(d))
     AddFile(dir, DirEntry->d_name);
   closedir(d);
}

/**
 * If the file is a SNES ROM, add it to the ROMlist
 * @return 0 on success, non-zero otherwise
 */
int s9x_ROMselector::AddFile(char*dir, char*file) {
   int r;
   char *filename;
   char *Data[] = {
      "[emtpy]", ""
   };
   s9x_ROMdata *RD;
	
   RD = new s9x_ROMdata();
   if((strstr(file, ".zip"))||(strstr(file, ".ZIP"))||(strstr(file, ".Zip")))
     RD->SetFile(dir, file);
   else
     RD->LoadROM(dir, file);
	
   if(RD->IsAROM()) {
      Data[0] = RD->GetName();
      filename = RD->GetFile();
      r = ROMlist.AddRow(Data);
      ROMlist.AssociateData(r, RD);
      if(filename && (strcmp(filename, InterfaceFile)==0)) {
	 Matched = r;
	 ROMlist.Select(Matched);
	 RowSelected(Matched);
      }
      CountFiles++;	
      return 0;
   };
	
   return -1;
}

char*s9x_ROMselector::GetSelectedFileName() {
   int r;
   s9x_ROMdata *RD;
	
   if((r=ROMlist.GetSelected()) < 0)
     return (char*)0;
   
   RD = (s9x_ROMdata*)ROMlist.AssociatedData(r);
   if(!RD)
     return (char*)0;
	
   return RD->GetFile();
}

void s9x_ROMselector::EventOccurred(fr_Event*e) {
   if(e->Is(BtnOK)||e->Is(ROMlist, fr_DoubleClick)) {
      ((s9x_ROM*)Parent)->SetFileName(GetSelectedFileName());
      SetVisibility(false);
   } else if(e->Is(ROMlist, fr_Select))
	RowSelected(e->intArg);
   else if(e->Is(BtnCancel))
     SetVisibility(false);
}

void s9x_ROMselector::RowSelected(int row) {
   long fs;
   static char FileSize[32];
   s9x_ROMdata *RD;
	
   RD = (s9x_ROMdata*)ROMlist.AssociatedData(row);
   if(!RD) {
      InfFile.SetLabel("-");
      InfSize.SetLabel("-");
   } else {
      fs = RD->GetSize();
      if(fs < 1<<10)
	sprintf(FileSize, "%ld bytes", fs);
      else if(fs < 1<<19)
	sprintf(FileSize, "%ld kB", fs/(1<<10));
      else
	sprintf(FileSize, "%.2f MB", fs/(float)(1<<20));
      InfFile.SetLabel(RD->GetFile());
      InfSize.SetLabel(FileSize);
   };
}

void s9x_ROMselector::SetVisibility(bool v) {
   fr_Window::SetVisibility(v);
   Parent->SetEditable(!v);
   fr_Flush();
   if(v) {
      RefreshContent();
   } else {
      DeleteContent();
   }
}

void s9x_ROMselector::RefreshContent() {
   if(CountFiles>0)
     DeleteContent();
   
   InterfaceFile = Interface->ROM.GetFileName();
   LoadDirectory(Interface->Prefs.GetROMdir());

   if(CountFiles<1) {
      BtnOK.SetEditable(false);
      RowSelected(0);
      fr_Mesg("No ROMs were found.\nMake sure you set your ROM"
	      " directory in the Preferences window.");
      
   } else if(Matched<0)
	ROMlist.Select(0);
   BtnOK.SetEditable(CountFiles>0);
}

void s9x_ROMselector::DeleteContent() {
   s9x_ROMdata *RD;

   Parent->SetEditable(true);
   if(!CountFiles)
     return;
   for(int r=0; r<CountFiles; r++) {
      RD = (s9x_ROMdata*)ROMlist.AssociatedData(r);
      delete RD;
   };
   ROMlist.RemoveAll();
   CountFiles = 0;
   Matched = -1;
}
