package org.apache.turbine.services.upload;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and 
 *    "Apache Turbine" must not be used to endorse or promote products 
 *    derived from this software without prior written permission. For 
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without 
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// Java Core Classes
import java.io.*;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;

// Turbine stuff.
import org.apache.turbine.util.*;
import org.apache.turbine.util.upload.*;
import org.apache.turbine.services.*;
import org.apache.turbine.services.resources.*;

/**
 * <p> This class is an implementation of {@link UploadService}.
 *
 * <p> Files will be stored in temporary disk storage on in memory,
 * depending on request size, and will be available from the {@link
 * org.apache.turbine.util.ParameterParser} as {@link
 * org.apache.turbine.util.upload.FileItem}s.
 *
 * <p>This implementation of {@link UploadService} handles multiple
 * files per single html widget, sent using multipar/mixed encoding
 * type, as specified by RFC 1867.  Use {@link
 * org.apache.turbine.util.ParameterParser#getFileItems(String)} to
 * acquire an array of {@link
 * org.apache.turbine.util.upload.FileItem}s associated with given
 * html widget.
 *
 * @author <a href="mailto:Rafal.Krzewski@e-point.pl">Rafal Krzewski</a>
 * @version $Id: TurbineUploadService.java,v 1.13 2001/04/11 00:45:53 jon Exp $
 */
public class TurbineUploadService
    extends TurbineBaseService
    implements UploadService
{
    /**
     * A maximum lenght of a single header line that will be
     * parsed. (1024 bytes).
     */
    public static final int MAX_HEADER_SIZE = 1024;

    /**
     * Initializes the service.
     *
     * This method processes the repository path, to make it relative to the web
     * application root, if neccessary
     *
     * @param config the ServletConfig of the Turbine servlet
     */
    public void init(ServletConfig config)
    {
        String path = getProperties()
            .getProperty(UploadService.REPOSITORY_KEY,
                         UploadService.REPOSITORY_DEFAULT.toString());
        if(!path.startsWith("/")) 
        {
            String realPath = config.getServletContext().getRealPath(path);
            if(realPath != null)
            {
                path = realPath;
            }
        }
        getProperties().setProperty(UploadService.REPOSITORY_KEY, path);
    }

    /**
     * <p> Processes an <a href="http://rf.cx/rfc1867.html">RFC
     * 1867</a> compliant <code>multipart/form-data</code> stream.
     *
     * @param req The servlet request to be parsed.
     * @param params The ParameterParser instance to insert form
     * fields into.
     * @param path The location where the files should be stored.
     * @exception TurbineException If there are problems reading/parsing
     * the request or storing files.
     */
    public void parseRequest( HttpServletRequest req,
                              ParameterParser params,
                              String path )
        throws TurbineException
    {
        String contentType = req.getHeader("Content-type");
        if(!contentType.startsWith("multipart/form-data"))
        {
            throw new TurbineException("the request doesn't contain " + 
                "multipart/form-data stream");
        }
        int requestSize = req.getContentLength();
        if(requestSize == -1)
        {
            throw new TurbineException("the request was rejected because " + 
                "it's size is unknown");
        }
        if(requestSize > TurbineUpload.getSizeMax())
        {
            throw new TurbineException("the request was rejected because " + 
                "it's size exceeds allowed range");
        }

        try 
        {
            byte[] boundary = contentType.substring(
                                contentType.indexOf("boundary=")+9).getBytes();
            InputStream input = (InputStream)req.getInputStream();
            
            MultipartStream multi = new MultipartStream(input, boundary);
            boolean nextPart = multi.skipPreamble();
            while(nextPart)
            {
                parseHeaders(multi.readHeaders());
                String fieldName = getFieldName();
                if (fieldName != null)
                {
                    String subContentType = getHeader("Content-type");
                    if (subContentType != null && subContentType
                                                .startsWith("multipart/mixed"))
                    {
                        // Multiple files.
                        byte[] subBoundary = 
                            subContentType.substring(
                                subContentType
                                .indexOf("boundary=")+9).getBytes();
                        multi.setBoundary(subBoundary);
                        boolean nextSubPart = multi.skipPreamble();
                        while (nextSubPart)
                        {
                            parseHeaders(multi.readHeaders());
                            if (getFileName() != null)
                            {
                                FileItem item = createItem(path, requestSize);
                                OutputStream os = item.getOutputStream();
                                try
                                {
                                    multi.readBodyData(os);
                                }
                                finally
                                {
                                    os.close();
                                }
                                params.append(getFieldName(), item);
                            }
                            else
                            {
                                // Ignore anything but files inside
                                // multipart/mixed.
                                multi.discardBodyData();
                            }
                            nextSubPart = multi.readBoundary();
                        }
                        multi.setBoundary(boundary);
                    }
                    else
                    {
                        if (getFileName() != null)
                        {
                            // A single file.
                            FileItem item = createItem(path, requestSize);
                            OutputStream os = item.getOutputStream();
                            try
                            {
                                multi.readBodyData(os);
                            }
                            finally
                            {
                                os.close();
                            }
                            params.append(getFieldName(), item);
                        }
                        else
                        {
                            // A form field.
                            FileItem item = createItem(path, requestSize);
                            OutputStream os = item.getOutputStream();
                            try
                            {
                                multi.readBodyData(os);
                            }
                            finally
                            {
                                os.close();
                            }
                            params.append(getFieldName(), 
                                new String(item.get()));
                        }
                    }
                }
                else
                {
                    // Skip this part.
                    multi.discardBodyData();
                }
                nextPart = multi.readBoundary();
            }
        }
        catch(IOException e)
        {
            throw new TurbineException("Processing of multipart/form-data " + 
                "request failed", e);
        }
    }

    /**
     * <p> Retrieves field name from 'Content-disposition' header.
     *
     * @return A String with the field name for the current
     * <code>encapsulation</code>.
     */
    protected String getFieldName()
    {
        String cd = getHeader("Content-disposition");
        if(cd == null || !cd.startsWith("form-data"))
            return null;
        int start = cd.indexOf("name=\"");
        int end = cd.indexOf('"', start+6);
        if(start == -1 || end == -1)
            return null;
        return cd.substring(start+6,end);
    }

    /**
     * <p> Retrieves file name from 'Content-disposition' header.
     *
     * @return A String with the file name for the current
     * <code>encapsulation</code>.
     */
    protected String getFileName()
    {
        String cd = getHeader("Content-disposition");
        if(!cd.startsWith("form-data") && !cd.startsWith("attachment"))
            return null;
        int start = cd.indexOf("filename=\"");
        int end = cd.indexOf('"', start+10);
        if(start == -1 || end == -1 || (start + 10) == end)
            return null;
        String str = cd.substring(start+10,end).trim();
        if( str.length() == 0) 
            return null;
        else
            return str;
    }

    /**
     * <p> Creates a new instance of {@link
     * org.apache.turbine.util.upload.FileItem}.
     *
     * @param path The path for the FileItem.
     * @param requestSize The size of the request.
     * @return A newly created <code>FileItem</code>.
     */
    protected FileItem createItem( String path,
                                   int requestSize )
    {
        return FileItem.newInstance(path,
                                    getFileName(),
                                    getHeader("Content-type"),
                                    requestSize);
    }

    /**
     * Stores parsed headers as key - value pairs.
     */
    Hashtable headers;

    /**
     * <p> Parses the <code>header-part</code> and stores as key -
     * value pairs.
     *
     * <p> If there are multiple headers of the same names, the name
     * will map to a comma-separated list containing the values.
     *
     * @param headerPart The <code>header-part</code> of the current
     * <code>encapsulation</code>.
     */
    protected void parseHeaders( String headerPart )
    {
        if (headers == null)
            headers = new Hashtable();
        else
            headers.clear();
        char buffer[] = new char[MAX_HEADER_SIZE];
        boolean done = false;
        int j=0;
        int i;
        String header, headerName, headerValue;
        try
        {
            while (!done)
            {
                i=0;
                // Copy a single line of characters into the buffer,
                // omitting trailing CRLF.
                while (i<2 || buffer[i-2] != '\r' || buffer[i-1] != '\n')
                {
                    buffer[i++] = headerPart.charAt(j++);
                }
                header = new String(buffer, 0, i-2);
                if (header.equals(""))
                {
                    done = true;
                }
                else
                {
                    if (header.indexOf(':') == -1)
                    {
                        // This header line is malformed, skip it.
                        continue;
                    }
                    headerName = header
                        .substring(0, header.indexOf(':')).trim().toLowerCase();
                    headerValue = 
                        header.substring(header.indexOf(':')+1).trim();
                    if (headers.get(headerName) != null)
                    {
                        // More that one heder of that name exists,
                        // append to the list.
                        headers.put(headerName, 
                            (String)headers.get(headerName)+","+headerValue);
                    }
                    else
                    {
                        headers.put(headerName, headerValue);
                    }
                }
            }
        }
        catch(IndexOutOfBoundsException e)
        {
            // Headers were malformed. continue with all that was
            // parsed.
        }
    }

    /**
     * <p> Returns a header with specified name.
     *
     * @param name The name of the header to fetch.
     * @return The value of specified header, or a comma-separated
     * list if there were multiple headers of that name.
     */
    protected String getHeader( String name )
    {
        return (String)headers.get(name.toLowerCase());
    }
}
