/*
   Copyright (C) 1997-2001 Id Software, Inc.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

   See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

 */
#include "g_local.h"
#include "g_gametypes.h"

static struct
{
	mm_state_t state;
	unsigned int started;

	struct
	{
		int gametype;
		int numbots;
		char *password;
		int scorelimit;
		float timelimit, warmuptimelimit;
		qboolean warmupenabled;
		qboolean falldamage;
	} old;
} mm;

void G_SetupMM( int maxclients, int gametype, const char *pass, int scorelimit, float timelimit, qboolean falldamage )
{
	if( mm.state != MM_UNREADY )
		return;

	G_Printf( "Setting up server for matchmaker\n" );

	// store old cvar values
	mm.old.gametype = GS_Gametype_FindByShortName( g_gametype->string );
	mm.old.numbots = g_numbots->integer;
	mm.old.scorelimit = g_scorelimit->integer;
	mm.old.timelimit = g_timelimit->value;
	mm.old.warmuptimelimit = g_warmup_timelimit->value;
	mm.old.warmupenabled = g_warmup_enabled->integer;
	mm.old.falldamage = g_allow_falldamage->integer;
	mm.old.password = G_Malloc( strlen( password->string ) + 1 );
	Q_strncpyz( mm.old.password, password->string, strlen( password->string ) + 1 );

	// setup ready for match
	trap_Cvar_Set( "g_gametype", GS_Gametype_ShortName( gametype ) );
	trap_Cvar_Set( "g_numbots", "0" );
	trap_Cvar_Set( "g_scorelimit", va( "%d", scorelimit ) );
	trap_Cvar_Set( "g_timelimit", va( "%f", timelimit ) );
	trap_Cvar_Set( "g_warmup_timelimit", "5" );
	trap_Cvar_Set( "g_warmup_enabled", "1" );
	trap_Cvar_Set( "g_allow_falldamage", va( "%d", falldamage ) );
	trap_Cvar_Set( "password", pass );

	mm.state = MM_RESTART;

	G_Match_RestartLevel();
}

static void G_ResetMM( void )
{
	if( mm.state == MM_UNREADY )
		return;

	G_Printf( "Resetting server after matchmaker\n" );

	trap_Cvar_Set( "g_gametype", GS_Gametype_ShortName( mm.old.gametype ) );
	trap_Cvar_Set( "g_numbots", va( "%d", mm.old.numbots ) );
	trap_Cvar_Set( "g_scorelimit", va( "%d", mm.old.scorelimit ) );
	trap_Cvar_Set( "g_timelimit", va( "%f", mm.old.timelimit ) );
	trap_Cvar_Set( "g_warmup_timelimit", va( "%d", mm.old.warmuptimelimit ) );
	trap_Cvar_Set( "g_warmup_enabled", va( "%d", mm.old.warmupenabled ) );
	trap_Cvar_Set( "g_allow_falldamage", va( "%d", mm.old.falldamage ) );
	trap_Cvar_Set( "password", mm.old.password );

	G_Free( mm.old.password );
	memset( &mm.old, 0, sizeof( mm.old ) );
	mm.state = MM_UNREADY;
	mm.started = 0;
}

void G_CheckMM( void )
{
	int i, players = 0;
	// give clients a chance to connect
	if( game.realtime - mm.started >= 60000 && mm.state == MM_READY )
	{
		for( i = 1 ; i <= game.maxclients ; i++ )
		{
			if( !game.edicts[i].r.inuse )
				continue;

			// exclude tvclients and bots
			if( game.edicts[i].r.client && !game.edicts[i].r.client->pers.tv && !( game.edicts[i].r.svflags & SVF_FAKECLIENT ) )
				players++;
		}

		if( !players )
		{
			G_ResetMM();
			G_Match_RestartLevel();
		}
	}
}

void G_SetMMReadyState( void )
{
	if( mm.state != MM_RESTART )
		return;

	mm.state = MM_READY;
	mm.started = game.realtime;
}
