;; @(#) gnat-fix-error.el --- utilities for automatically fixing
;; errors reported by the GNAT Ada compiler.

;; Copyright (C) 1999-2000 Stephen Leake.

;; Author: Stephen Leake      <Stephen_Leake@acm.org>
;; Maintainer: Stephen Leake      <Stephen_Leake@acm.org>
;; CVS version:   $Revision: 1.18 $
;; Keywords: languages ada error

;; This file is not part of GNU Emacs

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Notes

;; You must use -gnatf (extended error messages) for this to work
;; well.

(require 'ada-mode)

;;; General functions

(defconst gnat-fix-compilation-unit-start-regexp
  "^separate\\|^package\\|^private package\\|^procedure\\|^function\\|^generic"
  "regexp matching start of compilation unit, after context clauses.
Assumes keyword is at left margin.")

(defun gnat-fix-add-with-clause (package-name)
  "Add a with_clause for PACKAGE_NAME, at the start of the compilation
unit in the current buffer."
  (goto-char (point-min))
  (ada-search-ignore-string-comment gnat-fix-compilation-unit-start-regexp nil)
  (beginning-of-line)
  (insert "with ")
  (gnat-fix-insert-unit-name package-name)
  (insert ";\n"))

(defun gnat-fix-extend-with-clause (child-name)
  "Assuming point is in a full name, just before CHILD-NAME, add or
extend a with_clause to include CHILD-NAME."
  (let ((parent-name-end (point)))
    ;; find the full parent name
    ;; search-backward-regexp matches the shortest string, not the
    ;; longest, so we can't use that here. We need the full parent
    ;; package name, in case it's not unique. So skip back to
    ;; whitespace, then match the name forward.
    (search-backward-regexp ada-whitespace-regexp)
    (forward-char 1) ; skip forward over found whitespace
    (search-forward-regexp ada-name-regexp parent-name-end)
    (let ((parent-name (match-string 0)))
      (goto-char (point-min))
      (ada-search-ignore-string-comment (concat "with " parent-name ";") nil)
      (if (eobp)
          ;; oops, no with_clause found (we are in a package body, with_clause for parent is in spec).
          ;; insert a new one
          (gnat-fix-add-with-clause (concat parent-name "." child-name))
        (progn
          (forward-char -1) ; skip back over semicolon
          (insert "." child-name))))))

(defun gnat-fix-insert-unit-name (unit-name)
  "Insert (at point) and capitalize unit-name, normally gotten from
file-name, and thus all lower-case."
  (let ((start-point (point))
        search-bound)
    (insert unit-name)
    (setq search-bound (point))
    (insert " ") ; separate from following words, if any, for ada-adjust-case-identifier
    (goto-char start-point)
    (while (search-forward "." search-bound t)
      (forward-char -1)
      (ada-adjust-case-identifier)
      (forward-char 1))
    (goto-char search-bound)
    (ada-adjust-case-identifier)
    (delete-char 1)))

;;; ada project file stuff

(defun gnat-fix-prj-show-source (file-name line)
  "Show source file at line (string), using ff-get-file-name, with
search path from current buffer's ada project file."
  (ada-require-prj-file)
  (find-file (ff-get-file-name ada-prj-src-dir file))
  (goto-line (string-to-number line)))

;;; gnat specific stuff

(defconst gnat-file-line-regexp
  "\\([a-z0-9-_./:]+\\):\\([0-9]+\\)"
  "match gnat-style file and line, possibly including full path due to preprocessor")

(defconst gnat-quoted-name-regexp
  "\"\\([a-zA-Z0-9_.]+\\)\""
  "regexp to extract the quoted names in error messages")

(defconst gnat-quoted-punctuation-regexp
  "\"\\([,:;=()|]+\\)\""
  "regexp to extract quoted punctuation in error messages")

(defconst gnat-quoted-operator-regexp
  "\\(\"[+*/-]+\"\\)"
  "regexp to extract quoted operator in error messages")

(defconst gnat-predefined-package-alist
  '(("interfac" . "Interfaces")
    ("i-c" . "Interfaces.C")
    ("i-cstrin" . "Interfaces.C.Strings")
    )
  "Alist (filename . package name) of GNAT file names for predefined Ada packages.")

(defun gnat-unit-name-from-file-name (file-name)
  "Return the Ada unit name corresponding to FILENAME, using gnat
default naming convention. Special case the predefined packages, since
gnat truncates them to 8 characters."
  (let* ((nondirname (file-name-nondirectory file-name))
         (unit-name
          (if (equal (file-name-extension file-name t) ".gp")
              (file-name-sans-extension (file-name-sans-extension nondirname))
            (file-name-sans-extension nondirname)) )
         (predefined (cdr (assoc unit-name gnat-predefined-package-alist))))

    (if predefined
        predefined
      (while (string-match "-" unit-name)
        (setq unit-name (replace-match "." nil t unit-name)))
      unit-name)))

(defun gnat-fix-compiler-error ()
  "Attempt to fix the current compiler error. Assumes point is at
error line and column (ie, after next-error). Leave point at fix."
  (interactive)

  (let ((source-buffer (current-buffer))
        compilation-buffer-start)

    ;; Goto the error message. Sometimes (the very first time the
    ;; compilation buffer is used?) point is not at the right place in
    ;; the compilation buffer. This sequence seems to fix it.
    (pop-to-buffer compilation-last-buffer)
    (pop-to-buffer source-buffer)
    (set-buffer compilation-last-buffer)

    ;; Save starting point in compilation buffer, in case we need to
    ;; repeat operation. We don't use save-excursion, because we want
    ;; point to change in source-buffer
    (setq compilation-buffer-start (point))
    (if (not (search-forward " " nil t))
        (error "gnat-fix-error: space not found from point %d" (point)))

    ;; recognize it, handle it
    (unwind-protect
        (cond
         ;; This list will get long, so let's impose some order.
         ;;
         ;; First expressions that start with a named regexp, alphabetical by variable name.
         ;;
         ;; Then expressions that start with a string, alphabetical by string.
         ;;
         ;; Then style errors.

         ((looking-at (concat gnat-quoted-name-regexp " is not visible"))
          (let ((ident (match-string 1)))
            (next-line 1)
            (let ((unit-file
                   (cond
                    ((looking-at (concat ".*at " gnat-file-line-regexp ".*at " gnat-file-line-regexp))
                      (match-string 3))
                    ((looking-at (concat "non-visible \\((private) \\)?declaration at " gnat-file-line-regexp))
                      (match-string 2)))))
              (pop-to-buffer source-buffer)
              ;; We either need to add a with_clause for a package, or
              ;; prepend the package name here (or add a use clause, but I
              ;; don't want to do that automatically). unit-name may be
              ;; only the prefix of the real package name, but in that
              ;; case we'll be back after the next compile; no way to get
              ;; the full package name (without the function/type name) now.
              (let ((unit-name (gnat-unit-name-from-file-name unit-file)))
                (cond
                 ((looking-at (concat unit-name "\\."))
                  (gnat-fix-add-with-clause unit-name))
                 (t
                  (gnat-fix-insert-unit-name unit-name)
                  (insert ".")))))))

         ((looking-at (concat gnat-quoted-name-regexp " is undefined"))
          ;; We either need to add a with_clause for a package, or
          ;; something is spelled wrong. Check next line for spelling error.
          (let ((unit-name (match-string 1))
                correct-spelling)
            (save-excursion
              (next-line 1)
              (if (looking-at (concat "possible misspelling of " gnat-quoted-name-regexp))
                  ;; correctable misspelling
                  (progn
                    (setq correct-spelling (match-string 1))
                    (pop-to-buffer source-buffer)
                    (search-forward unit-name)
                    (replace-match correct-spelling)
                    );; progn

                ;; assume missing with
                (pop-to-buffer source-buffer)
                (gnat-fix-add-with-clause unit-name)))))

         ((looking-at (concat gnat-quoted-name-regexp " is not a component of type"))
          ;; Check next line for spelling error.
          (let ((unit-name (match-string 1))
                correct-spelling)
            (save-excursion
              (next-line 1)
              (if (looking-at (concat "possible misspelling of " gnat-quoted-name-regexp))
                  ;; correctable misspelling
                  (progn
                    (setq correct-spelling (match-string 1))
                    (pop-to-buffer source-buffer)
                    (search-forward unit-name)
                    (replace-match correct-spelling)
                    );; progn

                ;; else can't deal with it
                (error "error not recognized"))
                )))

         ((looking-at (concat gnat-quoted-name-regexp " not declared in " gnat-quoted-name-regexp))
          (let ((child-name (match-string 1)))
            ;; guess that "child" is a child package, and extend the with_clause
            (pop-to-buffer source-buffer)
            (gnat-fix-extend-with-clause child-name)))

         ((or
           (looking-at (concat gnat-quoted-punctuation-regexp " should be " gnat-quoted-punctuation-regexp))
           (looking-at (concat gnat-quoted-punctuation-regexp " illegal here, replaced by "
                               gnat-quoted-punctuation-regexp))
           )
          (let ((bad (match-string-no-properties 1))
                (good (match-string-no-properties 2)))
            (pop-to-buffer source-buffer)
            (looking-at bad)
            (delete-region (match-beginning 0) (match-end 0))
            (insert good)))

         ;; Now expressions that start with a string, alphabetical by string.

         ((looking-at (concat "\"end " ada-identifier-regexp ";\" expected"))
          (let ((expected-name (match-string 1)))
            (pop-to-buffer source-buffer)
            (looking-at (concat "end " ada-identifier-regexp ";"))
            (goto-char (match-end 1))   ; just before ';'
            (delete-region (match-beginning 1) (match-end 1))
            (insert expected-name)))

         ((looking-at "extra right paren")
          (pop-to-buffer source-buffer)
          (delete-char 1))

         ((looking-at (concat "expected type " gnat-quoted-name-regexp))
          (let ((type (match-string 1)))
            (pop-to-buffer source-buffer)
            (if (looking-at "\\.")
                (progn
                  (message "move to beginning of expression and repeat")
                  (ding))
              (progn
                (insert type " (")
                (message "move to end of expression and insert matching paren")
                (ding)))))

         ((looking-at (concat "missing " gnat-quoted-punctuation-regexp))
          (let ((stuff (match-string-no-properties 1)))
            (progn
              (set-buffer source-buffer)
              (insert stuff))))

         ((looking-at (concat "missing string quote"))
          (progn
            (set-buffer source-buffer)
            (insert "\"")))

         ((looking-at (concat "missing #"))
          (progn
            (set-buffer source-buffer)
            (insert "#")))

         ((looking-at (concat "missing body for " gnat-quoted-name-regexp " declared at " gnat-file-line-regexp))
          (let ((file (match-string 2))
                (line (match-string 3)))
            (set-buffer source-buffer)  ; for ada project file
            (gnat-fix-prj-show-source file line)))

         ((looking-at (concat "missing with_clause for child unit " gnat-quoted-name-regexp))
          (let ((child-name (match-string-no-properties 1)))
            (pop-to-buffer source-buffer)
            (gnat-fix-extend-with-clause child-name)))

         ((looking-at (concat "missing with for " gnat-quoted-name-regexp))
          (let ((package-name (match-string-no-properties 1)))
            (pop-to-buffer source-buffer)
            (gnat-fix-add-with-clause package-name)))

         ((looking-at (concat "no selector " gnat-quoted-name-regexp))
          ;; Check next line for spelling error.
          (let ((unit-name (match-string 1))
                correct-spelling)
            (save-excursion
              (next-line 1)
              (if (looking-at (concat "possible misspelling of " gnat-quoted-name-regexp))
                  ;; correctable misspelling
                  (progn
                    (setq correct-spelling (match-string 1))
                    (pop-to-buffer source-buffer)
                    (search-forward unit-name)
                    (replace-match correct-spelling)
                    );; progn

                ;; else can't deal with it
                (error "error not recognized"))
                )))

         ((or
           (looking-at (concat "not fully conformant with declaration at " gnat-file-line-regexp))
           (looking-at (concat "not type conformant with declaration at " gnat-file-line-regexp)))
          (let ((file (match-string 1))
                (line (match-string 2)))
            (pop-to-buffer source-buffer) ; for ada project file
            (gnat-fix-prj-show-source file line)))

         ((looking-at "numeric literal cannot start with point")
          (progn
            (pop-to-buffer source-buffer)
            (insert "0")))

         ;; types in pre-defined packages don't give file.
         ((looking-at (concat "operator for \\(private \\)?type " gnat-quoted-name-regexp
                              "\\( defined at " gnat-file-line-regexp "\\)?"))
          (let ((file (match-string 4))
                (type (match-string 2)))
            (pop-to-buffer source-buffer)
            ;; search back to either start of current declarative region,
            ;; or end of current subprogram's declarative region
            (ada-search-ignore-string-comment "\\<begin\\>\\|\\<is\\>" t)
            (cond
             ((looking-at "is")
              ;; might be a case statement
              (while (progn
                       (ada-search-ignore-string-comment (concat "\\<case\\>\\|" ada-subprog-start-re) t)
                       (if (looking-at "case")
                           t
                         (ada-search-ignore-string-comment "is")
                         nil)))
              (end-of-line))

             ((looking-at "begin")
              (forward-line -1)
              (end-of-line))
             )
            (newline-and-indent)
            (insert "use type ")
            (if file
                (progn
                  (gnat-fix-insert-unit-name (gnat-unit-name-from-file-name file))
                  (insert ".")))
            (insert type ";")))

         ((looking-at (concat "undefined selector for type " gnat-quoted-name-regexp
                              " defined at " gnat-file-line-regexp))
          (let ((file (match-string 2))
                (line (match-string 3)))
            (pop-to-buffer source-buffer) ; for ada project file
            (gnat-fix-prj-show-source file line)))

         ((looking-at "unexpected right parenthesis")
            (set-buffer source-buffer)
            (delete-char 1))

         ((looking-at "unexpected semicolon ignored")
            (set-buffer source-buffer)
            (delete-char 1))

         ((looking-at (concat "warning: \\(instantiation of\\|call to\\) "
                              gnat-quoted-name-regexp
                              " may raise Program_Error"))
          (progn
            (next-line 1)
            (if (looking-at (concat "warning: missing pragma Elaborate_All for " gnat-quoted-name-regexp))
                (let ((unit (match-string 1)))
                  (pop-to-buffer source-buffer)
                  (goto-char (point-min))
                  (ada-search-ignore-string-comment unit nil)
                  (forward-line)
                  (insert "pragma Elaborate_All (")
                  (gnat-fix-insert-unit-name unit)
                  (insert ");\n"))
              (message "error not recognized"))))

         ;; Now style errors.
         ((looking-at "(style) bad capitalization, mixed case required")
          (progn
            (set-buffer source-buffer)
            (ada-capitalize-word)))

         ((or (looking-at "(style) bad indentation")
              (looking-at "(style) bad column"))
          (progn
            (set-buffer source-buffer)
            (ada-indent-current)))

         ((or (looking-at (concat "(style) bad identifier casing, should be " gnat-quoted-name-regexp)) ; gnat 3.12
              (looking-at (concat "(style) bad casing of " gnat-quoted-name-regexp))) ; gnat 3.13
          (let ((correct (match-string-no-properties 1))
                end)
            ;; gnat leaves point on first bad character, but we need to replace the whole word
            (set-buffer source-buffer)
            (skip-syntax-backward "w")
            (setq end (point))
            (skip-syntax-forward "w")
            (delete-region (point) end)
            (insert correct)))

         ((looking-at (concat "(style) \"end " gnat-quoted-operator-regexp))
          (let ((correct (match-string-no-properties 1)))
            (set-buffer source-buffer)
            (skip-syntax-forward "w")
            (insert (concat " " correct))))

         ((looking-at "(style) \"end\" in wrong column")
          (progn
            (set-buffer source-buffer)
            (ada-indent-current)))

         ((looking-at "(style) horizontal tab not allowed")
          (progn
            (set-buffer source-buffer)
            (delete-char 1)
            (ada-indent-current)))

         ((looking-at "(style) misplaced \"then\"")
          (let (then-pos)
            (set-buffer source-buffer)
            (setq then-pos (point))
            (previous-line 1)
            (if (looking-at "if\\|elsif")
                (progn
                  ;; delete new-line and whitespace before "then"
                  (end-of-line)
                  (delete-region (point) then-pos)
                  (insert " "))
              (error "can't fix \"then\""))))

         ((looking-at "(style) reserved words must be all lower case")
          (progn
            (set-buffer source-buffer)
            ;; error places point on first char in word
            (downcase-word 1)))

         ((or (looking-at "(style) space not allowed")
              (looking-at "(style) trailing spaces not permitted")
              (looking-at "(style) form feed not allowed"))
          (progn
            (set-buffer source-buffer)
            ;; Error places point on space. More than one trailing
            ;; space should be fixed by ada-remove-trailing-spaces,
            ;; once the file is modified.
            (delete-char 1)))

         ((looking-at "(style) space required")
          (progn
            (set-buffer source-buffer)
            (insert " ")))

         (t
          (error "error not recognized"))

         )    ;; end of (cond
      ;; restore compilation buffer point
      (set-buffer compilation-last-buffer)
      (goto-char compilation-buffer-start)
      )))

(provide 'gnat-fix-error)
;; end of file

