# Lexical analyzer base classes.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/token"
require "adlint/source"
require "adlint/traits"
require "adlint/error"
require "adlint/util"

module AdLint #:nodoc:

  # == DESCRIPTION
  # Token queue to interface to the parser.
  class TokenQueue < Array
    # === DESCRIPTION
    # Constructs an empty token queue or a solid token queue from specified
    # token array.
    #
    # === PARAMETER
    # _token_array_:: TokenArray -- Array of tokens.
    def initialize(token_array = nil)
      if token_array
        super
      else
        super()
      end
    end

    def expect(token_type)
      token = self.first
      token && token.type == token_type ? true : false
    end
  end

  # == DESCRIPTION
  # Object represents the whole content of something.
  class Content
    def location
      subclass_responsibility
    end

    def empty?
      subclass_responsibility
    end
  end

  # == DESCRIPTION
  # Object represents the whole content of the string.
  class StringContent < Content
    include CodingStyleAccessor
    extend Memoizable

    # === DESCRIPTION
    # Constructs the content object of the string.
    #
    # === PARAMETER
    # _str_:: String -- Target string.
    # _fpath_:: Pathname -- File path name contains the target string.
    # _line_no_:: Integer -- Initial line-no of the target string.
    def initialize(str, fpath = nil, line_no = 1, column_no = 1)
      @scanner = StringScanner.new(str)
      @fpath = fpath
      @line_no = line_no
      @column_no = column_no
      @appearance_column_no = column_no
    end

    def location
      Location.new(@fpath, @line_no, @column_no, @appearance_column_no)
    end
    memoize :location

    # === DESCRIPTION
    # Scans a token.
    #
    # === PARAMETER
    # _regexp_:: Regexp -- Token pattern.
    #
    # === RETURN VALUE
    # String -- Token string or nil.
    def scan(regexp)
      token = @scanner.scan(regexp)
      if token
        update_location(token)
        token
      else
        nil
      end
    end

    def check(regexp)
      @scanner.check(regexp)
    end

    # === DESCRIPTION
    # Skips a content.
    #
    # === PARAMETER
    # _length_:: Integer -- Skipping content length.
    #
    # === RETURN VALUE
    # String -- Eaten string.
    def eat!(length = 1)
      scan(/.{#{length}}/m)
    end

    # === DESCRIPTION
    # Checks whether this content is currently empty or not.
    #
    # === RETURN VALUE
    # Boolean -- Returns true if already empty.
    def empty?
      @scanner.eos?
    end

    private
    def update_location(token)
      if (nl_count = token.count("\n")) > 0
        @line_no += nl_count
        last_line = token[token.rindex("\n")..-1]
        @column_no = last_line.length
        @appearance_column_no = last_line.gsub(/\t/, " " * tab_width).length
      else
        @column_no += token.length
        @appearance_column_no += token.gsub(/\t/, " " * tab_width).length
      end
      forget_location_memo
    end
  end

  # == DESCRIPTION
  # Object represents the whole content of the source file.
  class SourceContent < StringContent
    # === DESCRIPTION
    # Constructs the content object of the source file.
    #
    # === PARAMETER
    # _source_:: Source -- Target source object.
    def initialize(source)
      super(source.open { |io| io.read }, source.fpath)
    end
  end

  class TokensContent < Content
    def initialize(token_array)
      @token_array = token_array
    end

    def location
      empty? ? nil : @token_array.first.location
    end

    def empty?
      @token_array.empty?
    end

    def next_token
      empty? ? nil : @token_array.shift
    end
  end

  # == DESCRIPTION
  # Generic lexical analysis context.
  class LexerContext
    # === DESCRIPTION
    # Constructs a lexical analysis context object.
    #
    # === PARAMETER
    # _content_:: SourceContent | StringContent -- Target content.
    def initialize(content)
      @content = content
    end

    # === VALUE
    # SourceContent | StringContent -- The target content of this context.
    attr_reader :content

    # === DESCRIPTION
    # Reads the current location of the target content.
    #
    # === RETURN VALUE
    # Location -- Current location.
    def location
      @content.location
    end
  end

  # == DESCRIPTION
  # Base class of the lexical analyzer of the string.
  class StringLexer
    # === DESCRIPTION
    # Constructs a lexical analyzer of the string.
    #
    # === PARAMETER
    # _str_:: String -- Target string.
    def initialize(str)
      @str = str
    end

    # === DESCRIPTION
    # Executes the lexical analysis.
    #
    # === RETURN VALUE
    # TokenArray -- Scanned tokens.
    def execute
      context = create_context(@str)
      tokenize(context)
    rescue Error
      raise
    rescue => ex
      if context
        raise InternalError.new(ex, context.location)
      else
        raise InternalError.new(ex, nil)
      end
    end

    private
    # === DESCRIPTION
    # Creates the context object.
    #
    # Subclasses must implement this method.
    #
    # === PARAMETER
    # _str_:: String -- Target string object.
    def create_context(str)
      subclass_responsibility
    end

    # === DESCRIPTION
    # Tokenize the target content.
    #
    # Subclasses must implement this method.
    #
    # === PARAMETER
    # _context_:: LexerContext -- Lexical analysis context.
    def tokenize(context)
      subclass_responsibility
    end
  end

  class TokensRelexer
    def initialize(token_array)
      @context = create_context(token_array)
    end

    def next_token
      tokenize(@context)
    rescue Error
      raise
    rescue => ex
      raise InternalError.new(ex, @context.location)
    end

    private
    def create_context(token_array)
      subclass_responsibility
    end

    def tokenize(context)
      subclass_responsibility
    end
  end

end
