#define _VFUNC_EUTROPH_C_
/*
   --------------------------------------------------------------
   File vfunc_eutroph.C 
   of ADOL-C version 1.8.0                        as of Nov/01/98
   --------------------------------------------------------------

   Example: function module containing the eutroph example
            (based on eutroph.C of version 1.7)

   Each << function module >> contains:
          
     (1) const char* const controlFileName 
     (2) int indepDim; 
     (3) int depDim; 

     (4) void initProblemParameters( void )
     (5) void initIndependents( double* indeps )
     (6) void originalVectorFunction( double* indeps, double* deps )
     (7) void tapingVectorFunction( int tag, double* indeps, double* deps )   

   Last changes: 
     980824 olvo first version

   --------------------------------------------------------------
*/


/****************************************************************************/
/*                                                                 INCLUDES */
#include "adouble.h"
#include <math.h>


/****************************************************************************/
/*                                                         GLOBAL VARIABLES */

/*--------------------------------------------------------------------------*/
/*                                                        Control file name */
const char* controlFileName = "eutrophexam.ctrl";

/*--------------------------------------------------------------------------*/
/*                                                               Dimensions */
int indepDim;
int depDim;

/*--------------------------------------------------------------------------*/
/*                                       Other problem dependent parameters */
const double IK    = 0.11 ;
const double FRZ   = 0.3 ;
const double EFFUZ = 0.6;
const double PRITZ = 1.0e-3;
const double RESP  = 5.0e-3;
const double sinK  = 5.0e-3;
const double PRITA = 0.1;
const double RZ    = 1.0e-2;
const double K2    = 4.0e-2;
const double K3    = 5.0e-1;
const double KSP   = 2.0e2;
const double KSF   = 1.0;
const double BETA  = 100.0/1.25;
const double ALPHA = 0.002;
const double TRZ   = 2.0;
const double EPSP  = 0.4;
const double FI1   = 230.4;
const double FI3   = 282.8;
const double FI4   = 127.5;
const double FI5   = 141.9;
const double p     = 40.0;
const double DEPTH = 45;
const double MORITZ= 0.075;
const double Q     = 0.786E6;
const double VND   = 0.265E9;
/* fix controls */
const double PRFOS = 0.5*p;
const double M     = 0.1;
const double ZMIX  = (45+RZ)/2;  
const double QIV   = 0.297E-02/3;


/****************************************************************************/
/*                                                  INIT PROBLEM PARAMETERS */
void initProblemParameters( void )
{ fprintf(stdout,"EUTROPHEXAM (ADOL-C Example)\n\n");

  /* number of indeps & deps */ 
  indepDim = 5;
  depDim   = 5;
}


/****************************************************************************/
/*                                                        INITIALIZE INDEPs */
void initIndependents( double* indeps )
{ indeps[0]  = 0.5;   
  indeps[1]  = 0.0005;
  indeps[2]  = 4.0;
  indeps[3]  = 0.01; /* originally 0.0 */
  indeps[4]  = 0.02; /* originally 0.0 */
}


/****************************************************************************/
/*                                                 ORIGINAL SCALAR FUNCTION */

/*--------------------------------------------------------------------------*/
/*                                                     The eutroph function */
void eutroph( double* indeps, double* deps )
{ double T, TEMP, FOTOP, I, PIDI, EPS, temp, temp2, GROW;
  double V;
  T     = indeps[4];
  TEMP  = 9.5+7.9*sin(T+FI1);
  FOTOP = 12.0+4.19*sin(T+280.0);
  I     = 229.0+215.0*sin(T+FI3)+15.3*sin(2.0*T+FI4)+ 21.7*sin(3.0*T+FI5);
  PIDI  = 0.8+.25*cos(T)-.12*cos(2.*T);
  V     = VND;
  if (T < 72) 
    I *= 0.603;
  EPS   = ALPHA * indeps[0] + indeps[3] + EPSP;
  temp  = I * exp(-EPS*ZMIX);
  temp2 = 2*IK*FOTOP;
  GROW  = 1.2*FOTOP/EPS/ZMIX * (1.333 * atan (I / temp2)
          -IK*FOTOP / I * log( 1 + pow((I /temp2 ),2)) 
          -1.333 * atan (temp / temp2)
          +IK*FOTOP/temp* log( 1+pow(temp/temp2, 2) )) 
                          * indeps[2] /(KSF+indeps[2]) 
          * 0.366 * pow(K2,0.52) * exp(0.09*TEMP) * pow(indeps[0],(1-0.52));
  deps[0] = GROW - RESP * TEMP * indeps[0] - FRZ 
            * indeps[0] * indeps[1] - sinK * PIDI * indeps[0] 
            + (PRITA - indeps[0]) * Q/VND;
  deps[1] = FRZ * indeps[0] / K2 * indeps[1] / 1000 
            * EFFUZ*KSP / KSP+indeps[0] 
            - RZ * indeps[1] - MORITZ * indeps[1] + (PRITZ - indeps[1] ) * Q/V;
  deps[2] = K3 * (-GROW + RESP * TEMP * indeps[0] + FRZ * indeps[0] 
            * indeps[1] * (1 - EFFUZ*KSP /(KSP+indeps[0]) ) + RZ * K2 * 1000 *
              indeps[1] + MORITZ * K2 * 1000 * indeps[1] ) 
            + (PRFOS - indeps[2])* Q/V;
  deps[3] = (- indeps[3] * Q  + BETA * M / TRZ)/VND;
  deps[4] = 1;
}

/*--------------------------------------------------------------------------*/
/*                                                   The interface function */
void originalVectorFunction( double* indeps, double* deps )
{ eutroph(indeps,deps);
}


/****************************************************************************/
/*                                                   TAPING SCALAR FUNCTION */

/*--------------------------------------------------------------------------*/
/*                                              The active eutroph function */
void activeEutroph( adouble* indeps, adouble* deps )
{ adouble T, TEMP, FOTOP, I, PIDI, EPS, temp, temp2, GROW;
  double V;
  T     = indeps[4];
  TEMP  = 9.5+7.9*sin(T+FI1);
  FOTOP = 12.0+4.19*sin(T+280.0);
  I     = 229.0+215.0*sin(T+FI3)+15.3*sin(2.0*T+FI4)+ 21.7*sin(3.0*T+FI5);
  PIDI  = 0.8+.25*cos(T)-.12*cos(2.*T);
  V     = VND;
  if (T < 72) 
    I *= 0.603;
  EPS   = ALPHA * indeps[0] + indeps[3] + EPSP;
  temp  = I * exp(-EPS*ZMIX);
  temp2 = 2*IK*FOTOP;
  GROW  = 1.2*FOTOP/EPS/ZMIX * (1.333 * atan (I / temp2)
          -IK*FOTOP / I * log( 1 + pow((I /temp2 ),2)) 
          -1.333 * atan (temp / temp2)
          +IK*FOTOP/temp* log( 1+pow(temp/temp2, 2) )) 
                          * indeps[2] /(KSF+indeps[2]) 
          * 0.366 * pow(K2,0.52) * exp(0.09*TEMP) * pow(indeps[0],(1-0.52));
  deps[0] = GROW - RESP * TEMP * indeps[0] - FRZ 
            * indeps[0] * indeps[1] - sinK * PIDI * indeps[0] 
            + (PRITA - indeps[0]) * Q/VND;
  deps[1] = FRZ * indeps[0] / K2 * indeps[1] / 1000 
            * EFFUZ*KSP / KSP+indeps[0] 
            - RZ * indeps[1] - MORITZ * indeps[1] + (PRITZ - indeps[1] ) * Q/V;
  deps[2] = K3 * (-GROW + RESP * TEMP * indeps[0] + FRZ * indeps[0] 
            * indeps[1] * (1 - EFFUZ*KSP /(KSP+indeps[0]) ) + RZ * K2 * 1000 *
              indeps[1] + MORITZ * K2 * 1000 * indeps[1] ) 
            + (PRFOS - indeps[2])* Q/V;
  deps[3] = (- indeps[3] * Q  + BETA * M / TRZ)/VND;
  deps[4] = 1;
}


/*--------------------------------------------------------------------------*/
/*                                                   The interface function */
void tapingVectorFunction( int tag, double* indeps, double* deps )
{ int i;
  trace_on(tag);
  adouble* activeIndeps = new adouble[indepDim];
  adouble* activeDeps   = new adouble[depDim];
  adouble* aIP = activeIndeps;
  double*  iP  = indeps; 
  for (i=0; i<indepDim; i++)
     *aIP++ <<= *iP++; 
  activeEutroph(activeIndeps,activeDeps);
  aIP = activeDeps;
  iP  = deps; 
  for (i=0; i<depDim; i++)
    *aIP++ >>= *iP++; 
  trace_off();
}

#undef _VFUNC_EUTROPH_C_





