/*
 *	aegis - project change supervisor
 *	Copyright (C) 2000, 2001 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate batch_fakes
 */

#include <ac/stdio.h>
#include <change.h>
#include <change/file.h>
#include <change/test/batch_fake.h>
#include <error.h> /* for assert */
#include <fstate.h>
#include <project.h>
#include <project/file.h>
#include <str_list.h>
#include <sub.h>
#include <trace.h>
#include <user.h>


batch_result_list_ty *
change_test_batch_fake(cp, wlp, up, bl)
	change_ty	*cp;
	string_list_ty	*wlp;
	user_ty		*up;
	int		bl;
{
	size_t		j;
	string_ty	*dir;
	int		(*run_test_command)_((change_ty *, user_ty *,
				string_ty *, string_ty *, int, int));
	cstate		cstate_data;
	batch_result_list_ty *result;
	int		persevere;

	/*
	 * which command
	 */
	trace(("change_test_batch_fake(cp = %08lX, wlp = %08lX, up = %08lX, bl = %d)\n{\n",
		(long)cp, (long)wlp, (long)up, bl));
	cstate_data = change_cstate_get(cp);
	run_test_command = change_run_test_command;
	if (cstate_data->state != cstate_state_being_integrated)
		run_test_command = change_run_development_test_command;
	persevere = user_persevere_preference(up, 1);

	/*
	 * directory depends on the state of the change
	 *
	 * During long tests the automounter can unmount the
	 * directory referenced by the ``dir'' variable.
	 * To minimize this, it is essential that they are
	 * unresolved, and thus always trigger the automounter.
	 */
	dir = project_baseline_path_get(cp->pp, 0);
	if (!bl && !cp->bogus)
	{
		switch (cstate_data->state)
		{
		case cstate_state_awaiting_development:
			assert(0);

		case cstate_state_completed:
			break;

		case cstate_state_being_integrated:
			dir = change_integration_directory_get(cp, 0);
			break;

		case cstate_state_being_developed:
		case cstate_state_awaiting_review:
		case cstate_state_being_reviewed:
		case cstate_state_awaiting_integration:
			dir = change_development_directory_get(cp, 0);
			break;
		}
	}

	/*
	 * allocate a place to store the results
	 */
	result = batch_result_list_new();

	/*
	 * one command per file
	 */
	for (j = 0; j < wlp->nstrings; ++j)
	{
		string_ty	*fn;
		string_ty	*fn_abs;
		fstate_src	src_data;
		int		inp;
		int		exit_status;
		sub_context_ty	*scp;

		/*
		 * resolve the file name
		 */
		fn = wlp->string[j];
		src_data = change_file_find(cp, fn);
		if (src_data)
		{
			fn_abs = change_file_path(cp, fn);
		}
		else
		{
			src_data = project_file_find(cp->pp, fn);
			assert(src_data);
			fn_abs = project_file_path(cp->pp, fn);
		}
		assert(fn_abs);

		/*
		 * figure the command execution flags
		 */
		inp = (src_data->usage == file_usage_manual_test);

		/*
		 * run the command
		 */
		exit_status = run_test_command(cp, up, fn_abs, dir, inp, bl);

		/*
		 * remember what happened
		 */
		batch_result_list_append(result, fn, exit_status);

		/*
		 * verbose progress message
		 */
		switch (exit_status)
		{
		case 1:
			if (bl)
			{
				scp = sub_context_new();
				sub_var_set_string(scp, "File_Name", fn);
				change_verbose(cp, scp, i18n("$filename baseline fail, good"));
				sub_context_delete(scp);
				result->pass_count++;
			}
			else
			{
				scp = sub_context_new();
				sub_var_set_string(scp, "File_Name", fn);
				change_verbose(cp, scp, i18n("$filename fail"));
				sub_context_delete(scp);
				result->fail_count++;
			}
			break;

		case 0:
			if (bl)
			{
				scp = sub_context_new();
				sub_var_set_string(scp, "File_Name", fn);
				change_verbose(cp, scp, i18n("$filename baseline pass, not good"));
				sub_context_delete(scp);
				result->fail_count++;
			}
			else
			{
				scp = sub_context_new();
				sub_var_set_string(scp, "File_Name", fn);
				change_verbose(cp, scp, i18n("$filename pass"));
				sub_context_delete(scp);
				result->pass_count++;
			}
			break;
		
		default:
			scp = sub_context_new();
			sub_var_set_string(scp, "File_Name", fn);
			change_verbose(cp, scp, i18n("$filename no result"));
			sub_context_delete(scp);
			result->no_result_count++;
			break;
		}
		str_free(fn_abs);

		/*
		 * don't persevere if the user asked us not to
		 */
		if (result->fail_count && !persevere)
			break;
	}

	/*
	 * all done
	 */
	trace(("return %08lX;\n", (long)result));
	trace(("}\n"));
	return result;
}
