/* 
   Affix - Bluetooth Protocol Stack for Linux
   Copyright (C) 2001 Nokia Corporation
   Original Author: Dmitry Kasatkin <dmitry.kasatkin@nokia.com>

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at your
   option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* 
   $Id: bty.h,v 1.30 2003/09/29 12:15:47 kds Exp $

   BTY - RF Virutal terminal Driver for RFCOMM

   Fixes:	Dmitry Kasatkin <dmitry.kasatkin@nokia.com>
                Imre Deak <ext-imre.deak@nokia.com>
*/		


#ifndef _BTY_H
#define _BTY_H

#define	BTY_MAGIC		0x94765768
#define BTY_MAX_TTY		8

#include <linux/tty.h>
#include <linux/serial.h>

#include <affix/bluetooth.h>

#define BTY_SNDBUF_SIZE		0x7FFF
#define BTY_RCVBUF_SIZE		0x7FFF

#define BTY_MAJOR		60
#define BTY_MINOR		0

#define DO_RESTART
#define RELEVANT_IFLAG(iflag) (iflag & (IGNBRK|BRKINT|IGNPAR|PARMRK|INPCK))

/* like async_struct */
typedef struct {
	int			magic;

	atomic_t		refcnt;

	/* RFCOMM part */
	struct sock		*sk;
	rfcomm_con		*con;

	/* Linux tty part */
	int			flags;
	struct tty_struct	*tty;

	int			line;
	int 			count;                /* open count */
	int 			blocked_open;
	wait_queue_head_t	open_wait;
	wait_queue_head_t	close_wait;
	wait_queue_head_t	delta_msr_wait;
	wait_queue_head_t	tx_wait;

	struct tasklet_struct	rx_task;
	struct tasklet_struct	tx_wakeup_task;

	unsigned short		closing_wait;
	unsigned short		close_delay;
	struct async_icount	icount;

	int			read_status_mask;
	int			ignore_status_mask;

	/* and here status information about virtual con */
	/* msr, lsr */
	__u8			msr;
	__u8			mcr;
	
} bty_struct;

struct bty_table {
	rfcomm_con	*con;
	bty_struct	*bty;
};

extern int			bty_maxdev;
extern struct bty_table		*bty_table;
extern rwlock_t			bty_lock;


/* /dev/btyXX */
int bty_open(struct tty_struct * tty, struct file * filp);
void bty_close(struct tty_struct * tty, struct file * filp);
int bty_write(struct tty_struct * tty, int from_user, const unsigned char *buf, int count);
void bty_put_char(struct tty_struct *tty, unsigned char ch);
void bty_flush_chars(struct tty_struct *tty);
int bty_write_room(struct tty_struct *tty);
int bty_chars_in_buffer(struct tty_struct *tty);
void bty_flush_buffer(struct tty_struct *tty);
int bty_ioctl(struct tty_struct *tty, struct file * file, unsigned int cmd, unsigned long arg);
void bty_throttle(struct tty_struct *tty);
void bty_unthrottle(struct tty_struct *tty);
void bty_set_termios(struct tty_struct *tty, struct termios * old_termios);
void bty_stop(struct tty_struct *tty);
void bty_start(struct tty_struct *tty);
void bty_hangup(struct tty_struct *tty);
void bty_send_xchar(struct tty_struct *tty, char ch);
void bty_break(struct tty_struct *tty, int break_state);
void bty_wait_until_sent(struct tty_struct *tty, int timeout);
int bty_read_proc(char *buf, char **start, off_t offset, int len, int *eof, void *unused);


int __init bty_init(void);
void bty_exit(void);
void __bty_destroy(bty_struct *bty);
int bty_register_con(rfcomm_con *con, int line);
int bty_unregister_con(int line);
void bty_write_to_tty(bty_struct *bty);
void bty_data_ind(rfcomm_con *con);
void bty_control_ind(rfcomm_con *con, int event);
void bty_rx_task(void *data);
void bty_tx_wakeup_task(void *data);


static inline __u32 __tty_to_line(struct tty_struct *tty)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)
	return minor(tty->device) - tty->driver.minor_start;
#else
	return tty->index;
#endif
}

static inline void bty_hold(bty_struct *bty)
{
	atomic_inc(&bty->refcnt);
}

static inline void bty_put(bty_struct *bty)
{
	if (atomic_dec_and_test(&bty->refcnt)) {
		write_lock_bh(&bty_lock);
		if (atomic_read(&bty->refcnt) == 0)
			__bty_destroy(bty);
		write_unlock_bh(&bty_lock);
	}
}

static inline rfcomm_con *__bty_getcon(int line)
{
	if (line >= 0 && line < bty_maxdev)
		return bty_table[line].con;
	return NULL;
}

static inline rfcomm_con *bty_getcon(int line)
{
	rfcomm_con	*con = NULL;
	write_lock_bh(&bty_lock);
	con = __bty_getcon(line);
	if (con)
		rfcon_hold(con);
	write_unlock_bh(&bty_lock);
	return con;
}

static inline bty_struct *__bty_get(int line)
{
	if (line >= 0 && line < bty_maxdev)
		return bty_table[line].bty;
	return NULL;
}

static inline bty_struct *bty_get(int line)
{
	bty_struct	*bty = NULL;
	write_lock_bh(&bty_lock);
	bty = __bty_get(line);
	if (bty)
		bty_hold(bty);
	write_unlock_bh(&bty_lock);
	return bty;
}


static inline int bty_disabled(bty_struct *bty)
{
	return !rfcon_get_txfc(bty->con);
	//(!con->peer_param.rtr_output || con->peer_modem.mr & RFCOMM_RTR)
}

void rpf_send_data(struct sock *sk);
struct sk_buff *rpf_wmalloc(struct sock *sk, unsigned long size, int force, int priority);


#endif
