// ---------------------------------------------------------------------------
// - Argument.hpp                                                           -
// - afnix engine - argument class definition                               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_ARGUMENT_HPP
#define  AFNIX_ARGUMENT_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

namespace afnix {

  /// The argument class defines a binding between a name and an index for
  /// the execution stack. The index is relative to the frame pointer. The 
  /// class behaves like the symbol class.
  /// @author amaury darsch

  class Argument : public Literal {
  private:
    /// the argument quark
    long   d_quark;
    /// the argument index
    long   d_index;
    /// the const flag
    bool   d_const;

  public:
    /// create a new argument with a quark and an index
    /// @param quark  the argument quark
    /// @param index  the index in the runnable stack
    Argument (const long quark, const long index);

    /// copy construct this argument
    /// @param that the argument to copy
    Argument (const Argument& that);

    /// @return the class name
    String repr (void) const;

    /// @return a clone of this argument
    Object* clone (void) const;

    /// @return a literal representation of this argument
    String toliteral (void) const;
    
    /// @return a string representation of this argument
    String tostring (void) const;

    /// set the const flag for this argument
    /// @param flag the flag to set
    void setconst (const bool flag);

    /// @return the const flag for this argument
    bool getconst (void) const;

  private:
    // make the assignment operator private
    Argument& operator = (const Argument&);

  public:
    /// set the argument object
    /// @param robj   the current runnable
    /// @param object the object to set
    void setobj (Runnable* robj, Object* object);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// set this argument as a const object
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* cdef (Runnable* robj, Nameset* nset, Object* object);

    /// set this argument with an object
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, Object* object);

    /// evaluate an object in this argument
    /// @param robj the current runnable
    /// @param nset the current nameset
    /// @return the evaluated object
    Object* eval (Runnable* robj, Nameset* nset);

    /// apply this argument with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
