// ---------------------------------------------------------------------------
// - Property.cpp                                                            -
// - standard object library - property class implementation                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Input.hpp"
#include "Stdsid.hxx"
#include "Vector.hpp"
#include "Output.hpp"
#include "Property.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a new default property

  Property::Property (void) {
    d_name = "";
    d_pval = "";
  }

  // create a new property by name

  Property::Property (const String& name) {
    d_name = name;
    d_pval = "";
  }

  // create a new property by name and value
  
  Property::Property (const String& name, const Literal& lval) { 
    d_name = name;
    d_pval = lval.tostring ();
  }
  
  // copy constructor for this property class

  Property::Property (const Property& that) {
    that.rdlock ();
    d_name = that.d_name;
    d_pval = that.d_pval;
    that.unlock ();
  }

  // return the class name

  String Property::repr (void) const {
    return "Property";
  }

  // return a clone of this property

  Object* Property::clone (void) const {
    return new Property (*this);
  }

  // return the property serial code

  t_byte Property::serialid (void) const {
    return SERIAL_PROP_ID;
  }

  // serialize this property

  void Property::wrstream (Output& os) const {
    rdlock ();
    d_name.wrstream (os);
    d_pval.wrstream (os);
    unlock ();
  }

  // deserialize this property

  void Property::rdstream (Input& is) {
    wrlock ();
    d_name.rdstream (is);
    d_pval.rdstream (is);
    unlock ();
  }

  // set a property with a property

  Property& Property::operator = (const Property& that) {
    wrlock ();
    that.rdlock ();
    d_name = that.d_name;
    d_pval = that.d_pval;
    that.unlock ();
    unlock ();
    return *this;
  }
  
  // set the property name

  void Property::setname (const String& name) {
    wrlock ();
    d_name = name;
    unlock ();
  }

  // get the property name

  String Property::getname (void) const {
    rdlock ();
    String result = d_name;
    unlock ();
    return result;
  }

  // set the property name

  void Property::setpval (const Literal& lval) {
    wrlock ();
    d_pval = lval.tostring ();
    unlock ();
  }

  // get the property name

  String Property::getpval (void) const {
    rdlock ();
    String result = d_pval;
    unlock ();
    return result;
  }

  // set the property name/value at once

  void Property::set (const String& name, const Literal& lval) {
    wrlock ();
    d_name = name;
    d_pval = lval.tostring ();
    unlock ();
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 5;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SET     = zone.intern ("set");
  static const long QUARK_SETNAME = zone.intern ("set-name");
  static const long QUARK_GETNAME = zone.intern ("get-name");
  static const long QUARK_SETPVAL = zone.intern ("set-value");
  static const long QUARK_GETPVAL = zone.intern ("get-value");

  // create a new object in a generic way

  Object* Property::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new Property;
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Property (name);
    }
    // check for 2 arguments
    if (argc == 2) {
      String   name = argv->getstring (0);
      Object*   obj = argv->get (1);
      Literal* lobj = dynamic_cast <Literal*> (obj);
      if (lobj == nilp) {
	throw Exception ("type-error", "invalid object with property",
			 Object::repr (obj));
      }
      return new Property (name, *lobj);
    }		     
    throw Exception ("argument-error", 
		     "too many argument with property constructor");
  }

  // return true if the given quark is defined

  bool Property::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Serial::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* Property::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETNAME) return new String (getname ());
      if (quark == QUARK_GETPVAL) return new String (getpval ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETNAME) {
	String name = argv->getstring (0);
	setname (name);
	return nilp;
      }
      if (quark == QUARK_SETPVAL) {
	Object*   obj = argv->get (0);
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if (lobj == nilp) {
	  throw Exception ("type-error", "invalid object with set-value",
			   Object::repr (obj));
	}
	setpval (*lobj);
	return nilp;
      }
    }
    // dispatch 2 argument
    if (argc == 2) {
      if (quark == QUARK_SET) {
	String   name = argv->getstring (0);
	Object*   obj = argv->get (1);
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if (lobj == nilp) {
	  throw Exception ("type-error", "invalid object with set-value",
			   Object::repr (obj));
	}
	set (name, *lobj);
	return nilp;
      }
    }
    // call the serial method
    return Serial::apply (robj, nset, quark, argv);
  }
}
