/**********************************************************************
 ** ErrLog class: Creates and maintains the error log of all non-fatal
 **               bootup errors and all other errors, fatal or non-fatal
 **               that exist in normal server uptime
 **    
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef ERRLOG_C
#define ERRLOG_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "logs.h"
#include "errlog.h"
#include "utils.h"
#include "timespan.h"
#include "adminflags.h"
#include "global.h"

/***********************************************************************
 ** ErrLog (constructor) - Opens the logfile by a specified name
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

ErrLog::ErrLog()
{
	valid = 0;
   if (open_log(ERRLOGNAME) <= 0)
   {
      printf("Error - could not open the errorlog\n");
      return;
   }
   fprintf(the_log, "&+W<%s>&+b ** Mud Starting up **&*\n", get_time_str());
   fflush(the_log);
   valid = 1;
}


/***********************************************************************
 ** ~ErrLog (destructor)  - empty (closing actions located in Logs)
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

ErrLog::~ErrLog()
{
      /* If the logfile is not there, return */ 
   if (the_log == NULL)
      return;

   fprintf(the_log, "&+W<%s>&+b ** Mud Closing down **&*\n", get_time_str());
   fflush(the_log);

}


/***********************************************************************
 ** log_err - logs an error in the error log
 **
 ** Parameters: err_str - the error string to print
 **             the_function - the name of the function this came from 
 **
 ** Returns:  1 if success
 **          -1 if error file is not open
 **
 ***********************************************************************/

int ErrLog::log_err(char *err_str, char *the_function)
{
   Timespan the_time;

   /* If the logfile is not there, return */ 
   if (the_log == NULL)
      return -1;

   if (mainstruct != NULL)
      mainstruct->send_all_players(
                "&+Y[&+WAn &+Rerror&+W has been logged&+Y]&*\n", NULL, 
                                                        ADMINFLAG_SEEERRORS);

   fprintf(the_log, "&+W<%s>&* %s (%s)\n", get_time_str(), err_str, 
                                                             the_function);
   fflush(the_log);
   return 1;
}


/***********************************************************************
 ** archive_errlog - archives the error log and opens a new one
 **
 ** Parameters: None
 **
 ** Returns:  1 if success
 **          -1 if error file is not open
 **
 ***********************************************************************/

int ErrLog::archive_errlog()
{
   int results; 

   fprintf(the_log, "&+W<%s>&+Y ** Archiving ErrLog **&*\n", get_time_str());
   fflush(the_log);
   
   if ((results = archive_log(ERRLOGNAME)) != -1)
   {
      if (open_log(ERRLOGNAME) <= 0)
      {
         printf("Error - could not open the errlog\n");
	 valid = 0;
         return -1;
      }
      fprintf(the_log, 
                "&+W<%s>&+Y ** Error log reopened after archive **&*\n", 
                                                          get_time_str());
      fflush(the_log);
      valid = 1;
   }
   return results;
}


/***********************************************************************
 ** invalid_format - log the standard invalid format error message
 **
 ** Parameters: objname - the object that is the source of the error
 **             areaname - the name of the area the object came from 
 **             functname - the function name the error was raised in
 **
 ***********************************************************************/

void ErrLog::invalid_format(char *objname, char *areaname, char *functname)
{
  Strings holder;

  holder.sprintf(_("Invalid format in mudobject %s@%s"), objname, areaname);
  log_err(holder.str_show(), functname);
}

/***********************************************************************
 ** invalid_attr_format - log the standard invalid format error message
 **                       for an invalid format in a specified attribute
 **
 ** Parameters: objname - the object that is the source of the error
 **             areaname - the name of the area the object came from 
 **             attr - the name of the attribute
 **             functname - the function name the error was raised in
 **
 ***********************************************************************/

void ErrLog::invalid_attr_format(char *objname, char *areaname, char *attr, 
				 char *functname)
{
  Strings holder;

  holder.sprintf(_("Invalid format in attribute %s for mudobject %s@%s"), 
		                                   attr, objname, areaname);
  log_err(holder.str_show(), functname);
}

/***********************************************************************
 ** invalid_header - log an error with an area's header
 **
 ** Parameters: areaname - the name of the area the object came from 
 **             functname - the function name the error was raised in
 **
 ***********************************************************************/

void ErrLog::invalid_header(char *areaname, char *functname)
{
  Strings holder;

  holder.sprintf(_("Invalid format in area %s header"), areaname);
  log_err(holder.str_show(), functname);
}


#endif

