//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it
//  and/or modify it under the terms of the GNU General
//  Public License as published by the Free Software
//  Foundation; either version 2 of the License, or
//  (at your option) any later version.
//==============================================

#ifndef GUI_EDITING_EDITINGINTERFACE_H
#define GUI_EDITING_EDITINGINTERFACE_H

//--------------------
//forward declarations
class QLabel;
class QImage;
class QToolButton;
class QPushButton;
class QComboBox;
class ClickableLabel;
class LayoutWidget;
class SelectionInterface;
class Subalbum;
class Photo;
class ManipulationOptions;
//--------------------

///Manipulations listed in the order in which they will
///be displayed in the drop down list
typedef enum
{  
  BW_EFFECT = 0,
  SEPIA_EFFECT,
  INVERT_EFFECT,
  EMBOSS_EFFECT,
  MOSAIC_EFFECT,
  PAINTING_EFFECT,
  //===============
  //place effects that are not release quality below
  //NUM_MANIPULATIONS to exclude them from the drop-down menu
  NUM_MANIPULATIONS,
  POINTILLISM_EFFECT,
} MANIPULATION;

#include <qwidget.h>
#include "../../backend/tools/imageTools.h"

//=====================================
/*! \brief Interface for editing photo. */
//=====================================

class EditingInterface : public QWidget
{
Q_OBJECT

public:
  ///Constructs layout
  EditingInterface(QWidget *parent=0, const char* name=0);

  ~EditingInterface();

  ///Sets the photo pointer and constructs scaled qimage's for painting
  void setPhoto(Subalbum* collection, Photo* photo);

  ///Returns a pointer to the currently selected photo
  Photo* getPhoto();
  
  //returns true if currently selected photo is revertable
  bool currentPhotoRevertable();
  
  ///reverts current photo and updates display
  void revertCurrentPhoto();
//----------------------
signals:
  void photoModified();
//----------------------    
public slots:
  void handleSelectionChanged();
  void handleAspectRatioChanged();
  void setFocus();
  //----------------------
protected:
  void keyPressEvent( QKeyEvent *e );
//----------------------
private slots:
  ///Show prev photo
  void showPrevPhoto();

  ///Show next photo
  void showNextPhoto();

  ///Show first photo
  void showFirstPhoto();
  
  ///Show last photo
  void showLastPhoto();
  
  ///Rotate image right 90 degrees
  void rotateRight();

  ///Rotate image left 90 degrees
  void rotateLeft();

  ///Flip image horizontally
  void flipHorizontal();

  ///Enter correct image tilt mode
  void startCorrectTilt();
  
  ///Finish correcting and image's tilt
  void finishCorrectTilt( QPoint p1, QPoint p2 ); 
  
  ///Flip image vertically
  void flipVertical();

  ///Aspect ratio selection changed
  void selectAspectRatio();

  ///Update recorded screen resolution and selection if necessary
  void screenResolutionChanged();
  
  ///Rotate current selection
  void rotateSelection();
  
  ///Cropped image
  void crop();

  ///Enhance image contrast
  void enhanceContrast();

  ///Improve color balance
  void colorBalance();
  
  ///Applies redeye removal
  void removeRedeye();

  ///Opens levels editor for manual histogram and brightness/contrast adjustments
  void tuneLevels();
  
  ///Opens image grain editor
  void adjustGrain();
  
  ///Effect seletion changed
  void selectEffect();
  
  ///Apply selected effect
  void applyEffect();
  
  ///Exit editing interface
  void returnAction();
  //----------------------
private:
  ///Utility method for show prev/next photos
  void showNextPrevFirstLastPhoto( Photo* newPhoto );    
    
  ///Utility method for rotation + flip slots
  void rotateFlip( TRANSFORM_CODE rotationFlipType );

  ///Finds the selected region of the image
  bool findSelection(QPoint& topLeft, QPoint& bottomRight);
  
  ///Return coordinates that select entire image
  void selectAll(QPoint& topLeft, QPoint& bottomRight);

  ///utlity function for apply effect and preview effect slots
  QImage* applyEffect(QString filename, ManipulationOptions* options=NULL);
  
  ///Applies update to image
  void applyImageUpdate(QImage* editedImage, bool resetSelection);
  ///------------
  ///Buttons for changing the shown image
  ClickableLabel *previousButton, *nextButton;
  
  ///This widget scales and displays the photo to fit the available screen space, and provides
  ///extensive support for selecting regions of the photo for editing/cropping.
  SelectionInterface* selectionInterface;

  ///Widget for selection aspect ratio to crop to
  QComboBox* aspectRatios;
  
  //Widget for selecting orientation for cropping
  QComboBox* orientations;
  
  ///Index for screen resolution, needed if this value pair changes during program executing
  int displayResolutionIndex;
  
  ///array of common aspect ratios to crop to
  QSize* aspectRatioValues;     
  double* maxDimensions;

  ///state variable indicating if the user wants to use a rotate aspect ratio
  bool selectionRotated;
  
  ///The start tilt button is disabled while a line is being selected
  QToolButton* correctTiltButton;

  ///The crop buttons is disabled when no seletion is present
  QToolButton* cropButton;
  
  ///The red eye reduction button is disabled when no selection is present
  ClickableLabel* redEyeReductionButton;
    
  ///List of effects that can be applied to photos
  QComboBox* effectsList;
  
  ///The apply effect button is disabled when no effect has been chosen
  QPushButton* applyEffectButton;

  ///Label that shows preview of effect
  QLabel* effectPreview;
  ///==========
  ///Pointer to backend collection
  Subalbum* collection;
  
  ///Pointer to backend photo
  Photo* photo;

  ///Path to scaled down version of image for fast generation of previews of effects
  QString effectPreviewImageFilename;
  
  ///Pointer to the parent layout widget
  LayoutWidget* layout;
  
  ///Dimension of photo in display coordinates
  int displayWidth, displayHeight;
  
  ///Dimension of photo in image coordinates
  int imageWidth, imageHeight; 
//----------------------
};
//======================

#endif //GUI_EDITING_EDITINGINTERFACE_H
