// ---------------------------------------------------------------------------
// - Qualified.hpp                                                           -
// - aleph engine - qualified name class definition                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_QUALIFIED_HPP
#define  ALEPH_QUALIFIED_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The qualified class is similiar to the lexical one, except that it 
  /// defines an object with a path. Each component of the qualified name is
  /// a lexical name. During construction, the qualified name is checked for
  /// adherence to this rule.
  /// @author amaury darsch

  class Qualified : public Literal {
  private:
    /// the qualified name
    String d_name;
    /// the qualified path
    long*  p_quarks;
    /// the qualified length
    long   d_length;
    /// the line number
    long   d_lnum;

  public:
    /// create an empty qualified
    Qualified (void);

    /// create a new qualified with a name
    /// @param name the qualified name
    Qualified (const String& nameK);

    /// create a new qualified with a name and a line number
    /// @param name the qualified name
    /// @param line the line number
    Qualified (const String& name, const long lnum);

    /// copy constructor for this qualified
    /// @param that the qualified to copy
    Qualified (const Qualified& that);

    /// destroy this qualified name
    ~Qualified (void);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this qualified
    String toliteral (void) const {
      return d_name;
    }

    /// @return a string representation of this qualified
    String tostring (void) const {
      return d_name;
    }

    /// @return a clone of this qualified name
    Object* clone (void) const;

    /// @return the qualified serial code
    t_byte serialid (void) const;

    /// serialize this qualified to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a qualified from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// @return the length of the qualified name path
    long length (void) const {
      return d_length;
    }

    /// @return a partial lexical name by index
    String getname (const long index) const;

    /// @return the qualified line number
    long getlnum (void) const {
      return d_lnum;
    }

    /// generate a new qualified
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// set this qualified name as a const symbol
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* cdef (Runnable* robj, Nameset* nset, Object* object);

    /// set this qualified name with an object
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, Object* object);

    /// evaluate this qualified name in the current nameset
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    Object* eval (Runnable* robj, Nameset* nset);

    /// apply this qualified with a set of arguments and a quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  
  private:
    // make the assignment operator private
    Qualified& operator = (const Qualified&);
  };
}

#endif
