// ---------------------------------------------------------------------------
// - TcpServer.cpp                                                           -
// - aleph:net library - tcp server socket implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Integer.hpp"
#include "TcpServer.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a tcp server with a port

  TcpServer::TcpServer (t_word port) {
    // bind and listen this socket
    if (bind (port) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (5) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with a port and backlog

  TcpServer::TcpServer (t_word port, const long backlog) {
    // bind and listen this socket
    if (bind (port) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (backlog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with a host and port

  TcpServer::TcpServer (const String& host, t_word port) {
    // get the host address
    Address addr (host);
    // bind and listen this socket
    if (bind (port, addr) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (5) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with an address and port

  TcpServer::TcpServer (const Address& addr, t_word port) {
    // bind and listen this socket
    if (bind (port, addr) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (5) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with a host and port and backlog

  TcpServer::TcpServer (const String& host, t_word port, const long backlog) {
    // get the host address
    Address addr (host);
    // bind and listen this socket
    if (bind (port, addr) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (backlog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // create a tcp server with an address, a port and backlog

  TcpServer::TcpServer (const Address& addr, t_word port, const long backlog) {
    // bind and listen this socket
    if (bind (port, addr) == false)
      throw Exception ("server-error", "cannot bind socket");
    if (listen (backlog) == false)
      throw Exception ("server-error", "cannot listen on socket");
  }

  // return the class name

  String TcpServer::repr (void) const {
    return "TcpServer";
  }

  // create a new tcp server in a generic way

  Object* TcpServer::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for one argument
    if (argc == 1) {
      t_word port = argv->getint (0);
      return new TcpServer (port);
    }
    // check for two arguments
    if (argc == 2) {
      Object* obj = argv->get (0);
      // check for a port
      Integer* iobj = dynamic_cast <Integer*> (obj);
      if (iobj != nilp) {
	t_word port    = iobj->tointeger ();
	long   backlog = argv->getint (1);
	return new TcpServer (port, backlog);
      }
      // check for a host
      String* sobj = dynamic_cast <String*> (obj);
      if (sobj != nilp) {
	t_word port    = argv->getint (1);
	return new TcpServer (*sobj, port);
      }
      // check for an address
      Address* aobj = dynamic_cast <Address*> (obj);
      if (aobj != nilp) {
	t_word port = argv->getint (1);
	return new TcpServer (*aobj, port);
      }
    }
    // check for three arguments
    if (argc == 3) {
      Object* obj = argv->get (0);
      // check for a host
      String* sobj = dynamic_cast <String*> (obj);
      if (sobj != nilp) {
	t_word port    = argv->getint (1);
	long   backlog = argv->getint (2);
	return new TcpServer (*sobj, port, backlog);
      }
      // check for an address
      Address* aobj = dynamic_cast <Address*> (obj);
      if (aobj != nilp) {
	t_word port    = argv->getint (1);
	long   backlog = argv->getint (2);
	return new TcpServer (*aobj, port, backlog);
      }
    }
    throw Exception ("argument-error", "invalid arguments with tcp server");
  }
}
