// ---------------------------------------------------------------------------
// - Object.cpp                                                              -
// - standard object library - base object implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Vector.hpp"
#include "Method.hpp"
#include "Lockrw.hpp"
#include "Monitor.hpp"
#include "Boolean.hpp"
#include "Finalize.hpp"
#include "Exception.hpp"
#include "cmem.hpp"

namespace aleph {

  // standard quarks object
  static const long QUARK_EQUL   = String::intern ("=");
  static const long QUARK_REPR   = String::intern ("repr");
  static const long QUARK_SHARED = String::intern ("shared-p");
  static const long QUARK_RDLOCK = String::intern ("rdlock");
  static const long QUARK_WRLOCK = String::intern ("wrlock");
  static const long QUARK_UNLOCK = String::intern ("unlock");

  // the structure for shared object
  struct s_shared {
    // the finalize flag
    bool d_final;
    // the reference count monitor
    Monitor d_mon;
    // the read-write lock
    Lockrw  d_lock;
    // simple constructor
    s_shared (void) {
      d_final = false;
    }
  };

  // the finalizer object
  static Finalize* finalizer = nilp;
  static bool      gclockfnl = false;

  // finalizer cleanup handler
  static void finalizer_cleanup (void) {
    gclockfnl = true;
    delete finalizer;
    finalizer = nilp;
  }

  // finalize an object
  static void finalize (Object* object) {
    if (gclockfnl == true) {
      delete object;
      return;
    }
    if (finalizer == nilp) finalizer = new Finalize;
    finalizer->append (object);
  }

  // the object memory allocator

  void* Object::operator new (const t_size size) {
    return c_galloc (size);
  }

  void* Object::operator new [] (const t_size size) {
    return c_galloc (size);
  }

  // the object memory deallocator

  void Object::operator delete (void* handle) {
    c_gfree (handle);
  }

  void Object::operator delete [] (void* handle) {
    c_gfree (handle);
  }

  // create a new object with a 0 reference count

  Object::Object (void) {
    p_shared = nilp;
    d_rcount = 0;
  }

  // destroy this object

  Object::~Object (void) {
    delete p_shared;
  }

  // return an object class name

  const String Object::repr (Object* object) {
    if (object == nilp) return "nil";
    return object->repr ();
  }

  // clone this object

  Object* Object::clone (void) const {
    throw Exception ("clone-error", "cannot clone object", repr ());
  }

  // make this object a shared one

  void Object::mksho (void) {
    p_shared = new s_shared;
  }

  // get a read lock for this object
  void Object::rdlock (void) const {
    if (p_shared == nilp) return;
    p_shared->d_lock.rdlock ();
  }

  // get a write lock for this object
  void Object::wrlock (void) const {
    if (p_shared == nilp) return;
    p_shared->d_lock.wrlock ();
  }

  // unlock a previous lock

  void Object::unlock (void) const {
    if (p_shared == nilp) return;
    p_shared->d_lock.unlock ();
  }

  // clear the finalizer - and lock it

  void Object::clrfnl (void) {
    finalizer_cleanup ();
  }

  // increment the object reference count

  Object* Object::iref (Object* object) {
    if (object == nilp) return nilp;
    if (object->p_shared != nilp) object->p_shared->d_mon.enter ();
    object->d_rcount++;
    if (object->p_shared != nilp) object->p_shared->d_mon.leave ();
    return object;
  }
    
  // decrement the reference count and destroy the object if null

  void Object::dref (Object* object) {
    if (object == nilp) return;

    // if not shared we delete
    if (object->p_shared == nilp) {
      if (--object->d_rcount <= 0) delete object;
      return;
    }

    // here the object is shared
    object->p_shared->d_mon.enter ();
    if (--object->d_rcount <= 0) {
      if (object->p_shared->d_final == true) {
	object->p_shared->d_mon.leave ();
	delete object;
	return;
      } else {
	object->p_shared->d_final = true;
	object->p_shared->d_mon.leave ();
	finalize (object);
	return;
      }
    }
    object->p_shared->d_mon.leave ();
  }

  // clean this object if the reference count is null
  
  void Object::cref (Object* object) {
    if (object == nilp) return;

    // if not shared we simply delete
    if (object->p_shared == nilp) {
      if (object->d_rcount <= 0) delete object;
      return;
    }

    // here the object is shared
    object->p_shared->d_mon.enter ();
    if (object->d_rcount <= 0) {
      if (object->p_shared->d_final == true) {
	object->p_shared->d_mon.leave ();
	return;
      } else {
	object->p_shared->d_final = true;
	object->p_shared->d_mon.leave ();
	finalize (object);
	return;
      }
    }
    object->p_shared->d_mon.leave ();
  }
  
  // decrement the object reference count but do not destroy if null

  void Object::tref (Object* object) {
    if (object == nilp) return;
    if (object->p_shared != nilp) object->p_shared->d_mon.enter ();
    if (object->d_rcount > 0) object->d_rcount--;
    if (object->p_shared != nilp) object->p_shared->d_mon.leave ();
  }

  // return true if the object has reference equal to 0 or 1

  bool Object::uref (Object* object) {
    if (object == nilp) return false;
    if (object->p_shared != nilp) object->p_shared->d_mon.enter ();
    bool result = (object->d_rcount <= 1);
    if (object->p_shared != nilp) object->p_shared->d_mon.leave ();
    return result;
  }

  // operate this object with another one

  Object* Object::oper (Runnable* robj, t_oper type, Object* object) {
    throw Exception ("operator-error", "invalid operator call for object",
		     repr ());
  }

  // create or set a const object to this object

  Object* Object::cdef (Runnable* robj, Nameset* nset, Object* object) {
    throw Exception ("const-error", "invalid const define with object",
		     repr ());
  }

  // create or set a const object to this object by quark

  Object* Object::cdef (Runnable* robj, Nameset* nset, const long quark,
			Object* object) {
    String mesg = "invalid const define with name ";
    mesg = mesg + String::qmap (quark);
    mesg = mesg + " from object type";
    throw Exception ("const-error", mesg, repr ());
  }

  // create or set an object to this object

  Object* Object::vdef (Runnable* robj, Nameset* nset, Object* object) {
    throw Exception ("trans-error", "invalid trans define with object",
		     repr ());
  }

  // create or set an object to this object by quark

  Object* Object::vdef (Runnable* robj, Nameset* nset, const long quark,
			Object* object) {
    String mesg = "invalid trans define with name ";
    mesg = mesg + String::qmap (quark);
    mesg = mesg + " from object type";
    throw Exception ("trans-error", mesg, repr ());
  }

  // evaluate an object in the current nameset

  Object* Object::eval (Runnable* robj, Nameset* nset) {
    return this;
  }

  // evaluate an object in the current nameset by quark

  Object* Object::eval (Runnable* robj, Nameset* nset, const long quark) {
    return new Method (quark, this);
  }

  // apply an object with a set of arguments

  Object* Object::apply (Runnable* robj, Nameset* nset, Cons* args) {
    throw Exception ("apply-error", "invalid call to apply method with object",
		     repr ());
  }

  // apply an object by quark with a set of arguments

  Object* Object::apply (Runnable* robj, Nameset* nset, const long quark,
			 Cons* args) {
    Vector* argv   = Vector::eval (robj, nset, args);
    Object* result = nilp;
    try {
      result = apply (robj, nset, quark, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    delete argv;
    return result;
  }

  // apply an object by object with a set of arguments

  Object* Object::apply (Runnable* robj, Nameset* nset, Object* object,
			 Cons* args) {
    String mesg = "invalid call to apply with object ";
    mesg = mesg + Object::repr (object);
    mesg = mesg + " from object type";
    throw Exception ("apply-error", mesg, repr ());
  }

  // apply an object by name with a vector of arguments

  Object* Object::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 arguments
    if (argc == 0) {
      if (quark == QUARK_REPR)   return new String  (repr ());
      if (quark == QUARK_SHARED) return new Boolean (issho ());
      if (quark == QUARK_RDLOCK) {
	rdlock ();
	return this;
      }
      if (quark == QUARK_WRLOCK) {
	wrlock ();
	return this;
      }
      if (quark == QUARK_UNLOCK) {
	unlock ();
	return this;
      }
    }

    // dispatch 1 argument
    if ((argc == 1) && (quark == QUARK_EQUL)) {
      Object* obj = argv->get (0);
      return this->vdef (robj, nset, obj);
    }

    // no way - error
    String mesg = "invalid call to apply with name ";
    mesg = mesg + String::qmap (quark);
    mesg = mesg + " from object type";
    throw Exception ("apply-error", mesg, repr ());
  }
}
