// ---------------------------------------------------------------------------
// - OutputFile.cpp                                                          -
// - standard object library - output file class implementation              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"
#include "OutputFile.hpp"
#include "csio.hpp"
#include "cerr.hpp"

namespace aleph {

  // the output file supported quarks
  static const long QUARK_CLOSE   = String::intern ("close");
  static const long QUARK_GETNAME = String::intern ("get-name");

  // create a new file output stream by name. 

  OutputFile::OutputFile (const String& name) {  
    // copy file name and check
    d_name = name;
    if (name.length () == 0) throw Exception ("name-error", "null file name");
    // try to open the file
    char* fname = name.tochar ();
    if ((d_sid = c_openw (fname, true, false)) < 0) {
      delete [] fname;
      throw Exception ("open-error","cannot open file for writing", name);
    }
    delete [] fname;
  }

  // create a new file output stream by name and flags.

  OutputFile::OutputFile (const String& name, const bool tflag, 
			  const bool aflag) {  
    // copy file name and check
    d_name = name;
    if (name.length () == 0) throw Exception ("name-error", "null file name");
    // try to open the file
    char* fname = name.tochar ();
    if ((d_sid = c_openw (fname, tflag, aflag)) < 0) {
      delete [] fname;
      throw Exception ("open-error","cannot open file for writing", name);
    }
    delete [] fname;
  }

  // destroy this class by closing this file

  OutputFile::~OutputFile (void) {
    close ();
  }

  // return the class name

  String OutputFile::repr (void) const {
    return "OutputFile";
  }

  // return the stream descriptor

  int OutputFile::getsid (void) const {
    return d_sid;
  }

  // write one character to the output stream
  
  void OutputFile::write (const char value) {
    wrlock ();
    // write one character
    long count = c_write (d_sid, &value,1);
    if (count < 0) {
      unlock ();
      throw Exception ("write-error", c_errmsg (count));
    }
    unlock ();
  }

  // write a character string to the output stream

  void OutputFile::write (const char* value) {
    wrlock ();
    long size = String::length (value);
    if (size == 0) return;
    // write the string
    long count = c_write (d_sid, value,size);
    if (count < 0) {
      unlock ();
      throw Exception ("write-error", c_errmsg (count));
    }
    unlock ();
  }

  // return the file name associated with this stream
  
  String OutputFile::getname (void) const {
    return d_name;
  }

  // close this output file
  
  bool OutputFile::close (void) {
    wrlock ();
    if ((d_sid == -1) || (Object::uref (this) == false)) {
      unlock ();
      return true;
    }
    if (c_close (d_sid) == false) {
      unlock ();
      return false;
    }
    d_sid = -1;
    unlock ();
    return true;
  }

  // create a new output file in a generic way

  Object* OutputFile::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new OutputFile (name);
    }
    // check for 3 arguments
    if (argc == 3) {
      String name  = argv->getstring (0);
      bool   tflag = argv->getbool (1);
      bool   aflag = argv->getbool (2);
      return new OutputFile (name, tflag, aflag);
    }
    throw Exception ("argument-error", 
		     "invalid arguments with with output file"); 
  }

  // apply this output file with a set of arguments and a quark

  Object* OutputFile::apply (Runnable* robj, Nameset* nset, const long quark,
			     Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETNAME) return new String (d_name);
      if (quark == QUARK_CLOSE)   return new Boolean (close ());
    }

    // call the output method
    return Output::apply (robj, nset, quark, argv);
  }
}
