
#define LOCAL_DEBUG
#include "debug.h"

#include "meta.h"
#include "con.h"
#include "job.h"
#include "header.h"
#include "dlcon.h"
#include "acbuf.h"

#include <sys/select.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <iostream>

using namespace MYSTD;

con::con(int fdId, const char *c) :
	m_confd(fdId),
    m_bStopActivity(false),
    m_pDlClient(NULL),
    m_pTmpHead(NULL),
    m_nLastIn(0),
    m_nLastOut(0)
    //,m_bNoLogs(false)
    
{
	if(c) // if NULL, pick up later when sent by the wrapper
		m_sClientHost=c;
	
    ldbg("Creating con " << fdId << " for " << c);

    
#ifdef KILLABLE
    wakepipe[0]=wakepipe[1]=-1;
    
    if(pipe(wakepipe) == 0) {
    	set_nb(wakepipe[0]);
    	set_nb(wakepipe[1]);
    }
    else
    	m_bStopActivity=true;
#endif
    
};

#ifdef KILLABLE
void con::SignalStop() {
    setLockGuard;
    m_bStopActivity=true;
    POKE(wakepipe[1]); // for select
    notifyAll();
}
#endif

void con::ShutDown()
{
	// no double calls
	if(m_confd>=0) 
	{
		shutdown(m_confd, SHUT_RDWR);
		close(m_confd);
		m_confd=-1;
	}
}

//void DumpItems();

con::~con() {
	ldbg("Destroying connection...");
    
#ifdef KILLABLE
	if (wakepipe[0]>=0)
		close(wakepipe[0]);

	if (wakepipe[1]>=0)
		close(wakepipe[1]);
#endif
	
	HandleStats(NULL);
	
	MYSTD::list<job*>::iterator jit;
	for (jit=m_jobs2send.begin(); jit!=m_jobs2send.end(); jit++)
		delete *jit;

    if(m_pDlClient) 
    {
    	m_pDlClient->SignalStop();
    	pthread_join(m_dlerthr, NULL);
    	
    	delete m_pDlClient;
    	m_pDlClient=NULL;
    }
    
    if(m_pTmpHead)
    {
    	delete m_pTmpHead;
    	m_pTmpHead=NULL;
    }
    				
    aclog::flush();
}

void con::WorkLoop() {
    
	signal(SIGPIPE, SIG_IGN);
	
    acbuf inBuf;
    inBuf.init(32*1024);
    
#ifdef KILLABLE
    int maxfd=MYSTD::max(wakepipe[0], m_confd);
#else
    int maxfd=m_confd;
#endif
    
    while(!m_bStopActivity) {
        fd_set rfds, wfds;
        FD_ZERO(&wfds);
        FD_ZERO(&rfds);
        
        FD_SET(m_confd, &rfds);
        if(inBuf.freecapa()==0)
        	return; // shouldn't even get here
        
        job *pjSender(NULL);
        
        
#ifdef KILLABLE
        // prepare to send data or to be notified by helper
        FD_SET(wakepipe[0], &rfds);
#endif

        if ( !m_jobs2send.empty())
		{
			pjSender=m_jobs2send.front();
			FD_SET(m_confd, &wfds);
		}
		
        
        ldbg("select con");

        struct timeval tv;
        tv.tv_sec = 90;
        tv.tv_usec = 0;
        int ready = select(maxfd+1, &rfds, &wfds, NULL, &tv);
        
        if(ready == 0)
        {
        	USRDBG(5, "Timeout occured, apt client disappeared silently?");
        	return;
        }
		else if (ready<0)
		{
			if (EINTR == errno)
				continue;
			
			ldbg("select error in con, errno: " << errno);
			return; // FIXME: good error message?
		}
        
        ldbg("select con back");

        if(FD_ISSET(m_confd, &rfds)) {
            int n=inBuf.sysread(m_confd);
            ldbg("got data: " << n <<", inbuf size: "<< inBuf.size());
            if(n<=0) // error, incoming junk overflow or closed connection
            {
              if(n==-EAGAIN)
                continue;
              else
                return;
            }
        }

        // split new data into requests
        while(inBuf.size()>0) {
        	MYTRY
        	{
				if(!m_pTmpHead)
					m_pTmpHead = new header();
				if(!m_pTmpHead)
					return; // no resources? whatever
				
				int nConsumed=m_pTmpHead->LoadFromBuf(inBuf.rptr(), inBuf.size());
				//ldbg("header parsed how? " << nConsumed);
				if(nConsumed==0)
				{ // Either not enough data received, or buffer full; make space and retry
					inBuf.move();
					break;
				}
				if(nConsumed<0)
				{
					ldbg("Bad request");
					return;
				}
				if (nConsumed>0)
					inBuf.drop(nConsumed);
				
				if (m_sClientHost.empty()) // may come from wrapper... MUST identify itself
				{
					if(m_pTmpHead->h[header::XORIG] && *(m_pTmpHead->h[header::XORIG]))
					{
						m_sClientHost=m_pTmpHead->h[header::XORIG];
						continue; // OK
					}
					else
						return;
				}

				ldbg("Parsed REQUEST:" << m_pTmpHead->frontLine);
				ldbg("Rest: " << inBuf.size());
				job * j = new job(m_pTmpHead, this);
				m_pTmpHead=NULL; // owned by job
				j->PrepareDownload();
				m_jobs2send.push_back(j);
			}
        	MYCATCH(bad_alloc)
        	{
        		return;
        	}
        }
        
        if(inBuf.freecapa()==0)
        	return; // cannot happen unless being attacked

#ifdef KILLABLE
        if(FD_ISSET(wakepipe[0], &rfds))
		{
			//ldbg("Aufgeweckt, outbuf:" << outBuf);
			int tmp;
			while(read(wakepipe[0], &tmp, 1) > 0);
			continue;
		}
#endif

		if(FD_ISSET(m_confd, &wfds) && pjSender)
		{
			ldbg("Sending data of " << pjSender);
			switch(pjSender->SendData(m_confd))
			{
				case(R_DISCON):
				{
					ldbg("Disconnect advise received, stoping connection");
					return;
				}
				case(R_DONE):
				{
					m_jobs2send.pop_front(); 
					HandleStats(pjSender);
						
					delete pjSender;
					pjSender=NULL;
		
					ldbg("Remaining jobs to send: " << m_jobs2send.size());
				}
				case(R_AGAIN):
				default:
					break;
			}
        }
	}
}

void * _StartDownloader(void *pVoidDler)
{
	static_cast<dlcon*>(pVoidDler) -> WorkLoop();
	return NULL;
}

bool con::SetupDownloader()
{

	if (m_pDlClient)
		return true;

	MYTRY
	{
		m_pDlClient=new dlcon;
		if(!m_pDlClient)
			return false;
	}
	MYCATCH(MYSTD::bad_alloc)
	{
		return false;
	}

	if (0==pthread_create(&m_dlerthr, NULL, _StartDownloader,
			(void *)m_pDlClient))
	{
		return true;
	}
	delete m_pDlClient;
	m_pDlClient=NULL;
	return false;
}

inline void con::WriteAndDropStats()
{
	if (m_nLastIn>0)
		aclog::transfer(true, m_nLastIn, m_sClientHost.c_str(), m_sLastFile.c_str());
	
	if(m_nLastOut>0)
		aclog::transfer(false, m_nLastOut, m_sClientHost.c_str(), m_sLastFile.c_str());
	
	m_nLastIn=m_nLastOut=0;
	m_sLastFile.clear();
}

void con::HandleStats(job *j)
{
	if(j)
	{
		off_t nNewIn, nNewOut;
		const string & sFile=j->TakeCounts(nNewIn, nNewOut);
		if(sFile==m_sLastFile)
		{
			m_nLastIn+=nNewIn;
			m_nLastOut+=nNewOut;
			cerr << "new in: " << m_nLastIn << " und out: " << m_nLastOut <<endl;
			return;
		}
		else
		{
			WriteAndDropStats();
			m_sLastFile=sFile;
			m_nLastIn=nNewIn;
			m_nLastOut=nNewOut;
		}
	}
	else // just flush
		WriteAndDropStats();
}

