#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Tests the debconf forwarding"""

import logging
import os
import subprocess
import tempfile
import unittest

import debian.deb822
import gobject

import sys
sys.path.insert(0, "../..")

DEBUG=False

from aptdaemon.debconf import DebconfProxy

class DebconfTestBasic(unittest.TestCase):

    def _stop(self):
        self.proxy.stop()
        self.loop.quit()

    def setUp(self):
        self.loop = gobject.MainLoop()
        self.debconf_socket_path = tempfile.mktemp(prefix="debconf-socket-")
        self._set_input_value()
        self.proxy = DebconfProxy("editor", self.debconf_socket_path)
        self.proxy.start()

    def _set_input_value(self, template="aptdaemon/test", value="lalelu"):

        os.environ["EDITOR"] = "sed -ie 's/\\(%s=\\).*/\\1\\\"%s\\\"/i'" % \
                               (template.replace("/", "\\/"), value)

    def _spawn_config_script(self, config_db_path,
                             command=["debconf/aptdaemon.config"]):
        env = {}
        env["DEBCONF_DB_REPLACE"] = "File{%s}" % config_db_path
        env["DEBIAN_FRONTEND"] = "passthrough"
        env["DEBCONF_PIPE"] = self.debconf_socket_path
        if DEBUG:
            env["DEBCONF_DEBUG"] = ".*"
        env_str = ["%s=%s" % (key, val) for key, val in env.iteritems()]

        pid, _, _, _ = gobject.spawn_async(command, envp=env_str,
                                          flags=gobject.SPAWN_DO_NOT_REAP_CHILD)
        return pid

    def testBasic(self):
        def config_done(pid, cond):
            self.assertEquals(cond, 0,
                              "Config script failed: %s" % os.WEXITSTATUS(cond))
            self._stop()
        debconf_db_path = tempfile.mktemp(suffix=".dat", prefix="config-basic-")
        pid = self._spawn_config_script(debconf_db_path)
        gobject.child_watch_add(pid, config_done)
        self.loop.run()
        # Check the results
        self._check_value(debconf_db_path)

    def testSerial(self):
        """Run several config scripts in a row."""
        def config_done(pid, cond):
            self.assertEquals(cond, 0,
                              "Config script failed: %s" % os.WEXITSTATUS(cond))
            self.config_scripts -= 1
            if self.config_scripts <= 0:
                self._stop()
            else:
                pid = self._spawn_config_script(debconf_db_path)
                gobject.child_watch_add(pid, config_done)
        debconf_db_path = tempfile.mktemp(suffix=".dat", prefix="config-row-")
        self.config_scripts = 10
        pid = self._spawn_config_script(debconf_db_path)
        gobject.child_watch_add(pid, config_done)
        self.loop.run()
        # Check the results
        self._check_value(debconf_db_path)

    def testRace(self):
        def config_done(pid, cond):
            self.assertEquals(cond, 0,
                              "Config script failed: %s" % os.WEXITSTATUS(cond))
            self.workers -= 1
            if self.workers <= 0:
                self._stop()
        debconf_dbs = []
        self.workers = 0
        for i in range(10):
            debconf_db_path = tempfile.mktemp(suffix=".dat",
                                              prefix="config-race-")
            pid = self._spawn_config_script(debconf_db_path)
            gobject.child_watch_add(pid, config_done)
            debconf_dbs.append(debconf_db_path)
            self.workers += 1
        self.loop.run()
        # Check the results
        for db_path in debconf_dbs:
            self._check_value(db_path)

    def _check_value(self, db_path, template=None, value="lalelu"):
        db_file = open(db_path)
        for section in debian.deb822.Deb822.iter_paragraphs(db_file):
            if template == section["Template"] or template is None:
                self.assertEquals(section["Value"], value)
                return
        db_file.close()
        os.remove(db_path)
        self.fail("Database doesn't contain any matching value or template")

    def tearDown(self):
        os.remove(self.debconf_socket_path)
        self.proxy = None
        self.loop.quit()
        self.loop = None


if __name__ == "__main__":
    if DEBUG:
        logging.basicConfig(level=logging.DEBUG)
    unittest.main()

# vim: ts=4 et sts=4
