# 
# build-by-patching.sh - get and apply the next patch                      
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "get and apply a patch to a source tree\\n"
		printf "usage: build-by-patching [options] revision\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf "Read the patch set for REVISION from the attached archive.\\n"
		printf "Apply that patch set to the source directory in the current\\n"
		printf "directory.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=

while test $# -ne 0 ; do

  case "$1" in 


    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "build-by-patching: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    --)			shift
			break
			;;

    -*)			printf "build-by-patching: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done


################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: build-by-patching [options] revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

revision="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

source="`pwd`"

larch valid-package-name -l -e build-by-patching -- "$revision"

category=`larch parse-package-name -b $revision`
branch=`larch parse-package-name $revision`
vsn=`larch parse-package-name -v $revision`
lvl=`larch parse-package-name -l $revision`

################################################################
# Greetings
# 
# This is an internal command.
# 

if test ! -z "$verbose" ; then 
  ARCH__OUTLINE_DEPTH="$ARCH__OUTLINE_DEPTH*"
  larch heading "build-by-patching\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent 
  larch heading --sub "build-by-patching start time: %s\\n" "`date`"
fi

if test ! -z "$report" ; then
  larch heading --sub "patching for revision %s\\n" $branch--$vsn--$lvl
fi


################################################################
# Obtain the patch
# 

wftp-home
wftp-cd $category/$branch/$branch--$vsn/$lvl

here="`pwd`"

bail()
{
  rm -rf "$here/,,patch.$branch--$vsn--$lvl"
  if test ! -z "$verbose" ; then
    larch heading --sub "build-by-patching finish time: %s\\n" "`date`"
  fi
  exit 1
}

trap "printf \"build-by-patching: interrupted -- cleaning up\\n\" 1>&2 ; bail" INT

mkdir ,,patch.$branch--$vsn--$lvl
cd ,,patch.$branch--$vsn--$lvl

if test ! -z "$verbose" ; then
  larch heading --sub "start retrieving patch from archive: %s\\n" "`date`"
fi

if ! wftp-get $branch--$vsn--$lvl.patches.tar.gz > $branch--$vsn--$lvl.patches.tar.gz ; then
  rm -f $branch--$vsn--$lvl.patches.tar.gz
  cd .. 
  rmdir ,,patch.$branch--$vsn--$lvl
  printf "\\n" 1>&2
  printf "build-by-patching: unable to retrieve patch\\n" 1>&2
  printf "  patch: $branch--%s--%s.patches.tar.gz\\n" "$vsn" "$lvl" 1>&2
  printf "\\n" 1>&2
  bail
fi

if test ! -z "$verbose" ; then
  larch heading --sub "finish retrieving patch from archive: %s\\n" "`date`"
fi

tar -m -zxf $branch--$vsn--$lvl.patches.tar.gz 

if ! larch nested larch dopatch $debug_opt --silent $report_opt $verbose_opt --delete-removed $branch--$vsn--$lvl.patches "$source" ; then
  printf "\\n" 1>&2
  printf "build-by-patching: INTERNAL ERROR! patch did not apply cleanly!\n" 1>&2
  printf "\\n" 1>&2
  bail
fi


cd ..
rm -rf ,,patch.$branch--$vsn--$lvl

if test ! -z "$verbose" ; then
  larch heading --sub "build-by-patching finish time: %s\\n" "`date`"
fi
