# 
# finish-branch.sh: make a project tree a new branch
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "commit project tree as a new branch\\n"
                printf "usage: finish-branch [options] [dir]\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
                printf "Commit DIR as the base revision of a new branch.\\n"
		printf "\\n"
		printf "This is usually preceeded by \"larch prepare-branch\" to create the\\n"
		printf "project tree DIR.\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=
__restart=

while test $# -ne 0 ; do

  case "$1" in 

    ----restart) shift
    		__restart=----restart
		;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "finish-branch: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "finish-branch: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "finish-branch: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;
			
    -*)                 printf "finish-branch: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: finish-branch [options] [dir]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -gt 0 ; then
  dir="$1"
  shift
else
  dir="."
fi

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"
wdroot="`larch tree-root --accurate`"

version="`larch tree-version`"

archive="`larch parse-package-name --arch \"$version\"`"
category="`larch parse-package-name --basename \"$version\"`"
branch="`larch parse-package-name --package \"$version\"`"
version="`larch parse-package-name --package-version \"$version\"`"

old_revision="`cat \"{arch}/++branch-revision\"`"

larch valid-package-name -e finish-branch --archive --patch-level "$old_revision"

################################################################
# Greetings
# 

if test "(" -z "$__restart" -a ! -z "$quiet" ")" -o ! -z "$report" ; then
  larch heading "finish-branch\\n"
  printf "arguments: %s\\n"  "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "finish-branch start time: %s\\n" "`date`"
  larch heading --sub "archive: %s\\n" "$archive"
  larch heading --sub "version: %s\\n" "$spec"
  larch heading --sub "branched from: %s\\n" "$old_revision"
  larch heading --sub "project tree: %s\\n" "$wdroot"
fi



################################################################
# Ensure that We Have an Archive Connection 
# 

if test "$WITHARCHIVE" != "$archive" ; then

  if test ! -z "$quiet" ; then
    larch heading --sub "restarting with connection to archive\\n"
  fi

  exec larch with-archive -A "$archive"  \
	    larch finish-branch \
		  $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
		  ----restart \
		  "$wdroot"

fi



################################################################
# Do It
# 

if test ! -z "$quiet" ; then
  larch heading --sub "committing new branch point\\n"
fi

wftp-home
if ! wftp-cd "$category" 2> /dev/null ; then
  larch make-category "$archive/$category"
fi

wftp-home
if ! wftp-cd "$category/$branch" 2> /dev/null ; then
  larch make-branch "$archive/$branch"
fi

wftp-home
if ! wftp-cd "$category/$branch/$version" 2> /dev/null ; then
  larch make-version "$archive/$version"
fi


larch nested --sub larch commit --no-lint \
		 	      $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
			      --dir "$wdroot" \
			      --continuation "$old_revision" "$archive/$version"


rm -f "$wdroot/{arch}/++branch-revision"

if test  ! -z "$quiet" ; then
  larch heading --sub "finish-branch finish time: %s\\n" "`date`"
fi


# tag: Tom Lord Mon Dec 10 22:10:31 2001 (branching-and-merging/finish-branch.sh)
#
