# build-config.sh: instantiate a multi-project tree
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "instantiate a multi-project tree\\n"
		printf "usage: build-config [options] config-name\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf " -d --dir DIR                  cd to DIR first\\n"
		printf " --config-dir CFG              get config from directory CFG\\n"
		printf "\\n"
		printf "Build the named configuration.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

dir=.
config_dir=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=

while test $# -ne 0 ; do

  case "$1" in 

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "build-config: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -d|--dir)		shift
    			if test $# -eq 0 ; then
			  printf "build-config: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir=$1
			shift
			;;

    --config-dir)       shift
			if test $# -eq 0 ; then
			  printf "build-config: --config-dir requires an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			config_dir="$1"
			shift
			;;

    --)			shift
    			break
			;;

    -*)			printf "build-config: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: build-config [options] name\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

name="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"

wdroot="`larch tree-root --accurate`"
cd "$wdroot"

config_file="`larch indicated-config-file -e build-config --config-dir \"$config_dir\" --dir \"$wdroot\" --exists \"$name\"`"

################################################################
# Greetings
# 

if test ! -z "$quiet" ; then
  larch heading "build-config\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "build-config start time: %s\\n" "`date`"
  larch heading --sub "project tree: %s\\n" "$wdroot"
  larch heading --sub "configuration: %s\\n" "$name"
  cat "$config_file" | larch body-indent --sub
fi

################################################################
# Preserve Conflicting Directories
# 

conflicts=

cd "$wdroot"

for d in `cat "$config_file" | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces | awk '{print $1}' | sort -r ` ; do
  if test -e "$d" ; then
    conflicts="$conflicts $d"
  fi
done


if test ! -z "$conflicts" ; then

  if test ! -z "$quiet" ; then
     larch heading --sub "preserving conflicting directories\\n"
  fi

  for d in $conflicts ; do
    cd "`dirname \"$d\"`"
    n="`basename \"$d\"`"
    saved="++saved.$n.`date +%Y-%m-%d`.$$"
    mv "$n" "$saved"

    if test ! -z "$quiet" ; then
      printf "%s => %s\\n" "$d" "$saved" | larch body-indent --sub --nonl
    fi
  done

  if test ! -z "$quiet" ; then
    echo | larch body-indent --sub 
  fi

fi


################################################################
# Get Requested Directories
# 

cd "$wdroot"

cat "$config_file" \
| larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
| sort -k 1 \
| awk -v silent_opt="$silent_opt" \
      -v quiet_opt="$quiet_opt" \
      -v report_opt="$report_opt" \
      -v debug_opt="$debug_opt" \
      'BEGIN { print "set -e" } \
       {
          printf("larch nested --sub larch get %s %s %s %s -- %s %s\n", silent_opt, quiet_opt, report_opt, debug_opt, $2, $1);
       }' \
| $ARCH_SHELL

# tag: Tom Lord Sat Dec 29 23:20:33 2001 (configurations/build-configuration.sh)
#
