# config-history.sh: report the history of a configuration
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "report the history of a configuration\\n"
		printf "usage: config-history [options] config-name [archive/]config-package"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf " -r --reverse                  list revisions from newest to oldest\\n"
		printf "\\n"
		printf "Report the history of a particular configuration in a\\n"
		printf "particular branch or version.\\n"
		printf "\\n"
		printf "The second parameter, CONFIG-PACKAGE, may be a category,\\n"
		printf "branch, version, or revision of a project tree which contains\\n"
		printf "a config directory (see \"larch build-config --help\").\\n"
		printf "\\n"
		printf "For all indicated revisions of the CONFIG-PACKAGE (e.g., all in\\n"
		printf "the category, or all in a particular branch), a series of output\\n"
		printf "lines are produced, each having the format:\\n"
		printf "\\n"
		printf "	\"%%s\\t%%s\" \$config_revision \$package_revision \\n"
		printf "\\n"
		printf "where \"\$package_revision\" is a revision spec of a nested project\\n"
		printf "tree from the config file \"\$config_revision\" of CONFIG-PACKAGE.\\n"
		printf "\\n"
		printf "One additional line is produced with the format:\\n"
		printf "\\n"
		printf "	\"%%s\\t%%s\" \$config_revision \$package_revision \\n"
		printf "\\n"
		printf "Note that this command obtains its data from your revision library,\\n"
		printf "not from the archive directly.  (See \"larch my-revision-library --help.)\"\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
reverse=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "config-history: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "config-history: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;

    -r|--reverse)	shift
    			reverse=-r
			;;

    --)			shift
    			break
			;;

    -*)			printf "config-history: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 2 ; then
  printf "usage: config-history [options] config-name [archive/]config-package\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

config_name="$1"
shift

config_package="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

larch my-revision-library -e config-history > /dev/null

larch valid-package-name -e config-history --basename --tolerant "$config_package"


archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$config_package\"`"
category="`larch parse-package-name --basename \"$config_package\"`"

if larch valid-package-name --tolerant "$config_package" ; then
  branch="`larch parse-package-name \"$config_package\"`"
else
  branch=
fi

if larch valid-package-name --vsn --tolerant "$config_package" ; then
  version="`larch parse-package-name --package-version \"$config_package\"`"
else
  version=
fi

if larch valid-package-name --lvl "$config_package" ; then
  revision="`larch parse-package-name --non-arch \"$config_package\"`"
else
  revision=
fi

larch valid-config-name -e config-history "$config_name"

################################################################
# Iterate Over Selected Revisions
#

if test -z $branch ; then 
  branches="`larch library-branches \"$archive/$category\"`"
else
  branches="$branch"
fi

for branch in $branches ; do

  if test -z "$version" ; then 
    versions="`larch library-versions $reverse \"$archive/$branch\"`"
  else
    versions="$version"
  fi

  for version in $versions ; do

    if test -z "$revision" ; then 
      revisions="`larch library-revisions --full $reverse \"$archive/$version\"`"
    else
      revisions="$revision"
    fi

    for revision in $revisions ; do

      printf "%s\\t%s\\n" "$revision" "$archive/$revision" 

      if larch library-file --silent "configs/$config_name" "$archive/$revision" > /dev/null ; then
	larch cat-library-file  "configs/$config_name" "$archive/$revision" \
	| larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
	| awk -v revision="$revision" '{print revision "\t" $2}'
	
      fi

    done


  done


done




# tag: Tom Lord Sat Jan 26 18:43:39 2002 (configurations/config-history.sh)
#
