# library-log.sh: print a log message from the library
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "print a log message from the library\\n"
                printf "usage: library-log [options] [archive/]revision\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
                printf " --silent                      suppress error messages for missing\\n"
                printf "                                log messages (exit status only)\\n"
                printf "\\n"
		printf "Print the log message for REVISION from the library.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
silent=

while test $# -ne 0 ; do

  case "$1" in 

    --silent)		shift
    			silent=--silent
			;;

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "library-log: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "library-log: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -*)                 printf "library-log: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: library-log [options] [archive/]revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

rvnspec="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

library="`larch my-revision-library $silent -e library-log`"

archive_revision=`larch indicated-revision -e library-log -R "$archroot" -A "$archive" "$rvnspec"`

archive=${archive_revision%%/*}
revision=${archive_revision#*/}
category="`larch parse-package-name --basename \"$revision\"`"
branch="`larch parse-package-name \"$revision\"`"
version="`larch parse-package-name --package-version \"$revision\"`"

################################################################
# Do It
# 

if test -d "$library/$archive/$category/$branch/$version/$revision" ; then

  cat "$library/$archive/$category/$branch/$version/$revision/,,patch-set/=log.txt"
  exit 0
    
else

  if test -z "$silent" ; then
    printf "\\n" 1>&2
    printf "%s: revision not in library\\n" "$errname" 1>&2
    printf "  archive: %s\\n" "$archive" 1>&2
    printf "  revision: %s\\n" "$revision" 1>&2
    printf "\\n" 1>&2
  fi

  exit 1

fi  

# tag: Tom Lord Sun Jan 27 07:09:40 2002 (library/library-log.sh)
#
