# 
# pristines.sh: list the pristine trees in a project tree
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "list the pristine trees in a project tree\\n"
		printf "usage: pristines [options] [[archive/]limit]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -D --dir DIR                  cd to DIR first\\n"
		printf " -l --long                     long format listing\\n"
		printf " -r --reverse                  reverse sort order\\n"
		printf " --unlocked                    list only unlocked pristines\\n"
		printf " --locked                      list only locked pristines\\n"
		printf "\\n"
		printf "Print the list of pristine revisions cached in project tree\\n"
		printf "DIR (or the current directory)."
		printf "\\n"
		printf "Cached revisions are printed in fully qualified form:\\n"    
		printf "\\n"
		printf "	\"%s/%s\\n\"  $archive $revision\\n"
		printf "\\n"
		printf "It is necessary to have a cached copy of a prisine tree\\n"
		printf "in order to compute differences between a project tree\\n"
		printf "and that other revision.  See also \"larch get-pristine --help\"\\n"
		printf "and \"larch remove-pristine --help\".\\n"
		printf "\\n"
		printf "If a LIMIT parameter is specified, only show pristines from\\n"
		printf "the indicated ARCHIVE, CATEGORY, BRANCH, VERSION or PATCH LEVEL\\n"
		printf "\\n"
		printf "In contrast to the usual rule, the ARCHIVE does not have a default\\n"
		printf "value.  So:\\n"
		printf "\\n"
		printf "	larch pristines joe.hacker@gnu.org--archive/foo\\n"
		printf "\\n"
		printf "lists all pristines for any version in the category \"foo\",\\n"
		printf "in the archive \"joe.hacker@gnu.org--archive\", while:\\n"
		printf "\\n"
		printf "	larch pristines foo\\n"
		printf "\\n"
		printf "lists pristine trees for all versions in category \"foo\", regardless\\n"
		printf "or archive.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

long=
reverse=
dir="."
lock_limit=

while test $# -ne 0 ; do

  case "$1" in 

    --locked)		shift
    			lock_limit=locked
			;;

    --unlocked)		shift
    			lock_limit=unlocked
			;;

    -l|--long)          shift
    			long=--long
			;;

    -r|--reverse)       shift
    			reverse=-r
			;;

    -D|--dir)		shift
    			if test $# -eq 0 ; then
			  printf "pristines: -D and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    --)			shift
			break
			;;


    -*)			printf "pristines: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: pristines [options] [[archive/]limit]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -ne 0 ; then
  limit="$1"
else
  limit=
fi

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"

cd "$dir"
wdroot="`larch tree-root`"

limit_archive=
limit_category=
limit_branch=
limit_version=
limit_revision=

if test ! -z "$limit" ; then

  larch valid-package-name -e pristines -t "$limit"

  if larch valid-package-name -t --archive "$limit" ; then
    limit_archive="`larch parse-package-name --arch \"$limit\"`"
  fi

  limit_category="`larch parse-package-name --basename \"$limit\"`"

  # BZZZT  limit_branch="`larch parse-package-name --package \"$limit\"`"

  if larch valid-package-name -t --vsn "$limit" ; then
    limit_version="`larch parse-package-name --package-version \"$limit\"`"
  fi

  if larch valid-package-name -t --patch-level "$limit" ; then
    limit_revision="`larch parse-package-name --non-arch \"$limit\"`"
  fi

fi


################################################################
# Print the list of pristine trees as fully-qualified branch names.
# 

cd "$wdroot/{arch}"

if test -z "$long" ; then
  sort_key="-k 1,1${reverse#-} -k 2,2${reverse#-}n"
else
  sort_key="-b -k 2,2${reverse#-} -k 3,3${reverse#-}n"
fi

( if test -d ++pristine-trees ; then

    for state in locked unlocked ; do

      if test ! -z "$lock_limit" -a "$state" != "$lock_limit" ; then
        continue
      fi

      if test ! -d "$wdroot/{arch}/++pristine-trees/$state" ; then
        continue
      fi

      cd "$wdroot/{arch}/++pristine-trees/$state"

      for category in `ls` ; do

        cd "$wdroot/{arch}/++pristine-trees/$state"

        if larch valid-package-name -b -- "$category" \
	   && test -z "$limit_category" -o "$limit_category" = "$category" ; then

          cd "$category"

          for branch in `ls` ; do

            cd "$wdroot/{arch}/++pristine-trees/$state/$category"

            if larch valid-package-name -- "$branch" \
	       && test -z "$limit_branch" -o "$limit_branch" = "$branch" ; then

              cd "$branch" 

	      for version in `ls` ; do

		cd "$wdroot/{arch}/++pristine-trees/$state/$category/$branch/"

		if larch valid-package-name --vsn -- "$version" \
		   && test -z "$limit_version" -o "$limit_version" = "$version" ; then

		   cd "$version"

                   for archive in `ls` ; do

		     cd "$wdroot/{arch}/++pristine-trees/$state/$category/$branch/$version"

		     if larch valid-archive-name -- "$archive" \
		        && test -z "$limit_archive" -o "$limit_archive" = "$archive" ; then

                       cd "$archive"

                       for revision in `ls` ; do

                        if larch valid-package-name -l -- "$revision" \
		           && test -z "$limit_revision" -o "$limit_revision" = "$revision" ; then

		          if test -z "$long" ; then
                            printf "%s/%s\\n" "$archive" "$revision"
			  else
                            printf "%-13s%s/%s\\n" "$state" "$archive" "$revision"
			  fi

                        fi
                      done
                    fi
                  done
                fi
              done
            fi
          done
        fi
      done
    done
  fi ) \
| sed -e 's/[[:digit:]]*$/ &/' \
| sort $sort_key \
| sed -e 's/ \([[:digit:]]*\)$/\1/'

# tag: Tom Lord Mon Dec 10 22:10:18 2001 (local-cache/pristines.sh)
#
