# sort-revisions.sh - sort a list of revisions
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "sort a list of revisions\\n"
		printf "usage: sort-revisions [options] id-string\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " --full                        assume fully qualified names\\n"
		printf " --lvl                         assume patch level names only\\n"
		printf "\\n"
		printf " -r --reverse                  reverse the sort order\\n"
		printf "\\n"
		printf "Sort the list of revisions on standard input.  With \"--full\",\\n"
		printf "assume that revision names include and explicit archive specification.\\n"
		printf "With \"--lvl\", assume that the input only patch level names (e.g. \\n"
		printf "\"patch-23\").  With no options, assume package names without explicit\\n"
		printf "archive names (e.g. \"arch--devo--1.0--patch-23\").\\n"
		printf "\\n"
		printf "\"-r\" means reverse the ordering of versions and patch levels -- \\n"
		printf "branch names are still sorted in forward lexical order.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

verbose=0
errname=
type=package
reverse=
reverse_r=

while test $# -ne 0 ; do

  case "$1" in 


    --full)		shift
    			type=full
			;;

    --lvl)		shift
    			type=lvl
			;;

    -r|--reverse)	shift
    			reverse=-r
			reverse_r=r
			;;

    -*)			printf "sort-revisions: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 0 ; then
  printf "usage: sort-revisions [options]"
  printf "try --help\\n"
  exit 1
fi


################################################################
# Do It
# 

case "$type" in

  full)		sed -e 's,\(.*\)/\(.*--\)\([^-]*\)-\([[:digit:]]*\),\1 \2 \3 \4 &,' \
		| sort -k1,1 -k2,2 -k3,3${reverse_r} -k4,4${reverse_r}n \
		| cut -d ' ' -f 5
		;;


  package)	sed -e 's/\(.*--\)\([^-]*\)-\([[:digit:]]*\)/\1 \2 \3 &/' \
		| sort -k1,1 -k2,2${reverse_r} -k3,3${reverse_r}n \
		| cut -d ' ' -f 4
		;;

  lvl)		sort -t- -k1,1${reverse_r} -k2,2${reverse_r}n
  		;;

  *)		printf "sort-revisions: internal error\\n"1>&2
  		printf "  unrecognized sort type %s\\n" "$type" 1>&2
		exit 1
		;;

esac


# tag: Tom Lord Sun Jan 27 15:17:18 2002 (naming-conventions/sort-fq-revisions.sh)
#
