# 
# log-ls.sh - print version patches recorded for a project tree
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "print version patches in a project tree\\n"
		printf "usage: log-ls [options] [[archive/]version] ...\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -d --dir DIR                  cd to DIR first\\n"
		printf "\\n"
		printf " -r --reverse                  sort from newest to oldest\\n"
		printf " -s --summary                  print a summary of each patch\\n"
		printf " -c --creator                  print the creator id of each patch\\n"
		printf " -D --date                     print the date of each patch\\n"
		printf " --full                        print full patch level names\\n"
		printf "\\n"
		printf "Print the list of patches applied to a project tree\\n"
		printf "DIR (or the current directory) from VERSION."
		printf "\\n"
		printf "Patch names are printed simply:\\n"    
		printf "\\n"
		printf "	\"%%s\\\\n\"  \$patch_level\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
reverse=
summary=
creator=
date=
full=
dir=
debug_opt=

while test $# -ne 0 ; do

  case "$1" in 

    --debug)		shift
    			debug_opt=--debug
			printf "\n" 1>&2
			printf "log-ls: DEBUGGING ACTIVATED\n" 1>&2
			printf "\n" 1>&2
			set -x
			;;
			
    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "log-ls: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "log-ls: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -d|--dir)           shift
                        if test $# -eq 0 ; then
                          printf "log-ls: -d and --dir require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        dir="$1"
                        shift
                        ;;

    -r|--reverse)	shift
    			reverse=-r
			;;

    -s|--summary)	shift
    			summary=--summary
			;;

    -c|--creator)	shift
    			creator=--creator
			;;

    -D|--date)		shift
    			date=--date
			;;

    --full)		shift
    			full=--full
			;;

    --)			shift
    			break
			;;

    ---)		break
    			;;

    -*)			printf "log-ls: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ge 1 ; then
  archive_version="$1"
  shift
else
  archive_version=
fi


################################################################
# Sanity Check and Process Defaults
# 

if test "$archive_version" = --- ; then
  exit 0
fi

if test -z "$dir" ; then
  dir="."
fi

cd "$dir"
dir="`pwd`"

cd "$dir"
wdroot="`larch tree-root --accurate`"
cd "$wdroot"

if test -z "$archive_version" ; then
  archive_version=`larch tree-version`
else
  larch valid-package-name -e log-ls --tolerant -- "$archive_version"
fi

archive=`larch parse-package-name -R "$archroot" -A "$archive" --arch $archive_version`
spec=`larch parse-package-name -R "$archroot" -A "$archive" --non-arch $archive_version`
category=`larch parse-package-name --basename $spec`
branch=`larch parse-package-name $spec`

if larch valid-package-name --vsn --tolerant "$spec" ; then
  version="`larch parse-package-name --package-version \"$spec\"`"
else
  cd "$wdroot"
  version="`larch logs $debug_opt --reverse -a \"$archive\" -b \"$branch\" | head -1`"
  if test -z "$version" ; then
    printf "\\n" 1>&2
    printf "log-ls: no patch logs for branch\\n" 1>&2
    printf "  tree: %s\\n" "$wdroot" 1>&2
    printf "  branch: %s\\n" "$archive/$branch" 1>&2
    printf "\\n" 1>&2
    exit 1
  else
    version="`larch parse-package-name --non-arch \"$version\"`"
  fi
fi

if larch valid-package-name --lvl "$spec" ; then
  lvl="`larch parse-package-name --lvl \"$spec\"`"
else
  lvl=-
fi


################################################################
# Print the list of patches as patch-level names.
# 

cd "$wdroot/{arch}"

logdir="$category/$branch/$version/$archive/patch-log"

if test ! -d "$logdir" ; then
  printf "log-ls: project tree has no patch-log for version\\n" 1>&2
  printf "\\n" 1>&2
  printf "  project tree: %s\\n" "$wdroot" 1>&2
  printf "  version: %s\\n" $version 1>&2
  printf "\\n" 1>&2
  printf "try \"larch add-log --help\"\\n" 1>&2
  exit 1
fi

patchlvlre="base-0|patch-[0-9]+|version-0|versionfix-[0-9]+"

cd "$logdir"

if test $# -gt 0 -a -z "$full" ; then
  printf "%s:\\n" "$archive/$version"
  printf "\\n"
fi


if test -z "$summary" -a -z "$creator" -a -z "$date" ; then

  if test -z "$full" ; then
    ls | grep -E "^($patchlvlre)\$" | grep -F $lvl | sort -t- -k 1,1${reverse#-} -k 2,2${reverse#-}n 
  else
    ls | grep -E "^($patchlvlre)\$" | grep -F $lvl | sort -t- -k 1,1${reverse#-} -k 2,2${reverse#-}n \
    | awk -v arch="$archive" -v vsn="$version" '{ printf("%s/%s--%s\n", arch, vsn, $1); }'
  fi



else

  for patch in `ls | grep -E "^($patchlvlre)\$" | grep -F $lvl | sort -t- -k 1,1${reverse#-} -k 2,2${reverse#-}n` ; do

    if test -z "$full" ; then
      printf "%s\\n" "$patch"
    else
      printf "%s/%s--%s\\n" "$archive" "$version" "$patch"
    fi

    cat $patch \
    | awk -v summary="$summary" \
	  -v creator="$creator" \
	  -v date="$date" \
	  -f "$ARCH_SUBCMD_ROOT/patch-logs/patch-list-description.awk"

  done

fi

if test $# -gt 0 ; then
  if test -z "$full" ; then
    printf "\\n"
    printf "\\n"
  fi
  larch log-ls $debug_opt --dir "$dir" $reverse $summary $full "$@" ---
fi

