# 
# valid-log-file.sh - validate a log file
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "validate a log file\\n"
		printf "usage: valid-log-file [options] [--] file [dir]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -e --errname name             use ERRNAME for error messages\\n"
		printf "\\n"
		printf "Check the validity of a log file.\\n"
		printf "\\n"
		printf "If it is valid, exit silently with status 0, otherwise status 1.\\n"
		printf "\\n"
		printf "If an invalid log file, and an error name is provided (-e or --errname)\\n"
		printf "print an error message before exiting.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

errname=

while test $# -ne 0 ; do

  case "$1" in 

    --)			shift
    			break
			;;
			
    -e|--errname)	shift
    			if test $# -eq 0 ; then
			  printf "valid-log-file: -e and --errname require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			errname="$1"
			shift
			;;

    -*)			printf "valid-log-file: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -lt 1 -o $# -gt 2 ; then
  printf "usage: valid-log-file [options] [--] file [dir]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

logfile="$1"
shift

if test $# -gt 0 ; then
  dir="$1"
  shift
else
  dir=.
fi

cd "$dir"
dir="`pwd`"

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
wdroot="`larch tree-root`"


################################################################
# Check the Format of the Log File
# 
# This needs work:
# 

if test ! -e "$logfile" ; then
  if test "x$errname" != x ; then
    printf "%s: no log file found\\n" "$errname" 1>&2
    printf "\\n" 1>&2
    printf "  should be: %s\\n" "$logfile" 1>&2
    printf "  in: %s\\n" "$wdroot" 1>&2
    printf "\\n" 1>&2
    printf "try \"larch make-log --help\\n" 1>&2
    printf "\\n" 1>&2
  fi
  exit 1
fi


################################################################
# If we reach this point, the log file is valid.
# 

cat "$logfile" \
| awk \
    '
     function safe_getline()
     {
       status = getline;
       if (status < 0)
         {
	   print "I/O ERROR in valid-log-file" | "cat 1>&2";
	   exit (1);
	 }
       if (!status)
         eof_seen = 1;
       return status;
     }

     BEGIN { 
	     eof_seen = 0;

	     if (safe_getline())
	       {
                 while (!eof_seen && !match ($0, "^$"))
                   {
                     field = tolower ($1);
                     sub (":.*", "", field);
                     headers[field] = $0;
                     sub ("^[^:]*:[ \t]*", "", headers[field]);
                     while (safe_getline() && match ($0, "^[ \t]"))
                       {
                         headers[field] = headers[field] $0;
                       }
		   }
               }
           }

     { if (body == "") body = "    " $0; else body = body "\n    " $0; }
                  
     END {
	   if (headers["summary"] == "")
	      error = "missing (or empty) \"Summary:\" header";
	   else if (body == "")
	      error = "missing (or empty) body";

	   if (error != "")
	     {
	       printf("\n") | "cat 1>&2"; 
	       printf("valid-log-file: invalid log file\n") | "cat 1>&2";
	       printf("    %s\n", error) | "cat 1>&2"; 
	       printf("\n") | "cat 1>&2";
	       exit (1);
	     }
         }'



exit 0

