# 
# my-default-archive.sh - the user's default archive
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "print or change your default archive\\n"
		printf "usage: my-default-archive [options] [archive]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive dfltarchive      specify the default archive name\\n"
		printf " -e --errname prog             specify program name for errors\\n"
		printf "\\n"
		printf " -s --silent                   don't print a reassuring message\\n"
		printf " -d --delete                   unspecify your default archive\\n"
		printf "\\n"
		printf "With no argument, and without -d, print the name of your default\\n"
		printf "archive.\\n"
		printf "\\n"
		printf "With an argument, record ARCHIVE as your default archive\\n"
		printf "in ~/.arch-params/=default-archive\\n"
		printf "\\n"
		printf "With the option -d (--delete) and no argument, ensure that\\n"
		printf "you do not have a default archive set in ~/.arch-params.\\n"
		printf "\\n"
		printf "Your default archive is determined this way:\\n"
		printf "\\n"
		printf "If the option -A (--archive) is given and not empty, that archive\\n"
 		printf "is the the default (which makes this script useful for processing\\n"
		printf "a -A argument that was passed to another script).\\n"
		printf "\\n"
		printf "If -A is not given, but ~/.arch-params/=default-archive exists\\n"
		printf "and is not empty, that is your default archive.\\n"
		printf "\\n"
		printf "Otherwise, your default archive is the name of the local archive\\n"
		printf "rooted at the argument to -R (--root) or specified in the environment\\n"
		printf "variable ARCHROOT.\\n"
		printf "\\n"
		printf "If no default archive can be found by any of these means, the\\n"
		printf "program exits with status 1, printing an error message unless\\n"
		printf "the -s (--silent) option is given.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

silent=0
delete=0
errname=my-default-archive
archroot=
archive=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "my-default-archive: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "my-default-archive: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;

    -e|--errname)	shift
			if test $# -eq 0 ; then
			  printf "my-default-archive: -e and --errname require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			errname="$1"
			shift
			;;

    -s|--silent)	shift
    			silent=1
			;;

    -d|--delete)	shift
    			delete=1
			;;

    -*)			printf "my-default-archive: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;

  esac

done



################################################################
# Ordinary Arguments
# 

if test $delete -ne 0 ; then

  if test $# -ne 0 ; then
    printf "usage: my-default-archive [options] -d\\n" 1>&2
    printf "try --help\\n" 1>&2
    exit 1
  fi

  type=delete

else

  case $# in 

    0)	type=print
	;;

    1)	type=set
  	newdflt="$1"
	;;

    *)	if test $# -gt 1 ; then
	  printf "usage: my-default-archive [options] [archive]\\n" 1>&2
	  printf "try --help\\n" 1>&2
	  exit 1
	fi
	;;
  esac

fi


################################################################
# Sanity Check
# 

if test $type = set ; then
  larch valid-archive-name --errname "$errname" -- "$newdflt"
fi

if test "x$archive" != x ; then
  larch valid-archive-name --errname "$errname" -- "$archive"
fi

################################################################
# Do It
# 

case $type in

  delete)	mkdir -p ~/.arch-params
  		cd ~/.arch-params

		if test -e =default-archive ; then
		  rm =default-archive
 		fi

		if test $silent -eq 0 ; then
		  printf "user default archive removed\\n"
		fi

		exit 0
		;;

  set)		mkdir -p ~/.arch-params/
  		printf "%s\\n" "$newdflt" > ~/.arch-params/=default-archive

		if test $silent -eq 0 ; then
		  printf "default archive set (%s)\\n" "$newdflt"
		fi

		exit 0
		;;

  print)	if test "x$archive" != x ; then

		  printf "%s\\n" "$archive"

		elif test -e ~/.arch-params/=default-archive ; then

		  archive="`cat  ~/.arch-params/=default-archive`"

		  if ! larch valid-archive-name -- "$archive" ; then
		    printf "%s: invalid archive name in ~/.arch-params/=default-archive\\n" "$errname" 1>&2
		    printf "  invalid archive name: %s\\n" "$archive" 1>&2
		    exit 1
		  fi

		  printf "%s\\n" "$archive"

		elif test "x$ARCHIVE" != x ; then

		  if ! larch valid-archive-name -- "$ARCHIVE" ; then
		    printf "%s: invalid archive name in \$ARCHIVE (%s)\\n" "$errname" "$ARCHIVE" 1>&2
		    exit 1
		  fi

		  printf "%s\\n" "$ARCHIVE"

		elif test "x$archroot$ARCHROOT" != x ; then

		  larch get-archive-name -R "$archroot" 

		else

		  if test $silent -eq 0 ; then
		    printf "%s: no default archive\\n" "$errname" 1>&2
		    printf "  try \"larch my-default-archive --help\"\\n" 1>&2
		  fi

		  exit 1

		fi
		;;


esac
