/*
 * Copyright (c) 2000 QoSient, LLC
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of QoSient not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * QOSIENT, LLC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL QOSIENT, LLC BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/*
 * rapolicy - check argus records against cisco access control
 *            configuration.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <rapolicy.h>


int RaInitialized = 0;

int RaReadPolicy (struct RaAccessPolicyStruct **, char *);
int RaParsePolicy (struct RaAccessPolicyStruct *, char *);
int RaCheckPolicy (struct ArgusRecord *, struct RaAccessPolicyStruct *);
int RaMeetsPolicyCriteria (struct ArgusRecord *, struct RaAccessPolicyStruct *);
int RaDoNotification (struct ArgusRecord *, struct RaAccessPolicyStruct *);

struct RaAccessPolicyStruct *policy = NULL;

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaInitialized++;


      if (ArgusFlowModelFile != NULL) {
         RaReadPolicy (&policy, ArgusFlowModelFile);
      }
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if (!RaParseCompleting) {
      RaParseCompleting++;
   }
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Rapolicy Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [-f policyfile] [ra-options] [- filter-expression]\n", ArgusProgramName);

   fprintf (stderr, "options:    -f <policyfile> read Cisco access control policy file.\n");
   fprintf (stderr, "ra-options: -a              print record summaries on termination.\n");
   fprintf (stderr, "            -A              print application bytes.\n");
   fprintf (stderr, "            -b              dump packet-matching code.\n");
   fprintf (stderr, "            -c              print packet and byte counts.\n");
   fprintf (stderr, "            -C              treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>      specify debug level\n");
   fprintf (stderr, "            -E <file>       write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "            -F <conffile>   read configuration from <conffile>.\n");
   fprintf (stderr, "            -g              print record time duration.\n");
   fprintf (stderr, "            -G              print both start and last time values.\n");
   fprintf (stderr, "            -h              print help.\n");
   fprintf (stderr, "            -I              print transaction state and option indicators.\n");
   fprintf (stderr, "            -l              print last time values [default is start time].\n");
   fprintf (stderr, "            -m              print MAC addresses.\n");
   fprintf (stderr, "            -n              don't convert numbers to names.\n");
   fprintf (stderr, "            -p <digits>     print fractional time with <digits> precision.\n");
   fprintf (stderr, "            -P <portnum>    specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -q              quiet mode. don't print record outputs.\n");
   fprintf (stderr, "            -r <file>       read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -R              print out response data when availabile.\n");
   fprintf (stderr, "            -S <host>       specify remote argus <host>.\n");
   fprintf (stderr, "            -t <timerange>  specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:   timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                            timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                mm/dd[/yy]\n");
   fprintf (stderr, "            -T <secs>       attach to remote server for T seconds.\n");
   fprintf (stderr, "            -u              print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth>  specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>       write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "            -z              print Argus TCP state changes.\n");
   fprintf (stderr, "            -Z <s|d|b>      print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}


void
process_man (struct ArgusRecord *argus)
{
}


void
process_tcp (struct ArgusRecord *argus)
{
   if (RaCheckPolicy (argus, policy))
      printf ("%s\n", get_tcp_string(argus));
}


void
process_icmp (struct ArgusRecord *argus)
{
   if (RaCheckPolicy (argus, policy))
      printf ("%s\n", get_icmp_string(argus));
}


void
process_udp (struct ArgusRecord *argus)
{
   if (RaCheckPolicy (argus, policy))
      printf ("%s\n", get_udp_string(argus));
}


void
process_ip (struct ArgusRecord *argus)
{
   if (RaCheckPolicy (argus, policy))
      printf ("%s\n", get_ip_string(argus));
}


void
process_arp (struct ArgusRecord *argus)
{
}


void
process_non_ip (struct ArgusRecord *argus)
{
}

void RaSendArgusRecord(struct ArgusRecordStore *argus) {};


#include <errno.h>


char *RaParseErrorStr [POLICYFIELDNUM] = {
   "access-list identifier not found",
   "policy id number not found",
   "permit/deny indication not found",
   "protocol indentifier not found",
   "no source address defined",
   "no source address mask defined",
   "no destination address defined",
   "no destination address mask defined",
   "port operator not found",
   "port number not found",
   "access violation notification not found",
};

int
RaReadPolicy (struct RaAccessPolicyStruct **policy, char *file)
{
   int retn = 1, linenum = 0;
   struct RaAccessPolicyStruct *pol, *policyLast = NULL;
   FILE *fd;
   char buffer [1024], *str;

   if (file) {
      if ((fd = fopen (file, "r")) != NULL) {
         while (fgets (buffer, 1024, fd)) {
            linenum++;
            if ((*buffer != '#') && (*buffer != '\n') && (*buffer != '!')) {
               if ((pol = (struct RaAccessPolicyStruct *) calloc (1, sizeof (struct RaAccessPolicyStruct))) != NULL) {
                  pol->str = strdup (buffer);

                  if (!(RaParsePolicy (pol, buffer))) {
                     str = RaParseErrorStr[RaParseError];
                     ArgusLog (LOG_ERR, "RaReadPolicy %s parse error line %d: %s\n", file, linenum, str);
                     retn = 0;
                  }
                  if (policyLast)  {
                     policyLast->nxt = pol;
                     pol->prv = policyLast;
                     policyLast = pol;
                  } else
                     *policy = policyLast = pol;
               }
            }
         }
         fclose (fd);
      } else {
         retn = 0;
         ArgusLog (LOG_ERR, "RaReadPolicy: fopen %s\n", strerror(errno));
      }
   }

#if defined(ARGUSDEBUG)
   ArgusDebug (2, "RaReadPolicy (0x%x, %s) returning %d\n", policy, file, retn);
#endif

   return (retn);
}

#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <ctype.h>

int
RaParsePolicy (struct RaAccessPolicyStruct *policy, char *buf)
{
   int retn = 1, error = 0, i = 0;
   char *ptr;

   if (strncmp (buf, "access", 6)) {
      if (strstr (buf, "ip"))
         if (strstr (buf, "source-route"))
            policy->type = RA_SRCROUTED;

   } else {
      policy->type = RA_ACCESSLIST;
      for (i = 0; ((i < POLICYFIELDNUM) && !(error)); i++) {
         if ((ptr = strtok (buf, " ")) && (*ptr != '\n')) {
            switch (i) {
               case POLICYSTRING: 
                  if ((strcmp (ptr, POLICY_STRING))) error++; 
                  break;
   
               case POLICYID: 
                  if (!(policy->policyID = atoi (ptr))) error++;
                  break;
   
               case POLICYACTION: 
                  if (!(strcmp (ptr, "permit")))
                     policy->flags |= RA_PERMIT;
                  else if (!(strcmp (ptr, "deny")))
                     policy->flags |= RA_DENY;
                  else error++;
                  break;
   
               case POLICYPROTO: 
                     if (isdigit((int)*ptr)) {
                        policy->proto = atoi(ptr);
                        policy->flags |= RA_PROTO_SET;
                     } else {
                        struct protoent *proto;
                        if ((proto = getprotobyname(ptr)) != NULL) {
                           policy->proto = proto->p_proto;
                           policy->flags |= RA_PROTO_SET;
                        } else {
                           error++;
                        }
                     }


                     break;

               case POLICYDSTADDR: 
                  policy->dst.addr = ntohl(inet_addr (ptr)); break;
               case POLICYDSTMASK: 
                  policy->dst.mask = ntohl(inet_addr (ptr)); break;
               case POLICYSRCADDR: 
                  policy->src.addr = ntohl(inet_addr (ptr)); break;
               case POLICYSRCMASK: 
                  policy->src.mask = ntohl(inet_addr (ptr)); break;
   
               case POLICYPORTACT: 
                  if (!(strcmp (ptr, "eq")))
                     policy->action |= RA_EQ;
                  else if (!(strcmp (ptr, "lt")))
                     policy->action |= RA_LT;
                  else if (!(strcmp (ptr, "gt")))
                     policy->action |= RA_GT;
                  else if (!(strcmp (ptr, "neq")))
                     policy->action |= RA_NEQ;
                  else if (!(strncmp (ptr, "established",11)))
                     policy->action |= RA_EST;
                  else error++;
                  break;
   
               case POLICYPORT: 
                  if (!(policy->port = (arg_uint16) atoi (ptr))) error++;
                  break;
   
               case POLICYNOTIFICATION: 
                  break;
            }
   
            buf = NULL;
         } else
            break;
      }
   }

   if (error) {
      RaParseError = i - 1; retn = 0;
   }

   return (retn);
}


int
RaCheckPolicy (struct ArgusRecord *argus, struct RaAccessPolicyStruct *policy)
{
   int retn = 0, policymatch = 0;

   if (!((argus->ahdr.status) & ARGUS_MAR)) {
      if (policy) {
         while (policy) {
            if ((retn = RaMeetsPolicyCriteria (argus, policy))) {
               retn = RaDoNotification (argus, policy);
               policymatch = 1;
               break;
            }
            policy = policy->nxt;
         }
      }
   }

   return (retn);
}


int
RaMeetsPolicyCriteria (struct ArgusRecord *argus, struct RaAccessPolicyStruct *policy)
{
   int retn = 0, i = 0;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   u_char proto = flow->ip_flow.ip_p;

   arg_uint32 saddr = 0, daddr = 0;
   arg_uint16 sport = 0, dport = 0;

   switch (policy->type) {
   case RA_SRCROUTED:
      if ((argus->argus_far.attr_ip.soptions & (ARGUS_SSRCROUTE | ARGUS_LSRCROUTE)) || 
          (argus->argus_far.attr_ip.doptions & (ARGUS_SSRCROUTE | ARGUS_LSRCROUTE)))
         retn++;
      break;

   case RA_ACCESSLIST:
      saddr = flow->ip_flow.ip_src;
      daddr = flow->ip_flow.ip_dst;
      sport = flow->ip_flow.sport;
      dport = flow->ip_flow.dport;
   
      for (i = 0, retn = 1; ((i < POLICYTESTCRITERIA) && retn); i++) {
         retn = 0;
         switch (i) {
            case POLICYTESTPROTO:
               if (policy->flags & (RA_PROTO_SET)) {
                  if (policy->proto) {
                     if (policy->proto == proto)
                        retn++;
                  } else
                     retn++;
               }
               break;
   
            case POLICYTESTSRC:
               if ((saddr & ~policy->src.mask) == policy->src.addr)
                  retn++;
               break;
            case POLICYTESTDST:
               if ((daddr & ~policy->dst.mask) == policy->dst.addr)
                  retn++;
               break;
            case POLICYTESTPORT:
               switch (policy->action) {
                  case  RA_EQ:
                     if (dport == policy->port)
                        retn++; break;
                  case  RA_LT:
                     if (dport < policy->port)
                        retn++; break;
                  case  RA_GT:
                     if (dport > policy->port)
                        retn++; break;
                  case RA_NEQ:
                     if (dport != policy->port)
                        retn++; break;
                  case  RA_EST: {
                     if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
                        struct ArgusTCPObject *tcp = NULL;

                        tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

                        if (!(tcp->status & ARGUS_SAW_SYN))
                           retn++;
                     }
                     break;
                  }

                  default:
                     retn++; break;
               }
               break;
         }
      }
   }

   return (retn);
}

int
RaDoNotification (struct ArgusRecord *argus, struct RaAccessPolicyStruct *policy)
{
   int retn = 1;

   if (policy) {
      if (policy->flags & RA_PERMIT) {
         if (Argusdflag > 1) {
            printf ("%s %s ", "policy: permitted", policy->str);
         } else
            retn = 0;

      } else {
         if (Argusdflag)
            printf ("%s %s ", "policy: denyed", policy->str);
      }
   }

   return (retn);
}
