/*
 * Copyright (c) 2000-2001 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * Copyright (c) 1993, 1994 Carnegie Mellon University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 */

/*
 * rasort - sort argus records.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */



#ifndef ArgusClient
#define ArgusClient
#endif

#include <argus_client.h>
#include <rasort.h>


int RaSortRoutine (const void *, const void *);

int RaSortStartTime (struct ArgusRecord *, struct ArgusRecord *);
int RaSortLastTime (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDuration (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcAddr (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstAddr (struct ArgusRecord *, struct ArgusRecord *);
int RaSortProtocol (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcPort (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstPort (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcTos (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstTos (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcTtl (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstTtl (struct ArgusRecord *, struct ArgusRecord *);
int RaSortByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortPktsCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcPktsCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstPktsCount (struct ArgusRecord *, struct ArgusRecord *);

extern char *RaSortAlgorithmStrings[];

#define MAX_SORT_ALG_TYPES	18

char *RaSortKeyWords[MAX_SORT_ALG_TYPES] = {
   "startime",
   "lasttime",
   "duration",
   "srcaddr",
   "dstaddr",
   "proto",
   "sport",
   "dport",
   "stos",
   "dtos",
   "sttl",
   "dttl",
   "bytes",
   "srcbytes",
   "dstbytes",
   "packets",
   "srcpackets",
   "dstpackets",
};

int (*RaSortAlgorithm)(struct ArgusRecord *, struct ArgusRecord *) = NULL;
int (*RaSortAlgorithms[ARGUS_MAX_SORT_ALG])(struct ArgusRecord *, struct ArgusRecord *);

int (*RaSortAlgorithmTable[MAX_SORT_ALG_TYPES])(struct ArgusRecord *, struct ArgusRecord *) = {
   RaSortStartTime,
   RaSortLastTime,
   RaSortDuration,
   RaSortSrcAddr,
   RaSortDstAddr,
   RaSortProtocol,
   RaSortSrcPort,
   RaSortDstPort,
   RaSortSrcTos,
   RaSortDstTos,
   RaSortSrcTtl,
   RaSortDstTtl,
   RaSortByteCount,
   RaSortSrcByteCount,
   RaSortDstByteCount,
   RaSortPktsCount,
   RaSortSrcPktsCount,
   RaSortDstPktsCount
};

void
ArgusClientInit ()
{
   int i, x;

   if (!(RaInitialized)) {
      RaWriteOut = 0;

      if ((RaArgusRecordQueue = RaNewQueue(128)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: RaNewQueue error %s\n", strerror(errno));

      bzero ((char *) RaSortAlgorithms, sizeof(RaSortAlgorithms));
      RaSortAlgorithms[0] = RaSortAlgorithmTable[0];

      for (i = 0; i < ARGUS_MAX_SORT_ALG; i++) {
         if (RaSortAlgorithmStrings[i] != NULL) {
            for (x = 0; x < MAX_SORT_ALG_TYPES; x++) {
               if (!strncmp (RaSortKeyWords[x], RaSortAlgorithmStrings[i], strlen(RaSortKeyWords[x]))) {
                  RaSortAlgorithms[i] = RaSortAlgorithmTable[x];
                  break;
               }
            }
            if (x == MAX_SORT_ALG_TYPES)
               ArgusLog (LOG_ERR, "sort syntax error. \'%s\' not supported", RaSortAlgorithmStrings[i]);
         } else
            break;
      }

      if (nflag) {
         hfield = 15;
         pfield =  5;
      }

      RaInitialized++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusClientInit () returning\n");
#endif
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   struct ArgusRecordStore *obj = NULL;

   if (!RaParseCompleting) {
      RaParseCompleting++;

      if (Nflag == 0)
         Nflag = RaArgusRecordQueue->count;

      RaPackQueue (RaArgusRecordQueue);
      RaSortQueue (RaArgusRecordQueue);
      while (Nflag--) {
         if ((obj = (struct ArgusRecordStore *) RaPopQueue(RaArgusRecordQueue)) != NULL) {
            RaSendArgusRecord (obj);
            ArgusFree(obj->argus);
            ArgusFree(obj);
         } else
            break;
      }
   }
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   extern char version[];

   fprintf (stderr, "Rasort Version %s\n", version);
   fprintf (stderr, "usage: %s [[-s sortfield] [-s sortfield] ...] [ra-options] [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "options:    -s <sortfield> specify the <sortfield>(s) in order.\n");
   fprintf (stderr, "                           valid sorfields are:\n");
   fprintf (stderr, "                              startime, lasttime, duration,\n");
   fprintf (stderr, "                              srcaddr, dstaddr, proto, sport, dport,\n");
   fprintf (stderr, "                              stos, dtos, sttl, dttl,\n");
   fprintf (stderr, "                              bytes, srcbytes, dstbytes,\n");
   fprintf (stderr, "                              packets, srcpackets, dstpackets.\n");
   fprintf (stderr, "\n");
   fprintf (stderr, "ra-options: -b             dump packet-matching code.\n");
   fprintf (stderr, "            -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "            -D <level>     specify debug level\n");
   fprintf (stderr, "            -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "            -h             print help.\n");
   fprintf (stderr, "            -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "            -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "            -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "            -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                               mm/dd[/yy]\n");
   fprintf (stderr, "            -T <secs>      attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "            -U <user/auth> specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "            -w <file>      write output to <file>. '-' denotes stdout.\n");
   exit(1);
}

int RaAllocArgusRecordStore = 0;

void
RaProcessRecord (struct ArgusRecord *argus)
{
   struct ArgusRecordStore *obj = NULL;
   struct timeval lasttime;
   
   if (argus != NULL) {
      if ((obj = (struct ArgusRecordStore *) ArgusCalloc (1, sizeof (*obj))) != NULL) {
         RaAllocArgusRecordStore++;
         lasttime = argus->argus_far.time.last;
   
         obj->qhdr.lasttime = lasttime;
         obj->qhdr.logtime  = lasttime;
   
         if ((obj->argus = RaCopyArgusRecord(argus)) == NULL)
            ArgusLog (LOG_ERR, "RaProcessRecord: RaCopyArgusRecord error %s\n", strerror(errno));
   
         RaAddToQueue(RaArgusRecordQueue, &obj->qhdr);
   
      } else
         ArgusLog (LOG_ERR, "RaProcessRecord: ArgusCalloc(1, %d) error %s\n",
                         sizeof(*obj), strerror(errno));
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaProcessRecord(0x%x) done.\n", argus);
#endif
}


u_char ArgusRecordBuffer[1024];
int ArgusLastRecord = 0;

void
process_man (struct ArgusRecord *argus)
{
   if ((argus->ahdr.cause & ARGUS_STOP)) {
      bcopy ((char *) ArgusOriginal, ArgusRecordBuffer, argus->ahdr.length);
      ArgusLastRecord++;
   }
}


void
process_tcp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_icmp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_udp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_ip (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_arp (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
process_non_ip (struct ArgusRecord *argus)
{
   RaProcessRecord (argus);
}


void
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   struct ArgusRecord *argus = store->argus;

   if (wfile) {

#ifdef _LITTLE_ENDIAN
      ArgusHtoN(argus);
#endif

      ArgusWriteNewLogfile (wfile, argus);

#ifdef _LITTLE_ENDIAN
      ArgusNtoH(argus);
#endif

   } else {

      if (argus->ahdr.type & ARGUS_MAR)
         printf ("%s\n", get_man_string (argus));
         
      else {
         ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

         switch (argus->ahdr.status & 0xFFFF) {
            case ETHERTYPE_IP:
               switch (argus->argus_far.flow.ip_flow.ip_p) {
                  case IPPROTO_TCP:              
                     printf ("%s\n", get_tcp_string (argus));
                     break;
   
                  case IPPROTO_ICMP:              
                     printf ("%s\n", get_icmp_string (argus));
                     break;
   
                  default:
                     printf ("%s\n", get_ip_string (argus));
                     break;
               }
               break;
   
            case ETHERTYPE_ARP:
            case ETHERTYPE_REVARP:
               printf ("%s\n", get_arp_string (argus));
               break;
   
            default:
               printf ("%s\n", get_nonip_string (argus));
               break;
         }
      }
      fflush (stdout);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
}

int RaSortStartTime (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   unsigned int retn;

   if ((retn = a2->argus_far.time.start.tv_sec - a1->argus_far.time.start.tv_sec) == 0)
      retn = a2->argus_far.time.start.tv_usec - a1->argus_far.time.start.tv_usec;

   return (retn);
}

int RaSortLastTime (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   unsigned int retn;

   if ((retn = a2->argus_far.time.last.tv_sec - a1->argus_far.time.last.tv_sec) == 0)
      retn = a2->argus_far.time.last.tv_usec - a1->argus_far.time.last.tv_usec;

   return (retn);
}

int RaSortDuration (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return ((int)(RaGetuSecDuration(a2) - RaGetuSecDuration(a1)));
}

int RaSortSrcAddr (struct ArgusRecord *a2, struct ArgusRecord *a1)
{
  int retn = 0;

   if ((retn = (a1->ahdr.status & 0xFFFF) - (a2->ahdr.status & 0xFFFF)) == 0) {
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_IP))
         retn = (a1->argus_far.flow.ip_flow.ip_src - a2->argus_far.flow.ip_flow.ip_src);
      else
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_ARP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_ARP))
         retn = (a1->argus_far.flow.arp_flow.arp_spa - a2->argus_far.flow.arp_flow.arp_spa);
      else
         retn = bcmp ((char *)&a2->argus_far.flow.mac_flow.ehdr.ether_shost, (char *)&a1->argus_far.flow.mac_flow.ehdr.ether_shost, 6);
   }

   return (retn);
}

int RaSortDstAddr (struct ArgusRecord *a2, struct ArgusRecord *a1)
{
   int retn = 0;

   if ((retn = (a1->ahdr.status & 0xFFFF) - (a2->ahdr.status & 0xFFFF)) == 0) {
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_IP))
         retn = (a1->argus_far.flow.ip_flow.ip_dst - a2->argus_far.flow.ip_flow.ip_dst);
      else
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_ARP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_ARP))
         retn = (a1->argus_far.flow.arp_flow.arp_tpa - a2->argus_far.flow.arp_flow.arp_tpa);
      else
         retn = bcmp ((char *)&a2->argus_far.flow.mac_flow.ehdr.ether_dhost, (char *)&a1->argus_far.flow.mac_flow.ehdr.ether_dhost, 6);
   }
   return (retn);
}

int RaSortProtocol (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn;

   if ((retn = (a1->ahdr.status & 0xFFFF) - (a2->ahdr.status & 0xFFFF)) == 0)
      if (((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP) && ((a2->ahdr.status & 0xFFFF) == ETHERTYPE_IP))
         if ((retn = a1->argus_far.flow.ip_flow.ip_p - a2->argus_far.flow.ip_flow.ip_p) == 0)
            retn = a1->argus_far.flow.ip_flow.tp_p - a2->argus_far.flow.ip_flow.tp_p;

   return (retn);
}

int RaSortSrcPort (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return (a2->argus_far.flow.ip_flow.sport - a1->argus_far.flow.ip_flow.sport);
}

int RaSortDstPort (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return (a2->argus_far.flow.ip_flow.dport - a1->argus_far.flow.ip_flow.dport);
}

int RaSortSrcTos (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn = 0;

   if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP)
      retn = (a1->argus_far.attr_ip.stos - a2->argus_far.attr_ip.stos);

   return (retn);
}

int RaSortDstTos (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn = 0;

   if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP)
      retn =  (a1->argus_far.attr_ip.dtos - a2->argus_far.attr_ip.dtos);

   return (retn);
}

int RaSortSrcTtl (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn = 0;

   if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP)
      retn =  (a1->argus_far.attr_ip.sttl - a2->argus_far.attr_ip.sttl);

   return (retn);
}

int RaSortDstTtl (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   int retn = 0;

   if ((a1->ahdr.status & 0xFFFF) == ETHERTYPE_IP)
      retn = (a1->argus_far.attr_ip.dttl - a2->argus_far.attr_ip.dttl);

   return (retn);
}

int RaSortByteCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   if (Aflag)
      return ((a1->argus_far.src.appbytes + a1->argus_far.dst.appbytes) -
              (a2->argus_far.src.appbytes + a2->argus_far.dst.appbytes));
   else
      return ((a1->argus_far.src.bytes + a1->argus_far.dst.bytes) -
              (a2->argus_far.src.bytes + a2->argus_far.dst.bytes));
}

int RaSortSrcByteCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   if (Aflag)
      return (a1->argus_far.src.appbytes - a2->argus_far.src.appbytes);
   else
      return (a1->argus_far.src.bytes - a2->argus_far.src.bytes);
}

int RaSortDstByteCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   if (Aflag)
      return (a1->argus_far.dst.appbytes - a2->argus_far.dst.appbytes);
   else
      return (a1->argus_far.dst.bytes - a2->argus_far.dst.bytes);
}


int RaSortPktsCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return ((a1->argus_far.src.count + a1->argus_far.dst.count) -
           (a2->argus_far.src.count + a2->argus_far.dst.count));
}

int RaSortSrcPktsCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return (a1->argus_far.src.count - a2->argus_far.src.count);
}

int RaSortDstPktsCount (struct ArgusRecord *a1, struct ArgusRecord *a2)
{
   return (a1->argus_far.dst.count - a2->argus_far.dst.count);
}


int
RaSortRoutine (const void *void1, const void *void2)
{
   int retn = 0, i = 0;
   struct ArgusRecordStore *store1 = *(struct ArgusRecordStore **)void1;
   struct ArgusRecordStore *store2 = *(struct ArgusRecordStore **)void2;

   if ((store1 && store1->argus) && ((store2 && store2->argus))) {
      struct ArgusRecord *argus1 = store1->argus;
      struct ArgusRecord *argus2 = store2->argus;

      for (i = 0; i < ARGUS_MAX_SORT_ALG; i++)
         if (RaSortAlgorithms[i] != NULL)
            if ((retn = RaSortAlgorithms[i](argus2, argus1)) != 0)
               break;
   }

   return (retn);
}

void
RaSortQueue (struct RaQueueStruct *queue)
{
   int i;

   qsort ((char *) queue->array, queue->size, sizeof (void *), RaSortRoutine);
   for (i = 0; i < queue->count; i++)
      queue->array[i]->index = i;
}



struct RaQueueStruct *
RaNewQueue (int size)
{
   struct RaQueueStruct *retn =  NULL;

   if ((retn = (struct RaQueueStruct *) ArgusCalloc (1, sizeof (struct RaQueueStruct))) != NULL) {
      retn->head = 0;
      retn->tail = 0;
      retn->count = 0;
      if ((retn->array = (struct ArgusQueueHeader **) ArgusCalloc (size, sizeof(void *))) != NULL)
         retn->size = size;
      else {
         ArgusFree(retn);   
         retn = NULL;   
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaNewQueue () returning 0x%x\n", retn);
#endif

   return (retn);
}

void
RaDeleteQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *obj = NULL;

   while ((obj = RaPopQueue(queue)))
      ArgusFree(obj);

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaDeleteQueue (0x%x) returning\n", queue);
#endif
}

void
RaPackQueue (struct RaQueueStruct *queue)
{
   struct RaQueueStruct *newq;
   struct ArgusQueueHeader *obj;

   if (!((queue->head == 0) && (queue->count == queue->tail))) {
      if ((newq = RaNewQueue (queue->count)) != NULL) {
         while (queue->count)
            if ((obj = RaPopQueue(queue)) != NULL)
               RaAddToQueue(newq, obj);

         ArgusFree(queue->array);

         queue->head  = newq->head;
         queue->tail  = newq->tail;
         queue->count = newq->count;
         queue->size  = newq->count;
         queue->array = newq->array;
      }
   } else
      queue->size = queue->count;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaPackQueue (0x%x) returning\n", queue);
#endif
}

int
RaGetQueueCount(struct RaQueueStruct *queue)
{

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaGetQueueCount (0x%x) returning %d\n", queue, queue->count);
#endif

   return (queue->count);
}

void
RaPushQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   int start = queue->head;

   if (obj && (queue->count < queue->size)) {
      if (--start < 0)
         start = queue->size;
   
      if (queue->array[start] == NULL) {
         queue->count++;
         obj->index = start;
         queue->array[start] = obj;
         queue->head = start;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaPushQueue (0x%x, 0x%x) returning\n", queue, obj);
#endif
}

#define RA_MAXQUEUESIZE         67108864

void
RaAddToQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   struct RaQueueStruct *newq;
   struct ArgusQueueHeader *qhdr, **oarray;

   if (obj) {
      if (!((queue->count + 1) < (queue->size))) {
         if ((queue->size << 1) <= RA_MAXQUEUESIZE) {
            if ((newq = RaNewQueue (queue->size << 1)) != NULL) {
               while (queue->count)
                  if ((qhdr = RaPopQueue(queue)) != NULL)
                     RaAddToQueue(newq, qhdr);

               oarray = queue->array;

               queue->head  = newq->head;
               queue->tail  = newq->tail;
               queue->count = newq->count;
               queue->size  = newq->size;
               queue->array = newq->array;

               newq->array = oarray;
               newq->count = 0;
               RaDeleteQueue (newq);
            }
         }
   #ifdef ARGUSDEBUG
         ArgusDebug (1, "ArgusAddToQueue: expanded queue size to %d\n", queue->size);
   #endif
      }

      queue->count++;

      obj->index = queue->tail;
      queue->array[queue->tail] = obj;

      while (queue->array[queue->tail] != NULL) {
         queue->tail++;
         if (queue->tail == queue->size)
            queue->tail = 0;
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaAddToQueue (0x%x, 0x%x) returning\n", queue, obj);
#endif
}



struct ArgusQueueHeader *
RaPopQueue (struct RaQueueStruct *queue)
{
   struct ArgusQueueHeader *retn = NULL;
   struct ArgusQueueHeader *obj = NULL;

   if (queue && queue->count) {
      if ((obj = queue->array[queue->head]) != NULL)
         retn = RaRemoveFromQueue(queue, obj);
      else
         ArgusLog (LOG_ERR, "RaPopQueue(0x%x) internal queue error count %d head NULL\n", queue, queue->count);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveFromQueue (0x%x) returning 0x%x\n", queue, retn);
#endif
   
   return(retn);
}


struct ArgusQueueHeader *
RaRemoveFromQueue(struct RaQueueStruct *queue, struct ArgusQueueHeader *obj)
{
   struct ArgusQueueHeader *retn = NULL;

   if ((queue != NULL) && (obj != NULL)) {
      int index = obj->index;

      if (queue->count && (queue->array[index] == obj)) {
         queue->count--;
         queue->array[index] = NULL;
         obj->index = 0;
         retn = obj;

         if (queue->count) {
            if (queue->head == index) {
               while (queue->array[queue->head] == NULL) {
                  queue->head++;
                  if (queue->head == queue->size)
                     queue->head = 0;
               }
            }
         } else {
            queue->head = 0;
            queue->tail = 0;
         }

      } else
         ArgusLog (LOG_ERR, "RaRemoveFromQueue(0x%x, 0x%x) obj not in queue\n", queue, obj);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaRemoveFromQueue (0x%x, 0x%x) returning 0x%x\n", queue, obj, obj);
#endif

   return (retn);
}


struct ArgusRecord *
RaCopyArgusRecord (struct ArgusRecord *argus)
{
   int length = 0;
   struct ArgusRecord *retn = NULL;

   if (argus && ((length = argus->ahdr.length) > 0)) {
      if ((retn = (struct ArgusRecord *) ArgusCalloc(1, length)) != NULL)
         bcopy ((char *) argus, (char *) retn, length);
      else
         ArgusLog (LOG_ERR, "RaCopyArgusRecord(0x%x) ArgusCalloc(1, %d) error %s\n",
                         argus, length, strerror(errno));
   }

   return (retn);
}


long long
RaGetActiveDuration (struct ArgusRecord *argus)
{
   long long retn = 0;
   struct ArgusAGRStruct *agr;

   ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

   if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL)
      retn = agr->act.mean;
   else
      retn = RaGetuSecDuration (argus);
  
   return (retn);
}

long long
RaGetuSecDuration (struct ArgusRecord *argus)
{
   long long retn = 0;
   int sec, usec;

   sec  = argus->argus_far.time.last.tv_sec  - argus->argus_far.time.start.tv_sec;
   usec = argus->argus_far.time.last.tv_usec - argus->argus_far.time.start.tv_usec;

   if (usec < 0) {
      sec--; usec += 1000000;
   }

   retn = (sec * 1000000) + usec;

   return (retn);
}

