#include <glib.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <sys/stat.h>

#include "asdutil.h"
#include "esound-auth.h"
#include "protocol-esound-spec.h"

gboolean esound_auth_cookie_available = FALSE;
guint8 esound_auth_cookie[ESD_KEY_LEN];

static gchar _cookie_path[PATH_MAX] = "";
static gboolean _remove_cookie = FALSE;

static void _set_cookie_path()
{
  gchar *h = getenv("ESDCOOKIE");
  
  if (h)
    strncpy(_cookie_path, h, sizeof(_cookie_path));
  else
    g_snprintf(_cookie_path, sizeof(_cookie_path), "%s/.esd_auth", g_get_home_dir());
}

void esound_auth_write_cookie()
{
  int fd;

  g_assert(esound_auth_cookie_available);

  _set_cookie_path();

  if ((fd = open(_cookie_path, O_WRONLY|O_CREAT, S_IRUSR|S_IWUSR)) >= 0)
    {
      gboolean b;
      
      b = atomic_write(fd, esound_auth_cookie, ESD_KEY_LEN) == ESD_KEY_LEN;
      close(fd);
      
      if (!b)
	{
	  g_message("Could not write esound cookie to '%s'.", _cookie_path);
	  unlink(_cookie_path);
	}
      else
	_remove_cookie = TRUE;
    }
  else
    g_message("Could not open esound cookie file '%s'.", _cookie_path);
}

void esound_auth_generate_cookie()
{
  int fd;
  gboolean done = FALSE;

  if ((fd = open("/dev/urandom", O_RDONLY)) >= 0)
    {
      if (atomic_read(fd, esound_auth_cookie, ESD_KEY_LEN) == ESD_KEY_LEN)
	done = TRUE;
      close(fd);
    }

  if (!done)
    {
      gulong i;
      int* p = (int*) esound_auth_cookie;

      srand(time(0));

      for (i = 0; i < ESD_KEY_LEN/sizeof(int); i++)
	*(p++) = rand();
    }
  
  esound_auth_cookie_available = TRUE;
}

void esound_auth_read_cookie()
{
  int fd;
  _set_cookie_path();
  
  if ((fd = open(_cookie_path, O_RDONLY)) >= 0)
    {
      if (atomic_read(fd, esound_auth_cookie, ESD_KEY_LEN) == ESD_KEY_LEN)
	{
	  esound_auth_cookie_available = TRUE;
	  _remove_cookie = FALSE;
	}

      close(fd);
    }
}

gboolean esound_auth_compare_cookie(guint8 *data)
{
  g_assert(data);
  
  esound_auth_init(FALSE);
  return memcmp(data, esound_auth_cookie, ESD_KEY_LEN) == 0;
}

void esound_auth_init(gboolean b)
{
  if (esound_auth_cookie_available)
    return;

  esound_auth_read_cookie();
  
  if (!b || esound_auth_cookie_available)
    return;

  esound_auth_generate_cookie();
  esound_auth_write_cookie();
}

void esound_auth_done()
{
  if (_remove_cookie)
    {
      _set_cookie_path();
      unlink(_cookie_path);
      esound_auth_cookie_available = FALSE;
    }
}
