// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __path_manager__
#define __path_manager__

// Manages directories and methods to access the files in it.

#include "Puma/Array.h"
#include "Puma/PathInfo.h"
#include "Puma/Filename.h"
#include "Puma/ErrorStream.h"
#include "Puma/PathIterator.h"
#include "Puma/ProjectFile.h"

namespace Puma {

class RegComp;
class Config;
class Unit;

class PathManager {

  ErrorStream     *_err;
  Array<PathInfo>  _paths;     // List of directories path.
  Array<RegComp*>  _protected; // List of regular patterns specifying 
                               // protected paths.
  PFMap _files;                // Maps the canonical filename of a project
                               // to its related information

  // find all files of a directory tree and add them to _files
  void traverse (const char *path);

  // find the canonical filename representation for a file
  bool canonFilename (Filename name, Filename &abs_name) const;
  
  // Add a new file to the project file list
  PFMConstIter addFile (const ProjectFile &file);

protected:
  // Get the destination directory for a given source directory.
  const char *getDestination (Filename file, ostream * = 0) const;
        
  // Called for every matched file.
  virtual void action (PathIterator &) {}
        
  // Add the separator '/' to the copy of the given string.
  char *addSeparator (const char *) const;

public:
  PathManager (ErrorStream &e) : _err (&e) {} 
  // Copy-constructor.
  PathManager (PathManager &pm) { _err = pm._err; join (pm); }
  virtual ~PathManager ();

  // Start the globbing. Let call action for every match.
  void glob (char *pattern = (char*) 0);

  // Iterate the contents of the paths.
  const char *iterate (PathIterator &) const;
        
  // Add a source and destination directories pair.
  virtual void addPath (const char *, const char * = (const char*) 0);

  // Add a new file to the project file list
  PFMConstIter addFile (Filename name);

  // Add a new file to the project file list with destination path
  PFMConstIter addFile (Filename name, Filename dest);

  // search an absolute file name in the project tree
  bool lookupFile (Filename abs_name, PFMConstIter &iter) const;

  // Set the destination directory of the given source directory.
  void setDest (const char *, const char *);

  // Add a protected-path pattern.
  void protect (const char *);

  // Get path information.
  long numPaths () const { return _paths.length (); }
  long numProts () const { return _protected.length (); }
  const char *src (long pos) const { return _paths.lookup (pos).src (); }
  const char *dest (long pos) const { return _paths.lookup (pos).dest (); }
  RegComp *prot (long pos) const { return _protected.lookup (pos); }

  // Return true if the given file or path is protected.
  bool isProtected (const char *file) const;

  // checks if a give file (by name) is a registered file of this path manager
  // In case of success (found!) the an iterator is returned, which can be
  // used to access more information about the file.
  bool isBelow (const char *file, PFMConstIter &iter) const;

  // Returns true if the given file is registered with this path manager
  virtual bool isBelow (const char *file) const;

  // Configure the project from the command line or a file.
  virtual void configure (const Config &);

  // Join the paths of the given manager with the paths 
  // of this path manager.
  virtual void join (PathManager &);
        
  // Get the error stream object.
  ErrorStream &err () const { return *_err; }

  // Get the path of a file in the destination directory.
  bool getDestinationPath (const char *filename, ostream &out) const;
        
};

inline bool PathManager::isBelow (const char *file) const { 
  PFMConstIter iter;
  return isBelow (file, iter);
}


} // namespace Puma

#endif /* __path_manager__ */
