// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __join_point_loc_h__
#define __join_point_loc_h__

#include <vector>
using std::vector;
#include <list>
using std::list;
#include <set>
using std::set;
#include <map>
using std::map;
#include <string>
using std::string;

#include "Puma/Unit.h"

#include "MatchExpr.h"
#include "SourceLoc.h"
#include "RepoXMLNode.h"
#include "JoinPointModelElement.h"

namespace Puma {
  class CTree; // pointcuts are currently represented as a CTree
}

class ModelTransformInfo {
public:
  virtual ~ModelTransformInfo () {}
};
class JoinPointPlan;


// forward declaration (for cyclic relations)
class JPL_Name;
class JPL_AdviceCode;
class JPL_Introduction;
class JPL_Order;

/**
 * \class JoinPointLoc
 * \brief the common base class of all join point location types
 * 
 * The class contains elements to store join point locations in a hierarchical
 * structure, associated source locations, etc.
 * In order to associate a weaving plan and parser-specific syntax information
 * two (almost) untyped pointers are included. They can be used by higher
 * layer.
 */
 
class JoinPointLoc : public JoinPointModelElement {
public:
  enum join_point_type {
    None            = 0x000000,
    FieldReference  = 0x000001,
    FieldAssignment = 0x000002,
    MethodCall      = 0x000004,
    Method          = 0x000008,
    Construction    = 0x000010,
    Destruction     = 0x000020,
    Code            = 0x00003f,
    Class           = 0x000100,
    Aspect          = 0x000200,
    Function        = 0x000400,
    Type            = 0x000800,
    Namespace       = 0x001000,
    AdviceCode      = 0x002000,
    ClassSlice      = 0x004000,
    Name            = 0x007f00,
    Any             = 0x007f3f,
    Introduction    = 0x008000,
    Order           = 0x010000,
    Advice          = 0x018000
  };

protected:
  
  /// parent in the hierarchical join point model structure
  JPL_Name *_parent;
  /// source code location(s) of this join point (shadow)
  set<SourceLoc> _source_locs;
  /// the signature of this jp as a string
  string _sig;
  /// the weaving plan for advised jps
  JoinPointPlan *_plan;
  /// parser-provided info for code transformation
  ModelTransformInfo *_trans;
  /// translation units in which this element was detected
  IdSet _tunits;
  
 public:

  JoinPointLoc () : _parent (0), _plan (0), _trans (0) {}
  JoinPointLoc (const JoinPointLoc &jpl) : _parent (0), _plan (0), _trans (0) {
    _sig = jpl._sig;
  }
  JoinPointLoc (RepoXMLNode jn, RepoXMLNode::iter &curr);
  virtual ~JoinPointLoc () { if (_trans) delete _trans; }

  void parent (JPL_Name *p);
  JPL_Name *parent () const { return _parent; }

  const IdSet &tunits () const { return _tunits; }
  IdSet &tunits () { return _tunits; }
  virtual const char *signature () const { return _sig.c_str (); }
  string filename () const;
  int line () const;
  typedef const set<SourceLoc> SSet;
  const set<SourceLoc> &source_locs () const { return _source_locs; }
  set<SourceLoc> &source_locs () { return _source_locs; }
  void source_loc (const SourceLoc &sl) { _source_locs.insert (sl); }
  JoinPointPlan *plan () const { return _plan; }
  void plan (JoinPointPlan *p) { _plan = p; }
  ModelTransformInfo *transform_info () const { return _trans; }
  void transform_info (ModelTransformInfo *trans) { _trans = trans; }

  virtual join_point_type type () const { return None; }
  virtual const char *type_str () const { return (const char*)0; }

  bool operator == (const JoinPointLoc &right) const {
    return type () == right.type () && _sig == right._sig;
  }
  bool operator != (const JoinPointLoc &right) const {
    return !(*this == right);
  }

  virtual void dump (int indent = 0) const;

  virtual RepoXMLNode make_xml (RepoXMLNode parent) const;
  
  /// create a join point loc tree from an XML document subtree
  static JoinPointLoc *create (RepoXMLNode node);
  static IdElementMap *_map;
  
  /// check whether this join point is seen by a specific tranlation unit
  virtual bool is_known_in_tunit (File *tunit);
};

class JPL_Type : public JoinPointLoc {
protected:
  mutable MatchSignature _match_sig; // calculated on demand
  void parse_match_signature () const {
    Location l;
    ErrorStream e (cerr);
    _match_sig.parse (e, l, signature ());
  }
public:
  JPL_Type (const string &sig) { _sig = sig; }
  join_point_type type () const { return Type; }
  virtual const char *type_str () const { return "type"; }
  const MatchSignature &match_signature () const {
    if (_match_sig.is_new ())
      parse_match_signature ();
    return _match_sig;
  }
};

class JPL_Name : public JoinPointLoc {
public:
  typedef list<const JoinPointLoc*> CList;

private:
  typedef std::map<string, JPL_Name*> NameMap;
  NameMap _name_map;
  CList _children;
  string _name;
  bool _is_built_in;

protected:
  mutable MatchSignature _match_sig; // calculated on demand
  virtual void parse_match_signature () const {
    Location l ("<no filename>", 0);
    ErrorStream e (cerr);
    _match_sig.parse (e, l, signature ());
  }

public:
  JPL_Name () : _is_built_in (false) {}
  JPL_Name (const JPL_Name &jpl) : JoinPointLoc (jpl), _name (jpl._name),
    _is_built_in (jpl._is_built_in) {}
  JPL_Name (RepoXMLNode node, RepoXMLNode::iter &curr);
  const CList &children () const { return _children; }
  void add_child (JoinPointLoc *child) {
    _children.push_back (child);
    if (child->type () & JoinPointLoc::Name) {
      _name_map.insert (NameMap::value_type (child->signature (),
        (JPL_Name*)child));
    }
  }
  void remove_child (const JoinPointLoc *child) {
    _children.remove (child);
    _name_map.erase (child->signature ());
  }
  void name (const string &name) { _name = name; }
  const string &name () const { return _name; }
  virtual void dump (int indent = 0) const;
  virtual RepoXMLNode make_xml (RepoXMLNode parent) const;
  bool is_root () const { return _sig == "::"; }
  bool is_built_in () const { return _is_built_in; }
  void is_built_in (bool b) { _is_built_in = b; }
  const MatchSignature &match_signature () const {
    if (_match_sig.is_new ())
      parse_match_signature ();
    return _match_sig;
  }
  JPL_Name *lookup (const string &name) {
    NameMap::const_iterator i = _name_map.find (name);
    return (i == _name_map.end ()) ? 0 : i->second;
  } 
};

class JPL_Namespace : public JPL_Name {
public:
  JPL_Namespace (const string &name) { _sig = name; }
  JPL_Namespace (const JPL_Namespace &jpl) : JPL_Name (jpl) {}
  JPL_Namespace (RepoXMLNode node, RepoXMLNode::iter &curr);
  join_point_type type () const { return Namespace; }
  virtual const char *type_str () const { return "namespace"; }
};

class JPL_Class : public JPL_Name {
protected:
  bool _intro_target;
  set<int> _derived_ids;
  set<int> _base_ids;
  
public:
  JPL_Class (const string &name) { _sig = name; }
  JPL_Class (const JPL_Class &jpl) : JPL_Name (jpl), _intro_target (jpl._intro_target) {}
  JPL_Class (RepoXMLNode node, RepoXMLNode::iter &curr);
  virtual RepoXMLNode make_xml (RepoXMLNode parent) const;
  
  join_point_type type () const { return Class; }
  virtual const char *type_str () const { return "class"; }
  void intro_target (bool i) { _intro_target = i; }
  bool intro_target () const { return _intro_target; }

  // get the list of base classes (IDs) => can be mapped with 'map'
  const set<int> &base_class_ids () const { return _base_ids; }
  
  // get the list of derived classes (IDs) => can be mapped with 'map'
  const set<int> &derived_class_ids () const { return _derived_ids; }
  
  // declare a class to be a base class of this class
  void base (JPL_Class *base) {
    _base_ids.insert (base->id ());
    base->_derived_ids.insert (id ());
  }

  // generate back links
  void generate_back_links () {
    for (set<int>::const_iterator i = _base_ids.begin ();
      i != _base_ids.end (); ++i)
      ((JPL_Class*)map (*i))->_derived_ids.insert (id ());
  }
};

class JPL_Aspect : public JPL_Class {
  list<JPL_AdviceCode*> _advice_codes;
  list<JPL_Introduction*> _intro_infos;
  list<JPL_Order*> _order_infos;
  
public:
  JPL_Aspect (const string &name) : JPL_Class (name) {}
  JPL_Aspect (const JPL_Aspect &jpl) : JPL_Class (jpl) {}
  JPL_Aspect (RepoXMLNode node, RepoXMLNode::iter &curr) :
    JPL_Class (node, curr) {}
  virtual join_point_type type () const { return Aspect; }
  virtual const char *type_str () const { return "aspect"; }
  
  /// the list of associated code advice nodes
  list<JPL_AdviceCode*> &advice_codes () { return _advice_codes; }
  
  /// the list of associated order advice nodes
  list<JPL_Order*> &order_infos () { return _order_infos; }
  
  /// the list of associated introduction nodes
  list<JPL_Introduction*> &intro_infos () { return _intro_infos; }
  
  /// collect the advice codes from this aspect and all base aspects
  void collect_advice_codes (list<JPL_AdviceCode*> &codes) const {
    for (list<JPL_AdviceCode*>::const_iterator i = _advice_codes.begin ();
      i != _advice_codes.end (); ++i)
      codes.push_back (*i);
    const set<int> &bases = base_class_ids ();
    for (set<int>::const_iterator i = bases.begin (); i != bases.end (); ++i) {
      JPL_Class *cls = (JPL_Class*)map (*i);
      if (cls->type () == JoinPointLoc::Aspect)
        ((JPL_Aspect*)cls)->collect_advice_codes (codes);
    }
  }
  
  /// collect the advice codes from this aspect and all base aspects
  void collect_intros (list<JPL_Introduction*> &intros) const {
    for (list<JPL_Introduction*>::const_iterator i = _intro_infos.begin ();
      i != _intro_infos.end (); ++i)
      intros.push_back (*i);
    const set<int> &bases = base_class_ids ();
    for (set<int>::const_iterator i = bases.begin (); i != bases.end (); ++i) {
      JPL_Class *cls = (JPL_Class*)map (*i);
      if (cls->type () == JoinPointLoc::Aspect)
        ((JPL_Aspect*)cls)->collect_intros (intros);
    }
  }
  
  /// collect the advice codes from this aspect and all base aspects
  void collect_orders (list<JPL_Order*> &orders) const {
    for (list<JPL_Order*>::const_iterator i = _order_infos.begin ();
      i != _order_infos.end (); ++i)
      orders.push_back (*i);
    const set<int> &bases = base_class_ids ();
    for (set<int>::const_iterator i = bases.begin (); i != bases.end (); ++i) {
      JPL_Class *cls = (JPL_Class*)map (*i);
      if (cls->type () == JoinPointLoc::Aspect)
        ((JPL_Aspect*)cls)->collect_orders (orders);
    }
  }
};

class JPL_Function : public JPL_Name {
public:
  enum FunctionType {
    NON_MEMBER, STATIC_MEMBER, MEMBER, VIRTUAL_MEMBER, CONSTRUCTOR, DESTRUCTOR,
    UNKNOWN
  };

private:
  FunctionType _function_type;
  vector<JPL_Type *> _arg_types;
  JPL_Type *_result_type;

protected:
  virtual void parse_match_signature () const {
    JPL_Name::parse_match_signature ();
    if (is_virtual ())
      _match_sig.declare_virtual_function ();
  }
  
public:
  JPL_Function (const string &n, const string &s) :
    _function_type (UNKNOWN), _result_type (0) {
    name (n); _sig = s;
  }
  JPL_Function (const JPL_Function &jpl) :
    JPL_Name (jpl), _function_type (jpl._function_type) {}
  JPL_Function (RepoXMLNode node, RepoXMLNode::iter &curr);
  ~JPL_Function ();

  // set the function type
  void function_type (FunctionType ft) { _function_type = ft; }

  // helper functions to check the function type
  bool is_non_member () const        { return _function_type == NON_MEMBER; }
  bool is_static_member () const     { return _function_type == STATIC_MEMBER; }
  bool is_non_static_member () const { return _function_type == MEMBER ||
                                              _function_type == VIRTUAL_MEMBER; }
  bool is_virtual () const           { return _function_type == VIRTUAL_MEMBER; }
  bool is_constructor () const       { return _function_type == CONSTRUCTOR; }
  bool is_destructor () const        { return _function_type == DESTRUCTOR; }

  // argument handling
  int arg_count () const {
    return (int)_arg_types.size ();
  }
  const JPL_Type &arg_type (int i) const {
    assert (i >= 0 && i < arg_count ());
    return *_arg_types[i]; 
  }
  void add_arg_type (JPL_Type *type) { _arg_types.push_back (type); }
  void reset_types () {
    for (int i = 0; i < arg_count (); i++)
      delete _arg_types[i];
    _arg_types.clear ();
    if (_result_type) {
      delete _result_type;
      _result_type = 0;
    }
  }
  
  // manage the result type
  void result_type (JPL_Type *type) {
    if (_result_type) delete _result_type;
    _result_type = type;
  }
  const JPL_Type &result_type () const { return *_result_type; }
  
  join_point_type type () const { return Function; }
  virtual const char *type_str () const { return "function"; }
  bool has_same_name_and_args (const JPL_Function &func) const;
};

class JPL_ClassSlice : public JPL_Name {
public:
  enum class_slice_type { CS_OLD_BASE, CS_OLD_OTHER, CS_NORMAL, CS_ERROR };
  
private:
  class_slice_type _type;
  CProtection::Type _prot;
  Unit _pattern;
  
public:
  JPL_ClassSlice (const string &name) :
    _type (CS_NORMAL), _prot (CProtection::PROT_PRIVATE) {
    _sig = name;
  }
  JPL_ClassSlice (RepoXMLNode node, RepoXMLNode::iter &curr);
  JPL_ClassSlice (const JPL_ClassSlice &jpl) : _type (jpl._type),
    _prot (jpl._prot), _pattern (jpl._pattern) {}
  virtual RepoXMLNode make_xml (RepoXMLNode parent) const;
    
  void slice_type (class_slice_type t) { _type = t; }
  class_slice_type slice_type () const { return _type; }
  void prot (CProtection::Type p) { _prot = p; }
  CProtection::Type prot () const { return _prot; }
  const Unit &pattern () const { return _pattern; }
  Unit &pattern () { return _pattern; }
  virtual join_point_type type () const { return ClassSlice; }
  virtual const char *type_str () const {
    switch (_type) {
      case CS_OLD_BASE:  return "slice-dep-base";
      case CS_OLD_OTHER: return "slice-dep-member";
      case CS_NORMAL:    return "slice-class";
      default:           return "error";
    }
  }
};

/**
 * \class JPL_Code
 * \brief the common base class of all code join-point locations
 */
 
class JPL_Code : public JoinPointLoc {
protected:
  /// local id for this jp inside the current function (or -1)
  int _lid;
public:
  JPL_Code () : _lid (0) {}
  JPL_Code (const JPL_Code &jpl) : JoinPointLoc (jpl), _lid (jpl._lid) {}
  JPL_Code (RepoXMLNode node, RepoXMLNode::iter &curr) :
    JoinPointLoc (node, curr), _lid (0) {}

  virtual bool is_pseudo () const { return false; }
  
  // get the local id within the current function
  int lid () const { return _lid; }

  // set the local id within the current function
  void lid (int lid) { _lid = lid; }

  // the lexical scope of this code join point as needed by 'within'
  virtual JPL_Name *lexical_scope () const = 0;
  
  // argument handling
  virtual int arg_count () const = 0;
  virtual const JPL_Type &arg_type (int i) const = 0;
  
  // manage the result type
  virtual const JPL_Type &result_type () const = 0;
  
  // interface needed to generate proper proceed code
  virtual bool proceed_needs_args () const {
    return arg_count () > 0;
  }
  virtual bool proceed_needs_result () const {
    return result_type () != JPL_Type ("void");
  }
};

class JPL_Method : public JPL_Code {
public:
  JPL_Method () { _lid = -1; }
  JPL_Method (const JPL_Method &jpl) : JPL_Code (jpl) {}
  JPL_Method (RepoXMLNode node, RepoXMLNode::iter &curr) :
    JPL_Code (node, curr) {}
  join_point_type type () const { return Method; }
  virtual const char *type_str () const { return "exec"; }

  // argument handling
  virtual int arg_count () const { return function ()->arg_count (); }
  virtual const JPL_Type &arg_type (int i) const { return function ()->arg_type (i); }
  
  // manage the result type
  const JPL_Type &result_type () const { return function ()->result_type (); }
  
  // the lexical scope of this code join point as needed by 'within'
  virtual JPL_Name *lexical_scope () const { return (JPL_Name*)parent ()->parent (); }

  // the executed function
  JPL_Function *function () const { return (JPL_Function*)parent (); }
  
  // the signature of the exec join point (= signature of the function)
  virtual const char *signature () const { return function ()->signature (); }
  
  /// check whether this join point is seen by a specific tranlation unit
  virtual bool is_known_in_tunit (File *tunit) {
    return function ()->is_known_in_tunit (tunit);
  }
};

class JPL_MethodCall : public JPL_Code {
  int _target_id;

  vector<JPL_Type *> _arg_types;
  JPL_Type *_result_type;
  
public:
  JPL_MethodCall () { _lid = -1; }
  JPL_MethodCall (const JPL_MethodCall &jpl) : JPL_Code (jpl),
    _target_id (jpl._target_id) {}
  JPL_MethodCall (RepoXMLNode node, RepoXMLNode::iter &curr) :
    JPL_Code (node, curr) {
    _target_id = node.get_int_prop ("target");
  }
  ~JPL_MethodCall ();

  virtual bool is_pseudo () const { return !parent (); }

  JPL_Function *target_function () const {
    return (JPL_Function*)map (_target_id);
  }
  void target_function (JPL_Function *f) {
    _target_id = f->id ();
    _sig = f->signature ();
  }
  
  // the signature of the exec join point (= signature of the function)
  // TODO: better create the signature from the arg and result types
  virtual const char *signature () const { return target_function ()->signature (); }
  
  // for internal use only:
  int target_id () const { return _target_id; }
  
  JPL_Function *caller_function () const {
    return (parent () && parent ()->type () == JoinPointLoc::Function) ?
      (JPL_Function*)parent () : 0;
  }

  // argument handling
  virtual int arg_count () const {
    return (int)_arg_types.size ();
  }
  virtual const JPL_Type &arg_type (int i) const {
    assert (i >= 0 && i < arg_count ());
    return *_arg_types[i]; 
  }
  void add_arg_type (JPL_Type *type) { _arg_types.push_back (type); }
  
  // manage the result type
  void result_type (JPL_Type *type) { _result_type = type; }
  virtual const JPL_Type &result_type () const { return *_result_type; }

  join_point_type type () const { return MethodCall; }
  virtual const char *type_str () const { return "call"; }

  // the lexical scope of this code join point as needed by 'within'
  virtual JPL_Name *lexical_scope () const { return (JPL_Name*)parent (); }

  // save as XML
  virtual RepoXMLNode make_xml (RepoXMLNode parent) const;
};

class JPL_Construction : public JPL_Code {
public:
  JPL_Construction () { _lid = -1; }

  JPL_Construction (RepoXMLNode node, RepoXMLNode::iter &curr) :
    JPL_Code (node, curr) {}
    
  JPL_Construction (const JPL_Construction &jpl) : JPL_Code (jpl) {}

  join_point_type type () const { return Construction; }
  virtual const char *type_str () const { return "construction"; }

  // the executed constructor function
  JPL_Function *function () const { return (JPL_Function*)parent (); }

  // the lexical scope of this code join point as needed by 'within'
  virtual JPL_Name *lexical_scope () const { return (JPL_Name*)parent ()->parent (); }

  // argument handling
  virtual int arg_count () const { return function ()->arg_count (); }
  virtual const JPL_Type &arg_type (int i) const { return function ()->arg_type (i); }
  
  // manage the result type
  const JPL_Type &result_type () const { return function ()->result_type (); }
  
  // the signature of the exec join point (= signature of the function)
  virtual const char *signature () const { return function ()->signature (); }
  
  /// check whether this join point is seen by a specific tranlation unit
  virtual bool is_known_in_tunit (File *tunit) {
    return function ()->is_known_in_tunit (tunit);
  }
};

class JPL_Destruction : public JPL_Code {
public:
  JPL_Destruction () { _lid = -1; }

  JPL_Destruction (RepoXMLNode node, RepoXMLNode::iter &curr) :
    JPL_Code (node, curr) {}

  JPL_Destruction (const JPL_Destruction &jpl) : JPL_Code (jpl) {}

  join_point_type type () const { return Destruction; }
  virtual const char *type_str () const { return "destruction"; }

  // the executed destructor function
  JPL_Function *function () const { return (JPL_Function*)parent (); }

  // the lexical scope of this code join point as needed by 'within'
  virtual JPL_Name *lexical_scope () const { return (JPL_Name*)parent ()->parent (); }

  // argument handling
  virtual int arg_count () const { return function ()->arg_count (); }
  virtual const JPL_Type &arg_type (int i) const { return function ()->arg_type (i); }
  
  // manage the result type
  const JPL_Type &result_type () const { return function ()->result_type (); }
  
  // the signature of the exec join point (= signature of the function)
  virtual const char *signature () const { return function ()->signature (); }
  
  /// check whether this join point is seen by a specific tranlation unit
  virtual bool is_known_in_tunit (File *tunit) {
    return function ()->is_known_in_tunit (tunit);
  }
};

class JPL_Advice {
  CTree *_pce;

public:
  JPL_Advice (CTree *pce = 0) : _pce (pce) {}
  CTree *expr () const { return _pce; }
  void expr (CTree *pce) { _pce = pce; }
};

class JPL_AdviceCode : public JPL_Advice, public JPL_Function {
public:
  enum advice_type { ADVICE_BEFORE, ADVICE_AROUND, ADVICE_AFTER };

private:
  CTree *_pce;
  advice_type _type;

public:
  JPL_AdviceCode (const string &n, const string &s) :
    JPL_Function (n, s) {
    const char *nm = n.c_str ();
    if (strstr (nm, "before"))
      _type = ADVICE_BEFORE;
    else if (strstr (nm, "after"))
      _type = ADVICE_AFTER;
    else
      _type = ADVICE_AROUND;
  }
  JPL_AdviceCode (const JPL_AdviceCode &jpl) : JPL_Advice (jpl), JPL_Function (jpl),
    _pce (jpl._pce), _type (jpl._type) {}
  JPL_AdviceCode (RepoXMLNode node, RepoXMLNode::iter &curr);
  virtual RepoXMLNode make_xml (RepoXMLNode parent) const;

  virtual join_point_type type () const { return AdviceCode; }
  virtual const char *type_str () const {
    switch (_type) {
      case ADVICE_BEFORE: return "advice-before";
      case ADVICE_AROUND: return "advice-around";
      case ADVICE_AFTER:  return "advice-after";
    }
    return "error"; // just to suppress warnings
  }

  advice_type type () { return _type; }
};

class JPL_Introduction : public JPL_Advice, public JoinPointLoc {
  int _slice_id;
public:
  JPL_Introduction () : _slice_id (-1) {
    _sig = "<no sig>";
  }
  JPL_Introduction (RepoXMLNode node, RepoXMLNode::iter &curr);
  JPL_Introduction (const JPL_Introduction &jpl) : JPL_Advice (jpl), _slice_id (0) {}
  virtual RepoXMLNode make_xml (RepoXMLNode parent) const;

  void parent (JPL_Aspect *ai) {
    JoinPointLoc::parent (ai);
    ai->intro_infos ().push_back (this);
  }
  using JoinPointLoc::parent;
  
  int slice_id () const { return _slice_id; }

  virtual join_point_type type () const { return Introduction; }
  const char *type_str () const { return "intro"; }
  void introduced (JPL_ClassSlice *s) { _slice_id = s->id (); }
  JPL_ClassSlice *introduced () const { return (JPL_ClassSlice*)map (_slice_id); }
};

class JPL_Order : public JPL_Advice, public JoinPointLoc {
  list<CTree*> _pces;
public:
  JPL_Order () {
    expr (0);
    _sig = "<no sig>";
  }
  void parent (JPL_Aspect *ai) {
    JoinPointLoc::parent (ai);
    ai->order_infos ().push_back (this);
  }
  using JoinPointLoc::parent;

  virtual join_point_type type () const { return Order; }
  virtual const char *type_str () const { return "order"; }

  void add_pce (CTree *pce) { _pces.push_back (pce); }
  const list<CTree*> &pces () const { return _pces; }
};

#endif // __join_point_loc_h__
