// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "JoinPointModel.h"
#include "RepoXMLDoc.h"
#include "version.h"

void JoinPointModel::clear () {
  for (JPLL::iterator iter = _elements.begin ();
       iter != _elements.end (); ++iter)
    delete *iter;
  _elements.clear ();
  _id_mgr.reset_id ();
  _root_namespace = 0;
  _files.clear ();
  _id_element_map.clear ();
}

// save the join point model in a file
JoinPointModel::FileOpResult JoinPointModel::save (int fd,
  const string &filename) {
  // create an XML DOM
  RepoXMLDoc doc;
  doc.create ("ac-project");
  doc.root ().set_str_prop ("version", ac_version ());

  RepoXMLNode files = doc.root ().make_child ("files");
  // first insert all files
  for (list<File>::iterator i = _files.begin (); i != _files.end (); ++i)
    i->to_xml (files);

  // save the tree with all namespaces, classes, functions, and plans etc.
  if (root_namespace ())
    root_namespace ()->make_xml (doc.root ());

  // save the file
  if (!doc.save_fd (fd))
    return JPM_SAVE_ERR;
  return JPM_OK;
}

// discard the model and load new elements from a file
JoinPointModel::FileOpResult JoinPointModel::load (int fd,
  const string &filename) {

  // clear the model
  clear ();
  
  // load the XML file
  RepoXMLDoc doc;
  if (!doc.load_fd (fd, filename.c_str ()) ||
      !doc.root ().has_name ("ac-project"))
    return JPM_LOAD_ERR;

  // check the model version
  string version = doc.root ().get_str_prop ("version");
  if (version != ac_version ())
    return JPM_VERSION;

  // load file elements
  RepoXMLNode::iter curr = doc.root ().first_child ();
  if (curr == doc.root ().end_child ()) {
    return JPM_LOAD_ERR;
  }

  RepoXMLNode files = *curr;
  if (!files.has_name ("files"))
    return JPM_LOAD_ERR;
    
  RepoXMLNode::iter curr_file = files.first_child ();
  while (curr_file != files.end_child ()) {
    RepoXMLNode fn = *curr_file;
    _files.push_back (File ());
    File &file = _files.back ();
    file.map (&_id_element_map);
    file.from_xml (fn);
    _id_mgr.update (file.id ());
    ++curr_file;
  }
    
  // load tree with name and code join points
  ++curr;
  if (curr == doc.root ().end_child ()) {
    return JPM_LOAD_ERR;
  }
  JoinPointLoc::_map = &_id_element_map;
  JoinPointLoc *root_jpl = JoinPointLoc::create (*curr);
  JoinPointLoc::_map = 0;
  if (!root_jpl || !root_jpl->type () == JoinPointLoc::Namespace) {
    return JPM_LOAD_ERR;
  }

  _root_namespace = (JPL_Namespace*)root_jpl; 
  register_tree (_root_namespace);

  // Select all classes and fix pointers to base and derived classes
  Selection classes;
  select ((JoinPointLoc::join_point_type)(JoinPointLoc::Class|JoinPointLoc::Aspect), classes);
  for (Selection::iterator bi = classes.begin (); bi != classes.end (); ++bi) {
    JPL_Class *cls = (JPL_Class*)*bi;
    cls->generate_back_links ();
  }

  return JPM_OK;
}

void JoinPointModel::register_tree (JoinPointLoc *loc) {

  // register the element
  if (loc->type () & JoinPointLoc::Name) {
    JPL_Name *jpl_name = (JPL_Name*)loc;
    register_name (jpl_name, false);
    // for name join-points also register the children
    const JPL_Name::CList &children = jpl_name->children ();
    for (JPL_Name::CList::const_iterator i = children.begin (); i != children.end (); ++i) {
      register_tree ((JoinPointLoc*)*i);
    }
  }
  else
    register_entry (loc, false);
}

// reconcile elements and IDs of the project join point model (this) and
// a translation unit's join point model
void JoinPointModel::reconcile (JoinPointModel &tunit_jpm) {

  // create a searchable map of project files
  typedef map<string, File*> NameFileMap;
  NameFileMap prj_file_map;
  for (list<File>::iterator i = _files.begin (); i != _files.end (); ++i) {
    File &prj_file = *i;
    prj_file_map.insert (NameFileMap::value_type (prj_file.name (), &prj_file));
  }

  // first make sure that the main translation unit file is registered
  File *tunit_main_file = tunit_jpm._tunit_file;
  assert (tunit_main_file);
  File *prj_main_file = 0;
  NameFileMap::iterator mi = prj_file_map.find (tunit_main_file->name ());
  if (mi != prj_file_map.end ()) {
    prj_main_file = mi->second;
  }
  else {
    prj_main_file = register_file (tunit_main_file->name (),
      tunit_main_file->len (), true);
  }
  // reconcile both model elements
  reconcile (tunit_main_file, prj_main_file);
  
  // check all other entries in the translation unit file list
  list<File> &tunit_files = tunit_jpm._files;
  for (list<File>::iterator i = tunit_files.begin (); i != tunit_files.end (); ++i) {
    File *tunit_file = &(*i);
    // skip the main translation unit file
    if (tunit_file == tunit_main_file)
      continue;
      
    // find the corresponding project model file
    NameFileMap::iterator fi = prj_file_map.find (tunit_file->name ());
    File *prj_file;
    if (fi != prj_file_map.end ()) {
      // file from the current translation unit already known in the project repository
      // take the found File object
      prj_file = fi->second;
      // mark that this file is needed for the current translation unit
      prj_file->included_by (prj_main_file);
    }
    else {
      // a file which has been unknown in the project repository
      // create a new file entry in the project repository
      prj_file = register_file (tunit_file->name (), tunit_file->len (), false);
      prj_file->included_by (prj_main_file);
    }

    // reconcile both elements
    reconcile (tunit_file, prj_file);
  }
  
  // fix project file list
  list<list<File>::iterator> files_to_delete;
  for (list<File>::iterator i = _files.begin (); i != _files.end (); ++i) {
    File &prj_file = *i;
    if (prj_file.is_ref ()) {
      // reset the 'ref' bit
      prj_file.ref (false);
    }
    else {
      // the file is not relevant for this translation unit
      prj_file.not_included_by (prj_main_file);
      // check if the file is no longer needed
      if (prj_file.is_in_no_tunit ())
        files_to_delete.push_back (i);
    }
  }

  // unify the root namespace elements first
  unify (tunit_jpm.root_namespace ());

  // unify all other join points
  JPLL &locs = tunit_jpm._elements;
  for (JPLL::iterator li = locs.begin (); li != locs.end (); ++li) {
    if (!((*li)->type () & JoinPointLoc::Code))
      unify (*li);
  }
  
//  // delete join points that are no longer needed
//  for (int i = 0; i < NO; i++) {
//    list<JPLL::iterator> to_delete;
//    JPLL &locs = _elements[i];
//    for (JPLL::iterator li = locs.begin (); li != locs.end (); ++li) {
//      JoinPointLoc *loc = *li;
//      if (loc->is_known_in_tunit ((File*)tunit_jpm._tunit_file->partner ()) &&
//          !loc->is_ref ()) {
//        loc->parent ()->remove_child (loc);
//        to_delete.push_back (li);
//      }
//    }
//    for (list<JPLL::iterator>::iterator ii = to_delete.begin ();
//      ii != to_delete.end (); ++ii) {
//      JPLL::iterator iter = *ii;
//      delete **ii;
//      locs.erase (iter);
//    }
//  }
  
  // delete files to delete
  for (list<list<File>::iterator>::iterator i = files_to_delete.begin ();
    i != files_to_delete.end (); ++i) {
    _files.erase (*i);
  }
}

void JoinPointModel::unify (JoinPointLoc *tunit_loc) {
  // return if this join point is already unified (has a "partner")
  if (tunit_loc->partner ())
    return;
    
  // ignore pseudo join points from the tunit model in the project model
  // (pseudo join points don't have children => no problem to skip them)
  if (tunit_loc->type () & JoinPointLoc::Code &&
      ((JPL_Code*)tunit_loc)->is_pseudo ())
    return;
    
  // unify the parent if this hasn't been done already
  // (as the root namespace was explicitly unified first, this will terminate)
  JPL_Name *tunit_parent = tunit_loc->parent ();
  if (tunit_parent) { // root doesn't have a parent!
    if (!tunit_parent->partner ())
      unify (tunit_parent);
    
    // get the parent in the project model
    JPL_Name *prj_parent = (JPL_Name*)tunit_parent->partner ();
  
    if (!prj_parent) {
      cout << "could not unify parent of " << tunit_loc->signature () << endl;
      return;
    }
  
   // now reconcile the element
    if (tunit_loc->type () & JoinPointLoc::Name)
      reconcile ((JPL_Name*)tunit_loc);
  }
  else {
    // reconcile the root namespace
    reconcile (tunit_loc, root_namespace ());
  }
}

void JoinPointModel::reconcile (JoinPointModelElement *tunit_elem,
  JoinPointModelElement *prj_elem) {

  // mark the project model element as referenced (needed)
  prj_elem->ref ();
    
  // copy the id of the project model element into the translation unit element
  tunit_elem->assigned_id (prj_elem->id ());

  // rember the partner binding
  tunit_elem->partner (prj_elem);
  prj_elem->partner (tunit_elem);
}

void JoinPointModel::reconcile (JoinPointLoc *tunit_loc, JoinPointLoc *prj_loc) {

  reconcile ((JoinPointModelElement*)tunit_loc, (JoinPointModelElement*)prj_loc);
  
  // adapt and copy the tunits
  const IdSet &tunits = tunit_loc->tunits ();
  for (IdSet::const_iterator i = tunits.begin (); i != tunits.end (); ++i) {
    int tunit_id = *i;
    File *tunit = (File*)tunit_loc->map (tunit_id);
    prj_loc->tunits ().insert (tunit->partner ()->id ());
  }
}

void JoinPointModel::reconcile (JPL_Name *tunit_name) {
  
  // lookup the correspoding name in the project repository
  JPL_Name *tunit_parent = tunit_name->parent ();
  JPL_Name *prj_parent   = (JPL_Name*)tunit_parent->partner ();
  JPL_Name *prj_name     = prj_parent->lookup (tunit_name->signature ());
  
  if (!prj_name) {
    // not found! It has to be created.
    if (tunit_name->type () == JoinPointLoc::Namespace) {
      JPL_Namespace *tunit_namespace = (JPL_Namespace*)tunit_name;
      JPL_Namespace *name = new JPL_Namespace (*tunit_namespace);
      register_name (name);
      prj_name = name;
    }
    else if (tunit_name->type () == JoinPointLoc::Class) {
      JPL_Class *tunit_class = (JPL_Class*)tunit_name;
      JPL_Class *name = new JPL_Class (*tunit_class);
      register_name (name);
      prj_name = name;
    }
    else if (tunit_name->type () == JoinPointLoc::Aspect) {
      JPL_Aspect *tunit_aspect = (JPL_Aspect*)tunit_name;
      JPL_Aspect *name = new JPL_Aspect (*tunit_aspect);
      register_name (name);
      prj_name = name;
    }
    else if (tunit_name->type () == JoinPointLoc::Function) {
      JPL_Function *tunit_function= (JPL_Function*)tunit_name;
      JPL_Function *name = new JPL_Function (*tunit_function);
      register_name (name);
      prj_name = name;
    }
    else if (tunit_name->type () == JoinPointLoc::AdviceCode) {
      JPL_AdviceCode *tunit_advice_code = (JPL_AdviceCode*)tunit_name;
      JPL_AdviceCode *name = new JPL_AdviceCode (*tunit_advice_code);
      register_name (name);
      prj_name = name;
    }

    if (prj_name) {
      // set the parent node of the new node
      prj_name->parent ((JPL_Name*)tunit_name->parent ()->partner ());
    }
  }
  
  // temporarily insert as long as not all model elements can be copied
  if (!prj_name) {
    return;
  }

  // copy the source locations into the new node
  copy_source_locs (tunit_name, prj_name);

  // reconcile both elements
  reconcile (tunit_name, prj_name);
  
  // reconcile all non-name child join points of these two elements
  reconcile_children (tunit_name, prj_name);
}

struct LocDescriptor {
  SourceLoc _loc;
  LocDescriptor (const SourceLoc &loc) : _loc (loc) {}
  bool loc_less (const SourceLoc &r) const {
    return _loc.file_id () < r.file_id () ? true :
      (_loc.line () < r.line () ? true : (_loc.len () < r.len ()));
  }
  bool loc_equal (const SourceLoc &r) const {
    return _loc.file_id () == r.file_id () && _loc.line () == r.line () &&
      _loc.len () == r.len ();
  }
};
  
struct CallDescriptor : public LocDescriptor {
  string _target;
  JPL_MethodCall *_call;
  
  CallDescriptor (const SourceLoc &loc, const string &target,
    JPL_MethodCall *call) :
    LocDescriptor (loc), _target (target), _call (call) {}
  bool operator < (const CallDescriptor &r) const {
    return loc_equal (r._loc) ? (_target < r._target) : loc_less (r._loc);
  }
};

struct IntroDescriptor : public LocDescriptor {
  string _slice;
  JPL_Introduction *_intro;
  
  IntroDescriptor (const SourceLoc &loc, const string &s,
    JPL_Introduction *intro) :
    LocDescriptor (loc), _slice (s), _intro (intro) {}
  bool operator < (const IntroDescriptor &r) const {
    return loc_equal (r._loc) ? (_slice < r._slice) : loc_less (r._loc);
  }
};

struct ClassSliceDescriptor : public LocDescriptor {
  JPL_ClassSlice *_slice;
  ClassSliceDescriptor (const SourceLoc &loc, JPL_ClassSlice *s) :
    LocDescriptor (loc), _slice (s) {}
  bool operator < (const ClassSliceDescriptor &r) const {
    return loc_less (r._loc);
  }
};

void JoinPointModel::reconcile_children (JPL_Name *tunit_name, JPL_Name *prj_name) {

  // collect all code join points that are children of 'prj_name'
  JPL_Code *exec_cons_dest = 0;
  multiset<CallDescriptor> calls;
  multiset<IntroDescriptor> intros;
  multiset<ClassSliceDescriptor> class_slices;
  const JPL_Name::CList &prj_children = prj_name->children ();
  for (JPL_Name::CList::const_iterator i = prj_children.begin ();
    i != prj_children.end (); ++i) {
    if ((*i)->type () == JoinPointLoc::Method ||
        (*i)->type () == JoinPointLoc::Construction ||
        (*i)->type () == JoinPointLoc::Destruction) {
      exec_cons_dest = (JPL_Code*)*i;
    }
    else if ((*i)->type () == JoinPointLoc::MethodCall) {
      JPL_MethodCall *call = (JPL_MethodCall*)*i;
      const SourceLoc &loc = *call->source_locs ().begin ();
      string target  = call->target_function ()->signature ();
      calls.insert (CallDescriptor (loc, target, call));
    }
//    else if ((*i)->type () == JoinPointLoc::Introduction) {
//      JPL_Introduction *intro = (JPL_Introduction*)*i;
//      const SourceLoc &loc = *intro->source_locs ().begin ();
//      string sl = intro->introduced ()->signature ();
//      intros.insert (IntroDescriptor (loc, sl, intro));
//    }
    else if ((*i)->type () == JoinPointLoc::ClassSlice) {
      JPL_ClassSlice *class_slice = (JPL_ClassSlice*)*i;
      const SourceLoc &loc = *class_slice->source_locs ().begin ();
      class_slices.insert (ClassSliceDescriptor (loc, class_slice));
    }
  }
  
  // check each code join point from the translation unit model now
  const JPL_Name::CList &children = tunit_name->children ();
  for (JPL_Name::CList::const_iterator i = children.begin ();
    i != children.end (); ++i) {
    JoinPointLoc *prj_elem   = 0;

    // check if the join point is known in the project repository
    JoinPointLoc *tunit_elem = (JoinPointLoc*)*i;
    // ignore pseudo join points
    if (tunit_elem->type () & JoinPointLoc::Code &&
        ((JPL_Code*)tunit_elem)->is_pseudo ())
      continue;

    if (tunit_elem->type () == JoinPointLoc::Method ||
        tunit_elem->type () == JoinPointLoc::Construction ||
        tunit_elem->type () == JoinPointLoc::Destruction) {
      prj_elem = exec_cons_dest;
      exec_cons_dest = 0;
    }
    else if (tunit_elem->type () == JoinPointLoc::MethodCall) {
      JPL_MethodCall *call = (JPL_MethodCall*)tunit_elem;
      const SourceLoc &loc = *call->source_locs ().begin ();
      string target  = call->target_function ()->signature ();
      int file_id = call->map (loc.file_id ())->partner ()->id ();
      multiset<CallDescriptor>::iterator prj_iter =
        calls.find (CallDescriptor (SourceLoc (file_id,
          loc.line (), loc.len (), loc.kind ()), target, call));
      if (prj_iter != calls.end ()) {
        prj_elem = prj_iter->_call;
        calls.erase (prj_iter);
      }
    }
//    else if (tunit_elem->type () == JoinPointLoc::Introduction) {
//      JPL_Introduction *intro = (JPL_Introduction*)tunit_elem;
//      const SourceLoc &loc = *intro->source_locs ().begin ();
//      cout << "searching intro " << endl;
//      loc.print (1);
//      string sl  = intro->introduced ()->signature ();
//      multiset<IntroDescriptor>::iterator prj_iter =
//        intros.find (IntroDescriptor (SourceLoc ((File*)loc.file ()->partner (),
//          loc.line (), loc.len (), loc.kind ()), sl, intro));
//      if (prj_iter != intros.end ()) {
//        prj_elem = prj_iter->_intro;
//        cout << "found intro " << prj_elem->id () << endl;
//        intros.erase (prj_iter);
//      }
//    }
    else if (tunit_elem->type () == JoinPointLoc::ClassSlice) {
      JPL_ClassSlice *class_slice = (JPL_ClassSlice*)tunit_elem;
      const SourceLoc &loc = *class_slice->source_locs ().begin ();
      int file_id = class_slice->map (loc.file_id ())->partner ()->id ();
      multiset<ClassSliceDescriptor>::iterator prj_iter =
        class_slices.find (ClassSliceDescriptor (
        SourceLoc (file_id, loc.line (), loc.len (),
        loc.kind ()), class_slice));
      if (prj_iter != class_slices.end ()) {
        prj_elem = prj_iter->_slice;
        class_slices.erase (prj_iter);
      }
    }
  
    // the code join point is unknown, a new one has to be created
    if (!prj_elem) {
      if (tunit_elem->type () == JoinPointLoc::Method) {
        JPL_Method *tunit_method = (JPL_Method*)tunit_elem;
        JPL_Method *prj_method = new JPL_Method (*tunit_method);
        register_elem (prj_method);
        prj_elem = prj_method;
      }
      else if (tunit_elem->type () == JoinPointLoc::Construction) {
        JPL_Construction *tunit_construction = (JPL_Construction*)tunit_elem;
        JPL_Construction *prj_construction = new JPL_Construction (*tunit_construction);
        register_elem (prj_construction);
        prj_elem = prj_construction;
      }
      else if (tunit_elem->type () == JoinPointLoc::Destruction) {
        JPL_Destruction *tunit_destruction = (JPL_Destruction*)tunit_elem;
        JPL_Destruction *prj_destruction = new JPL_Destruction (*tunit_destruction);
        register_elem (prj_destruction);
        prj_elem = prj_destruction;
      }
      else if (tunit_elem->type () == JoinPointLoc::MethodCall) {
        JPL_MethodCall *tunit_call = (JPL_MethodCall*)tunit_elem;
        JPL_MethodCall *prj_call = new JPL_MethodCall (*tunit_call);
        // make sure that the target function is unified before we use it
        unify (tunit_call->target_function ());
        if (tunit_call->target_function ()->partner ())
          prj_call->target_function ((JPL_Function*)tunit_call->target_function ()->partner ());
        else
          cout << "target function " << tunit_call->target_function ()->signature () << " not unified" << endl;
        register_elem (prj_call);
        prj_elem = prj_call;
      }
//      else if (tunit_elem->type () == JoinPointLoc::Introduction) {
//        JPL_Introduction *tunit_intro = (JPL_Introduction*)tunit_elem;
//        JPL_Introduction *prj_intro = new JPL_Introduction (*tunit_intro);
//        // make sure that the introducted slice is unified before we use it
////        unify (tunit_intro->introduced ());
////        if (tunit_intro->introduced ()->partner ())
////          prj_intro->introduced ((JPL_ClassSlice*)tunit_intro->introduced ()->partner ());
//        register_elem (prj_intro);
//        cout << "created intro " << prj_intro->id () << endl;
//        prj_elem = prj_intro;
//      }
      else if (tunit_elem->type () == JoinPointLoc::ClassSlice) {
        JPL_ClassSlice *tunit_class_slice = (JPL_ClassSlice*)tunit_elem;
        JPL_ClassSlice *prj_class_slice = new JPL_ClassSlice (*tunit_class_slice);
        register_elem (prj_class_slice);
        prj_elem = prj_class_slice;
      }

      if (prj_elem) {
        // set the parent node of the new node
        prj_elem->parent (prj_name);
      }
    }
    
    if (!prj_elem)
      continue; // TODO: hack, all node types should be handled

    // TODO: this is a hack
    if (tunit_elem->plan ())
      prj_elem->plan ((JoinPointPlan*)1);
    
    //only call and intro join points have a source location
    if (prj_elem->type () == JoinPointLoc::MethodCall ||
//        prj_elem->type () == JoinPointLoc::Introduction ||
        prj_elem->type () == JoinPointLoc::ClassSlice) {
      copy_source_locs (tunit_elem, prj_elem);
    }
        
    // reconcile both elements
    reconcile (tunit_elem, prj_elem);
  }
}


void JoinPointModel::copy_source_locs (JoinPointLoc *tunit_loc, JoinPointLoc *prj_loc) {
  if (prj_loc->tunits ().empty ()) {
    const set<SourceLoc> &source_locs = tunit_loc->source_locs ();
    for (set<SourceLoc>::const_iterator i = source_locs.begin ();
      i != source_locs.end (); ++i) {
      const SourceLoc &tunit_src_loc = *i;
      int file_id = tunit_loc->map (tunit_src_loc.file_id ())->partner ()->id ();
      prj_loc->source_loc (SourceLoc (file_id,
        tunit_src_loc.line (), tunit_src_loc.len (), tunit_src_loc.kind ()));
    }
  }
}

bool JoinPointModel::equal_source_locs (JoinPointLoc *tunit_loc,
  JoinPointLoc *prj_loc) {
  if (tunit_loc->source_locs ().size () != prj_loc->source_locs ().size ())
    return false;
  const set<SourceLoc> &tunit_src_locs = tunit_loc->source_locs ();
  const set<SourceLoc> &prj_src_locs   = prj_loc->source_locs ();
  set<SourceLoc>::const_iterator i_tunit = tunit_src_locs.begin ();
  set<SourceLoc>::const_iterator i_prj   = prj_src_locs.begin ();
  for (; i_tunit != tunit_src_locs.end (); ++i_tunit, ++i_prj) {
    if (i_tunit->file_id () != i_prj->file_id () ||
        i_tunit->line () != i_prj->line () ||
        i_tunit->len () != i_prj->len ())
      return false;
  }
  return true;
}
