// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __PtrStack_h__
#define __PtrStack_h__

// Stepped pointer stack.

#include "Puma/Stack.h"
#include "Puma/BCStack.h"

namespace Puma {


template <class T>
class PtrStack
#ifndef __puma
: private Array<T*>
#endif
 {
  BCStack<long, 8192> _Indices;   // start indices
  
public:
  PtrStack (long = 8192, long = 8192);
  virtual ~PtrStack () {}

#ifndef __puma
  void Push (const T *);  // push item on current stack
#else
  void Push (...);        // simple version for PUMA
#endif
  void Pop ();            // pop item from current stack
  T *Top () const;        // get top item from current stack

  long Length () const;   // length of current stack
  T *Get (long) const;    // get n-th item from current stack

  void New ();            // provide new stack
  void Forget ();         // forget current stack (as if New() wasn't called)  
  void Reject ();         // reject current stack (call Remove() for item)
  void Destroy ();        // destroy current stack (call Delete() for item)
  long Stacks () const;   // current number of stacks
  
  void Reset ();          // reset stack to initial state

protected:
  virtual void Delete (); // delete item
  virtual void Remove (); // remove item
  
private:
  long Index () const;    // return current index
  long Bottom () const;   // return start index of current stack
};

#ifndef __puma

template <class T>
inline PtrStack<T>::PtrStack (long is, long incr) :
  Array<T*> (is, incr) {
  New ();
}

template <class T>
inline void PtrStack<T>::Reset () { 
  _Indices.reset ();
  Array<T*>::reset ();
  New ();
}
 
template <class T>
inline long PtrStack<T>::Index () const { 
  return Array<T*>::length () - 1; 
}
 
template <class T>
inline long PtrStack<T>::Bottom () const { 
  return _Indices.top (); 
}
 
template <class T>
inline long PtrStack<T>::Stacks () const { 
  return _Indices.length (); 
}

template <class T>
inline long PtrStack<T>::Length () const { 
  return Index () - Bottom (); 
}
 
template <class T>
inline T *PtrStack<T>::Top () const { 
  if (Length () > 0)
    return lookup (Index ()); 
  else 
    return (T*)0;
}

template <class T>
inline void PtrStack<T>::Push (const T *item) { 
  append ((T*)item); 
}

template <class T>
inline void PtrStack<T>::Pop () { 
  if (Length () > 0)
    Array<T*>::remove (Index ()); 
}

template <class T>
inline T *PtrStack<T>::Get (long idx) const {
  if (Length () <= 0 || idx >= Length () || idx < 0)
    return (T*)0;
  return lookup (Bottom () + 1 + idx);
}

template <class T>
inline void PtrStack<T>::New () { 
  _Indices.push (Index ()); 
}

template <class T>
inline void PtrStack<T>::Forget () { 
  _Indices.pop (); 
}
  
template <class T>
inline void PtrStack<T>::Reject () { 
  while (Length () > 0)
    Remove ();
  Forget (); 
}
  
template <class T>
inline void PtrStack<T>::Destroy () { 
  while (Length () > 0) {
    Delete ();
    Array<T*>::remove (Index ()); // Pop()
  }
  Forget (); 
}

template <class T>
inline void PtrStack<T>::Remove () {
  Array<T*>::remove (Index ()); // Same as Pop()
}
  
template <class T>
inline void PtrStack<T>::Delete () {
  T *item = Top (); 
  if (item)
    delete item;
}

#endif /* __puma */


} // namespace Puma

#endif /* __PtrStack_h__ */
