// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//
// This program is free software;  you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of
// the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
// MA  02111-1307  USA

#include "ModelBuilder.h"
#include "TrackerDog.h"
#include "IntroductionUnit.h"
#include "PointCutContext.h"
#include "PointCutEvaluator.h"
#include "PointCutExpr.h"
#include "ACConfig.h"
#ifdef ACMODEL
#include "Utils.h"
#endif

#include "Puma/CFileInfo.h"
#include "Puma/ACAspectInfo.h"
#include "Puma/CSemDatabase.h"
#include "Puma/VerboseMgr.h"
#include "Puma/ErrorStream.h"
#include "Puma/SysCall.h"

#ifdef ACMODEL
static string objname (CObjectInfo *info) {
  string result;
  CTemplateInstance *instance;
  DeducedArgument *arg;

  if (! info->Name ()/* || info->isAnonymous ()*/)
    result = "<null>";
  else
    result = info->Name ();

  instance = info->TemplateInstance ();
  if (instance) {
    result += "<";
    for (unsigned i = 0; i < instance->InstantiationArgs (); i++) {
      arg = instance->InstantiationArg (i);
      // do not list default arguments
      if (arg->isDefaultArg ())
        break;

      if (i)
        result += ",";
      ostringstream out;
      if (arg->Type ()) {
        out << *arg->Type ();
      } else if (arg->Value ()) {
        out << *arg->Value ();
      }
      result += out.str ();
    }

    if (result[result.length () -1] == '>')
      result += " ";
    result += ">";
  }
  return result;
}
#endif //ACMODEL

// start phase 1
void ModelBuilder::setup_phase1 (CTranslationUnit& tunit, int tunit_len) {
  assert(_phase == 0);
  _tunit_len = tunit_len;
#ifdef ACMODEL
  // TODO: should be handled by the generator
  set_root(0);
#endif
  _phase = 1;
  build (tunit);
}

// start phase 2
void ModelBuilder::setup_phase2 (CTranslationUnit& tunit, list<CTree*> &ah_trees) {
  assert(_phase == 1);
  _phase = 2;
  build (tunit);

  // run the tracker dog
  TrackerDog tracker (tunit, *this, _conf.dynamic());
  tracker.run ();

  // reset the phase => we are done
  _phase = 0;
}

#ifdef ACMODEL
void ModelBuilder::build (CStructure &scope, ACM_Name *jpl) {
#else
void ModelBuilder::build (CStructure &scope, JPL_Name *jpl) {
#endif
  // iterate through all namespace in this scope
  for (unsigned int n = 0; n < scope.Namespaces (); n++) {
    CNamespaceInfo *ni = scope.Namespace (n);

    // check if this namespace belongs to our project
    FileUnit *funit = source_unit (ni->Tree ());
    if (!funit || !_db->Project ()->isBelow (funit))
      continue;

#ifdef ACMODEL
    ACM_Namespace *jpl_namespace = register_namespace (ni, jpl);
#else
    JPL_Namespace *jpl_namespace = register_namespace (ni, jpl);
#endif
    build (*ni, jpl_namespace);
  }

  // first collect all classes and class template instances in this scope
  list<CClassInfo*> classes;
  for (unsigned int c = 0; c < scope.Types (); c++) {
    CClassInfo *ci = scope.Type (c)->ClassInfo ();
    if (!ci)
      continue;
    if (ci->isTemplate()) {
      // insert all class template instances
      CTemplateInfo *ti = ci->TemplateInfo();
      for (unsigned int tii = 0; tii < ti->Instances(); tii++)
        classes.push_back (ti->Instance(tii)->ClassInfo());
    }
    else
      // an ordinary class -> handle as well
      classes.push_back (ci);
  }

  // now handle all collected classes (and class template instances)
  for (list<CClassInfo*>::const_iterator iter = classes.begin();
      iter != classes.end (); ++iter) {
    CClassInfo *ci = *iter;

    // check if this class belongs to our project
    FileUnit *funit = source_unit (ci->Tree ());
    if (!funit || !_db->Project ()->isBelow (funit))
      continue;

    // check if this class is a class slice
    ACSliceInfo *acsi = _db->SliceInfo (ci);
    if (acsi) {
      if (!ci->isAnonymous()) {
        register_class_slice (acsi,
            !ci->CObjectInfo::QualifiedScope () &&
            ci->Scope () == &scope ? jpl : 0);
      }
    }
    else {
      // it is an ordinary class (or aspect)
      ACAspectInfo *ai = _db->AspectInfo (ci->DefObject ());
#ifdef ACMODEL
      ACM_Class *jpl_class = 0;
#else
      JPL_Class *jpl_class = 0;
#endif
      if (ai)
        jpl_class = register_aspect (ai, !ci->CObjectInfo::QualifiedScope () &&
          ci->Scope () == &scope ? jpl : 0);
      else
        jpl_class = register_class (ci, !ci->CObjectInfo::QualifiedScope () &&
          ci->Scope () == &scope ? jpl : 0);
      if (!jpl_class) continue;

      if (ci->isDefined ()) {

        // setup the base class relations
#ifdef ACMODEL
        jpl_class->get_bases().clear();
#endif
        for (unsigned b = 0; b < ci->BaseClasses (); b++) {
          CClassInfo *base = ci->BaseClass (b)->Class ();
#ifdef ACMODEL
          ACM_Class *base_loc = 0;
#else
          JPL_Class *base_loc = 0;
#endif
          ACAspectInfo *base_aspect = _db->AspectInfo (base->DefObject ());
          if (base_aspect)
            base_loc = register_aspect (base_aspect);
          else
            base_loc = register_class (base);
          // ignore base classes that are not part of the model
#ifdef ACMODEL
          if (base_loc && (base_loc->type_val () & (JPT_Class|JPT_Aspect))) {
            jpl_class->get_bases().insert(base_loc);
            base_loc->get_derived().insert(jpl_class);
          }
#else
          if (base_loc && (base_loc->type () == JoinPointLoc::Class ||
              base_loc->type () == JoinPointLoc::Aspect)) {
            jpl_class->base ((JPL_Class*)base_loc);
          }
#endif
//        else
  //          cout << "*** base not found: " << ci->QualName () << " " << base->QualName () << endl;
        }

        // recursively go down one step
        build (*ci, jpl_class);

        // setup the built-in member functions
        if (_phase == 2 && !ci->isStruct () &&
            (_db->Project ()->isBelow (ci->SourceInfo ()->SrcUnit ()) ||
             IntroductionUnit::cast (ci->SourceInfo ()->SrcUnit ()))) {
          for (unsigned f = 0; f < ci->Functions (); f++) {
            CFunctionInfo *fi = ci->Function (f)->DefObject ();
            if (!fi->isBuiltin ())
              continue;
            if (fi->isConstructor ()) {
              if (ci->isTemplateInstance())
                register_function (fi, jpl_class);
              else
                register_construction (register_function (fi, jpl_class));
            }
            else if (fi->isDestructor ()) {
              if (ci->isTemplateInstance())
                register_function (fi, jpl_class);
              else
                register_destruction (register_function (fi, jpl_class));

            }
          }
        }

        // is it a definition?
#ifdef ACMODEL
        if (jpl_class->type_val () == JPT_Aspect) {
          ACM_Aspect *jpl_aspect = (ACM_Aspect*)jpl_class;
#else
        if (jpl_class->type () == JoinPointLoc::Aspect) {
          JPL_Aspect *jpl_aspect = (JPL_Aspect*)jpl_class;
#endif
          ACAspectInfo *ai = _db->AspectInfo (ci->DefObject ());
          _vm << (ai->is_abstract () ? "Abstract" : "Concrete")
#ifdef ACMODEL
              << " aspect " << signature (*jpl_aspect) << endvm;
#else
              << " aspect " << jpl_aspect->signature () << endvm;
#endif
          _vm++;
          advice_infos (jpl_aspect, ai);
          _vm--;
        }
      }
    }
  }

  // insert all functions
  for (unsigned f = 0; f < scope.Functions (); f++) {
    CFunctionInfo *fi = scope.Function (f);
    if (fi->isBuiltin ())
      continue;

    // check if this function belongs to our project
    FileUnit *funit = source_unit (fi->Tree ());
    if (!funit || !_db->Project ()->isBelow (funit))
      continue;

    // only functions that belong to the project are registered here
    // other functions might be register later if they are referenced
    register_function (fi,
      !fi->CObjectInfo::QualifiedScope () && fi->Scope () == &scope ? jpl : 0);
  }
}

#ifdef ACMODEL
void ModelBuilder::advice_infos (ACM_Aspect *jpl_aspect, ACAspectInfo *acai) {
#else
void ModelBuilder::advice_infos (JPL_Aspect *jpl_aspect, ACAspectInfo *acai) {
#endif
  // collect all introduction advices
  if (_phase == 1) {
    for (int i = 0; i < acai->IntroNodes (); i++) {
      ACIntroductionInfo *acii = acai->IntroNode (i);
#ifdef ACMODEL
      TU_PointcutExpr *expr = newPointcutExpr();
      expr->set_val("*TODO*"); // TODO: set pointcut expr string
      expr->tree(acii->def_node ()->Pointcut ());
      TU_Introduction *intro = newIntroduction();
      intro->set_expr(expr);
      intro->intro_info(acii);
      jpl_aspect->get_intros().insert(intro);
#else
      TU_Introduction *intro = new TU_Introduction;
      intro->parent (jpl_aspect);
      intro->expr (acii->def_node ()->Pointcut ());
      intro->intro_info (acii);
      register_elem (intro);
#endif
      add_source_loc (intro, acii->def_node ());
      check (jpl_aspect, acii, intro);
#ifdef ACMODEL
      // TODO: how to get some infos about the slice?
      _vm << "intro: " << get_slice(*intro)->get_kind() << " "
          << /*intro->get_slice()->signature () << */ endvm;
#else
      _vm << "intro: " << intro->introduced ()->type_str () << " "
          << intro->introduced ()->signature () << endvm;
#endif
    }
    // collect the order advice of this aspect
    for (int i = 0; i < acai->OrderNodes (); i++) {
      CT_AdviceDecl *ad = acai->OrderNode (i);
#ifdef ACMODEL
      TU_Order *order = (TU_Order*)newOrder();
      TU_PointcutExpr *expr = newPointcutExpr();
      expr->set_val("*TODO*"); // TODO: set pointcut expr string
      expr->tree(ad->Pointcut ());
      order->set_expr(expr);
      jpl_aspect->get_orders().insert(order);
#else
      TU_Order *order = new TU_Order;
      order->parent (jpl_aspect);
      order->expr (ad->Pointcut ());
#endif
      CT_OrderList *lst = ((CT_OrderDecl*)ad->Decl ())->OrderList ();
      for (int n = 0; n < lst->Entries (); n++) {
#ifdef ACMODEL
        TU_PointcutExpr *expr = newPointcutExpr();
        expr->set_val("TODO"); // TODO: set pointcut expr string
        expr->tree(lst->Entry (n));
        order->get_aspect_exprs().insert(expr);
#else
        order->add_pce (lst->Entry (n));
#endif
      }
      order->tree (ad);
#ifndef ACMODEL
      register_elem (order);
#endif
      add_source_loc (order, ad);
    }
  }
  // collect the advice nodes
#ifdef ACMODEL
  // TODO: change of semantics: the previous version created entries
  // for the advice code in both phases. What does that mean?
  if (_phase == 2) {
#endif
  for (int i = 0; i < acai->AdviceNodes (); i++) {
    CT_AdviceDecl *ad = acai->AdviceNode (i);
    string sig  = TI_AdviceCode::name (ad);
    string name = ((CT_FctDef*)ad->Decl ())->Object ()->Name ().c_str ();
#ifdef ACMODEL
    TU_PointcutExpr *expr = newPointcutExpr();
    expr->set_val("TODO"); // TODO: set pointcut expr string
    expr->tree(ad->Pointcut ());
    TU_AdviceCode *new_elem = newAdviceCode();
    jpl_aspect->get_advices().insert(new_elem);
    new_elem->set_kind(name.rfind("before") != string::npos?ACT_BEFORE:
                       (name.rfind("after") != string::npos?ACT_AFTER:ACT_AROUND));
    new_elem->set_expr(expr);
#else
    TU_AdviceCode *new_elem = (TU_AdviceCode*)new_advice_code (jpl_aspect, name, sig);
    new_elem->expr (ad->Pointcut ());
    new_elem->function_type(JPL_Function::MEMBER);
#endif
    new_elem->tree (ad);
    new_elem->phase (_phase);
    add_source_loc (new_elem, ad);
  }
#ifdef ACMODEL
  }
#endif
}


#ifdef ACMODEL
void ModelBuilder::check (ACM_Aspect *jpl_aspect, ACIntroductionInfo *acii,
  ACM_Introduction *intro) {
#else
void ModelBuilder::check (JPL_Aspect *jpl_aspect, ACIntroductionInfo *acii,
  JPL_Introduction *intro) {
#endif

  // get the node type of the introduction declaration
  CTree *decl_node = acii->def_node ()->Decl ();
  const char *decl_node_name = decl_node->NodeName ();

#ifdef ACMODEL
  ClassSliceType _type = CST_ERROR;
#else
  JPL_ClassSlice::class_slice_type _type = JPL_ClassSlice::CS_ERROR;
#endif
  Unit *_slice_unit = 0;
  CObjectInfo *obj = 0;
  string name;

  if (decl_node_name == CT_Intro::NodeId ()) {
    // Copy all tokens of the intro pattern => intro source code can be removed
    // The pattern contains no whitespace or any kind of formatting!
    // Token positions are identical with the entry number.
    CT_Intro *intro = (CT_Intro*)decl_node;
    // set the name for #line directives
    Unit _pattern;
    _slice_unit = (Unit*)intro->token ()->belonging_to ();
    _pattern.name (_slice_unit->name ());
    bool name_truncated = false;
    for (int e = 0; e < intro->Entries (); e++) {
      Token *tok = intro->Entry (e)->token ();
      _pattern.append (*tok->duplicate ());
      if (name.length () < 40) {
        name += tok->text ();
        name += " ";
      }
      else
        name_truncated = true;
    }
    if (name_truncated)
      name += "...";
#ifdef ACMODEL
    _type = CST_DEP_NORMAL;
#else
    _type = JPL_ClassSlice::CS_OLD_OTHER;
#endif
  }
  else if (decl_node_name == CT_SliceRef::NodeId ()) {
#ifdef ACMODEL
    _type = CST_NORMAL;
#else
    _type = JPL_ClassSlice::CS_NORMAL;
#endif
    obj = ((CT_SliceRef*)decl_node)->name ()->Object ();
////    analyze_slice (obj);
//    CT_ClassSliceDecl *csd = (CT_ClassSliceDecl*)obj->Tree ();
//    _slice_unit = (Unit*)csd->token ()->belonging_to ();

  }
  else if (decl_node_name == CT_ClassSliceDecl::NodeId ()) {
#ifdef ACMODEL
    _type = CST_NORMAL;
#else
    _type = JPL_ClassSlice::CS_NORMAL;
#endif
    obj = ((CT_ClassSliceDecl*)decl_node)->Object ();
////    analyze_slice (obj);
    CT_ClassSliceDecl *csd = (CT_ClassSliceDecl*)obj->Tree ();
    _slice_unit = (Unit*)csd->token ()->belonging_to ();
  }
  else if (decl_node_name == CT_ObjDecl::NodeId ()) {
    CT_DeclaratorList *dl = ((CT_ObjDecl*)decl_node)->Declarators();
    if (dl->Entries () > 0) {
      obj = ((CT_InitDeclarator*)dl->Entry (0))->Object ();

      // check for baseclass introductions
      if (obj->FunctionInfo () && strcmp (obj->Name (), "baseclass") == 0) {
#ifdef ACMODEL
        _type = CST_DEP_BASE;
#else
        _type = JPL_ClassSlice::CS_OLD_BASE;
#endif
        CClassInfo *ci = obj->FunctionInfo ()->Argument (0u)->TypeInfo ()->ClassInfo ();
        _slice_unit = (Unit*)ci->Tree ()->token ()->belonging_to ();
        name = ci->QualName ();
      }
      else
        obj = 0;
    }
  }

  // to be found or created:
  TU_ClassSlice *class_slice = 0;

  // check whether the slice is defined
  Location loc = decl_node->token ()->location ();
#ifdef ACMODEL
  if (_type == CST_NORMAL) {
#else
  if (_type == JPL_ClassSlice::CS_NORMAL) {
#endif
    ACSliceInfo *acsi = obj->SemDB ()->SliceInfo (obj);
    if (!acsi) {
      _err << sev_error << loc
           << "'" << obj->QualName () << "' used as a slice, but isn't a slice"
           << endMessage;
    }
    else {
      if (!acsi->definition ()) {
        _err << sev_error << loc
            << "undefined slice '" << obj->QualName ()
            << "' used in advice" << endMessage;
      }
      if (acsi->type () == ACSliceInfo::SL_UNION) {
        _err << sev_error << loc
             << "union slices are not supported by ac++, yet" << endMessage;
      }
      if (obj->ClassInfo ()) {
#ifdef ACMODEL
        if (obj->ClassInfo ()->isAnonymous ()) {
          class_slice = (TU_ClassSlice*)newClassSlice();
          class_slice->set_name("%anon");
          class_slice->set_builtin(false);
          class_slice->set_kind(_type);
          class_slice->obj_info (obj);
          class_slice->slice_unit (_slice_unit);
          add_source_loc (class_slice, decl_node);
        }
        else
          class_slice = register_class_slice (acsi);
#else
        class_slice = register_class_slice (acsi);
#endif
      }
      if (!class_slice) {
        _err << sev_fatal << loc
             << "class slice '" << obj->QualName () << "' for intro lost" << endMessage;
      }
    }
  }
  else {
    if (_conf.warn_deprecated ())
      _err << sev_warning << loc
          << "deprecated introduction syntax, "
          << "use slice instead." << endMessage;
    // create an anonymous slice object in the model for this old-style introduction
#ifdef ACMODEL
    ACM_Name *found = map_lookup(*jpl_aspect, name);
    if (found && found->type_val() == JPT_ClassSlice)
      class_slice = (TU_ClassSlice*)found;
    else {
      class_slice = (TU_ClassSlice*)newClassSlice();
      class_slice->set_name(name);
      class_slice->set_builtin(false);
      class_slice->set_kind(_type);
      class_slice->set_prot(acii->prot() == CProtection::PROT_PRIVATE?
        SP_PRIVATE:(acii->prot() == CProtection::PROT_PROTECTED?
        SP_PROTECTED:(acii->prot() == CProtection::PROT_PUBLIC?
        SP_PUBLIC:SP_UNKNOWN)));
      jpl_aspect->get_children().insert(class_slice);
      map_insert(*jpl_aspect, *class_slice, name);
    }
#else
    class_slice = (TU_ClassSlice*)new_class_slice (jpl_aspect, name);
    class_slice->slice_type (_type);
    class_slice->prot (acii->prot ());
#endif
    class_slice->obj_info (obj);
    class_slice->slice_unit (_slice_unit);
    add_source_loc (class_slice, decl_node);
  }

  if (class_slice) {
    if (decl_node_name == CT_Intro::NodeId ()) {
      // Copy all tokens of the intro pattern => intro source code can be removed
      // The pattern contains no whitespace or any kind of formatting!
      // Token positions are identical with the entry number.
      CT_Intro *intro = (CT_Intro*)decl_node;
      // set the name for #line directives
      Unit &pattern = class_slice->pattern ();
      pattern.name (_slice_unit->name ());
      for (int e = 0; e < intro->Entries (); e++) {
        Token *tok = intro->Entry (e)->token ();
        pattern.append (*tok->duplicate ());
      }
    }

#ifdef ACMODEL
    if (class_slice->get_name ()[0] == '%')
      intro->set_anon_slice (class_slice);
    else
      intro->set_named_slice (class_slice);
#else
    intro->introduced (class_slice);
#endif
  }
}


string ModelBuilder::model_filename (FileUnit *unit) {
  const char *fname = unit->absolutePath ();
  ACProject &prj = _conf.project ();
  // if the file does not belong to the project return the absolute path
  if (!unit->belongsTo (prj))
    return fname;
  // iterate over all project paths
  for (long p = 0; p < prj.numPaths (); p++) {
    Filename dir_abs;
    if (!SysCall::canonical (prj.src (p), dir_abs)) {
      assert (false);
      return fname;
    }
    int dir_len = strlen (dir_abs.name ());
    if (strncmp (dir_abs.name (), fname, dir_len) == 0) {
      return fname + dir_len + 1;
    }
  }
  // the file has to be below any of the directories => fatal error here
  // assert (false); acgen.c does not belong to the project with this test
  return fname;
}


// get the modification time of a file (UNIX Epoch value)
long ModelBuilder::modification_time (FileUnit *unit) {
  FileInfo fileinfo;
  if (! SysCall::stat (unit->absolutePath(), fileinfo))
    return -1; // File does not exists.
  return fileinfo.modi_time ();
}


void ModelBuilder::build (CTranslationUnit& tunit) {
  _db = &tunit.db ();

  if (_phase == 1) {
#ifdef ACMODEL
    _tunit_file = newTUnit();
    _tunit_file->set_filename(model_filename ((FileUnit*)tunit.unit ()));
    _tunit_file->set_len(_tunit_len);
    _tunit_file->set_time(modification_time ((FileUnit*)tunit.unit ()));
    get_files().insert(_tunit_file);
#else
    _tunit_file = register_file (model_filename ((FileUnit*)tunit.unit ()), -1, true);
#endif
    _file_map.insert (FileMapPair (tunit.unit (), _tunit_file));
  }

  CFileInfo *fi = _db->FileInfo (0);
#ifdef ACMODEL
  ACM_Namespace *jpl_namespace = register_namespace (fi);
#else
  JPL_Namespace *jpl_namespace = register_namespace (fi);
#endif
  build (*fi, jpl_namespace);
}

bool ModelBuilder::inside_template (CScopeInfo *scope) const {
  if (scope->isTemplate () ||
      (scope->isTemplateInstance () &&
       scope->TemplateInstance ()->isPseudoInstance ()))
    return true;
  if (scope->QualifiedScope ())
    return inside_template (scope->QualifiedScope ());
  if (scope->Parent () != scope)
    return inside_template (scope->Parent ());
  return false;
}

bool ModelBuilder::inside_template_instance (CScopeInfo *scope) const {
  if (scope->isTemplateInstance ())
    return true;
  if (scope->QualifiedScope ())
    return inside_template_instance (scope->QualifiedScope ());
  if (scope->Parent () != scope)
    return inside_template_instance (scope->Parent ());
  return false;
}

bool ModelBuilder::is_valid_model_class (CClassInfo *ci) const {
  // Don't consider
  // * anonymous classes like template instances(?)
  // * the generated class JoinPoint
  // * classes defined in the special namespace AC
  if (strncmp (ci->Name (), "%", 1) == 0 ||
      strcmp (ci->QualName (), "JoinPoint") == 0 ||
      strncmp (ci->QualName (), "AC::", 4) == 0)
    return false;

  // Templates and classes nested in template class are not considered for
  // matching, only instances
  if (inside_template (ci))
    return false;

  return true;
}

bool ModelBuilder::is_valid_model_function (CFunctionInfo* fi) const {
  // Don't consider
  // * member functions of an invalid class
  // * ac++ generated functions
  // * pointcuts

  CRecord *cls = fi->ClassScope ();
  if (cls && !is_valid_model_class ((CClassInfo*)cls->DefObject ()))
    return false;

  CFunctionInfo *def = fi->DefObject ();
  if (inside_template (def))
    return false;

  if (strncmp (fi->Name (), "%a", 2) == 0 ||
      strncmp (fi->Name (), "__a", 3) == 0 ||
      strcmp (fi->Name (), "aspectof") == 0 ||
      strcmp (fi->Name (), "aspectOf") == 0)
    return false;

  if (_db->PointcutInfo (def))
    return false;

  if (!fi->isBuiltin () && !IntroductionUnit::cast (def->SourceInfo ()->SrcUnit ()) &&
      strcmp (def->SourceInfo ()->FileName (), "<anonymous unit>") == 0)
    return false;

  return true;
}

bool ModelBuilder::is_valid_model_namespace (CNamespaceInfo *ni) const {
  // no template instance namespace, but anonymous namespaces!
  if ((strcmp(ni->Name (), "<unnamed>") == 0 ||
       strstr (ni->Name (), "<") == 0) &&
      strcmp (ni->QualName (), "AC") != 0 &&
      strncmp (ni->Name (), "__puma", 6) != 0)
    return true;
  return false;
}

TU_Type *ModelBuilder::register_type (CTypeInfo *ti) {
#ifdef ACMODEL
  TU_Type *new_elem = newType();
  new_elem->set_signature(TI_Type::name (ti));
#else
  TU_Type *new_elem = (TU_Type*)new_type (TI_Type::name (ti));
#endif
  new_elem->type_info (ti);
  return new_elem;
}

TU_Function *ModelBuilder::register_function (CFunctionInfo *fi,
#ifdef ACMODEL
  ACM_Name *parent) {
#else
  JPL_Name *parent) {
#endif
  if (!is_valid_model_function (fi))
    return 0;

  CFunctionInfo *def = fi->DefObject ();

  // find the parent model element
  if (!parent && !(parent = register_scope (def)))
    return 0;

  // build the name and signature of the function
  string sig  = TI_Function::name (def);
  string name = fi->Name ().c_str ();
  
  // register the element
#ifdef ACMODEL
  // TODO: sig contains the result type, which is not needed
  TU_Function *elem = (TU_Function*)map_lookup(*parent, sig);
  if (!elem || elem->type_val() != JPT_Function ||
      ((fi->isStatic () && !fi->isStaticMethod ()) &&
          (!elem->has_static_in () || elem->get_static_in () != _tunit_file))) {
    elem = newFunction();
    elem->set_name(objname(fi));
    if (fi->isStatic () && !fi->isStaticMethod ())
      elem->set_static_in (_tunit_file);
    elem->phase(0);
    map_insert(*parent, *elem, sig);
    parent->get_children().insert(elem);
    FunctionType ft = FT_NON_MEMBER;
    if (fi->isConstructor ())
      ft = FT_CONSTRUCTOR;
    else if (fi->isDestructor ())
      ft = FT_DESTRUCTOR;
    else if (fi->isMethod ()) {
      if (fi->isStaticMethod ())
        ft = FT_STATIC_MEMBER;
      else if (fi->isVirtual () || fi->overridesVirtual ())
        ft = FT_VIRTUAL_MEMBER;
      else
        ft = FT_MEMBER;
    }
    else {
      if (fi->isStatic())
        ft = FT_STATIC_NON_MEMBER;
    }
    elem->set_kind (ft);
    elem->set_builtin (fi->isBuiltin ());
    CVQualifiers cvq = CVQ_NONE;
    if (fi->TypeInfo()->isConst())
      cvq = (CVQualifiers)(cvq | CVQ_CONST);
    if (fi->TypeInfo()->isVolatile())
      cvq = (CVQualifiers)(cvq | CVQ_VOLATILE);
    elem->set_cv_qualifiers(cvq);
  }
#else
  TU_Function * elem = (TU_Function*)new_function (parent, name, sig);

  // set function attributes
  if (elem->phase () == 0) {
    JPL_Function::FunctionType ft = JPL_Function::NON_MEMBER;
    if (fi->isConstructor ())
      ft = JPL_Function::CONSTRUCTOR;
    else if (fi->isDestructor ())
      ft = JPL_Function::DESTRUCTOR;
    else if (fi->isMethod ()) {
      if (fi->isStaticMethod ())
        ft = JPL_Function::STATIC_MEMBER;
      else if (fi->isVirtual () || fi->overridesVirtual ())
        ft = JPL_Function::VIRTUAL_MEMBER;
      else
        ft = JPL_Function::MEMBER;
    }
    else {
      if (fi->isStatic())
        ft = JPL_Function::STATIC_NON_MEMBER;
    }
    elem->function_type (ft);
    elem->is_built_in (fi->isBuiltin ());
  }
#endif

  if (elem->phase () != _phase) {
    elem->func_info (def);
    elem->phase (_phase);
    // TODO: too often?
#ifndef ACMODEL
    add_source_loc (elem, fi, fi->isFctDef () ? SLK_DEF : SLK_DECL);
#endif

#ifdef ACMODEL
    // TODO: We forget the arg type from arg 0. They will be deleted by jpm.
    elem->get_arg_types().clear();
#else
    elem->reset_types ();
#endif
    CTypeInfo *rtype = (fi->isConversion ()) ?
      fi->ConversionType() : fi->TypeInfo ()->BaseType ();
#ifdef ACMODEL
    if (!rtype->is_undefined())
      elem->set_result_type (register_type (rtype));
    // argument types are the types of the function declaration
    CTypeList *arg_types = fi->TypeInfo ()->ArgTypes ();
    elem->set_variadic_args(false);
    for (unsigned a = 0; a < arg_types->Entries (); a++)
      if (arg_types->Entry(a)->is_ellipsis())
        elem->set_variadic_args(true);
      else
        elem->get_arg_types().insert (register_type (arg_types->Entry (a)));
#else
    elem->result_type (register_type (rtype));
    // argument types are the types of the function declaration
    CTypeList *arg_types = fi->TypeInfo ()->ArgTypes ();
    for (unsigned a = 0; a < arg_types->Entries (); a++)
      elem->add_arg_type (register_type (arg_types->Entry (a)));
#endif

    if (_phase == 2 && !fi->isBuiltin () && !fi->isPureVirtual () &&
        !(inside_template_instance (fi)) &&
      (fi->SemDB ()->Project ()->isBelow (fi->SourceInfo ()->SrcUnit ()) ||
       IntroductionUnit::cast (fi->SourceInfo ()->SrcUnit ()))) {
      if (fi->isConstructor ())
        register_construction (elem);
      else if (fi->isDestructor ())
        register_destruction (elem);
      else
        register_execution (elem);

      // constructors and and destructors cannot be called
      if (!(fi->isConstructor () || fi->isDestructor ())) {
        // register a 'pseudo call join point'
        register_call (def, 0, 0, 0);
      }
    }
  }
#ifdef ACMODEL
  add_source_loc (elem, fi, fi->isFctDef () ? XSLK_DEF : XSLK_DECL);
#endif
  return elem;
}

TU_ClassSlice *ModelBuilder::register_class_slice (ACSliceInfo *acsi,
#ifdef ACMODEL
  ACM_Name *parent) {
#else
  JPL_Name *parent) {
#endif

  CClassInfo *ci = acsi->object ()->ClassInfo ();

  // find the parent model element
  if (!parent && !(parent = register_scope (ci)))
    return 0;

#ifdef ACMODEL
  string name = ci->Name().c_str();
  TU_ClassSlice *new_elem = 0;
  ACM_Name *found = map_lookup(*parent, name);
  if (!found || found->type_val() != JPT_ClassSlice) {
    new_elem = (TU_ClassSlice*)newClassSlice();
    new_elem->set_name(name);
    new_elem->set_builtin(false);
    parent->get_children().insert(new_elem);
    map_insert(*parent, *new_elem, name);
  }
  else
    new_elem = (TU_ClassSlice*)found;
#else
  // build the class slice name
  string name = TI_ClassSlice::name (acsi);

  // register the slice as a child of its parent
  TU_ClassSlice *new_elem = (TU_ClassSlice*)new_class_slice (parent, name);
#endif

  // set class slice attributes
  if (acsi->definition ()) {
    CObjectInfo *defobj = acsi->definition ()->object();
    Unit *unit = (Unit*)defobj->Tree ()->token ()->belonging_to ();
    new_elem->obj_info (defobj);
    new_elem->slice_unit (unit);
  }
#ifdef ACMODEL
  SliceProtection prot =
    (acsi->type () == ACSliceInfo::SL_STRUCT) ? SP_PUBLIC : SP_PRIVATE;
  new_elem->set_prot (prot);
  new_elem->set_kind (CST_NORMAL);
#else
  CProtection::Type prot = (acsi->type () == ACSliceInfo::SL_STRUCT) ?
    CProtection::PROT_PUBLIC : CProtection::PROT_PRIVATE;
  new_elem->prot (prot);
  new_elem->slice_type (JPL_ClassSlice::CS_NORMAL);
#endif
  add_source_loc (new_elem, ci);

  return new_elem;
}

#ifdef ACMODEL
TU_Class *ModelBuilder::register_class (CClassInfo *ci, ACM_Name *parent) {
#else
TU_Class *ModelBuilder::register_class (CClassInfo *ci, JPL_Name *parent) {
#endif
  // only classes are relevant
  if (!is_valid_model_class (ci))
    return 0;

  // find the parent model element
  if (!parent && !(parent = register_scope (ci)))
    return 0;

#ifdef ACMODEL
  string name = objname(ci);
  TU_Class *elem = 0;
  ACM_Name *found = map_lookup(*parent, name);
  if (!found || found->type_val() != JPT_Class) {
    elem = (TU_Class*)newClass();
    elem->set_name(name);
    elem->set_builtin(false);
    parent->get_children().insert(elem);
    map_insert(*parent, *elem, name);
  }
  else
    elem = (TU_Class*)found;
#else

  // deterine the name
  string name = TI_Class::name (ci);
  
  // register an object with that name
  TU_Class *elem = (TU_Class*)new_class (parent, name);
#endif
  // set the class attributes
  elem->class_info (ci);
  if (elem->phase () == 0) // only once ...
#ifdef ACMODEL
    elem->set_intro_target (is_intro_target (ci->DefObject ()));
#else
    elem->intro_target (is_intro_target (ci->DefObject ()));
#endif
  elem->phase (_phase);
  if (_phase == 2) {
#ifdef ACMODEL
    add_source_loc (elem, ci, ci->isDefined () ? XSLK_DEF : XSLK_DECL);
#else
    add_source_loc (elem, ci, ci->isDefined () ? SLK_DEF : SLK_DECL);
#endif
  }
  return elem;
}

#ifdef ACMODEL
TU_Aspect *ModelBuilder::register_aspect (ACAspectInfo *ai, ACM_Name *parent) {
#else
TU_Aspect *ModelBuilder::register_aspect (ACAspectInfo *ai, JPL_Name *parent) {
#endif
    // only classes are relevant
  CClassInfo *ci = ai->ClassInfo ();
  if (!is_valid_model_class (ci))
    return 0;

  // find the parent model element
  if (!parent && !(parent = register_scope (ci)))
    return 0;

  #ifdef ACMODEL
  string name = ai->name();
  TU_Aspect *elem = 0;
  ACM_Name *found = map_lookup(*parent, name);
  if (!found || found->type_val() != JPT_Aspect) {
    elem = (TU_Aspect*)newAspect();
    elem->set_name(name);
    elem->set_builtin(false);
    parent->get_children().insert(elem);
    map_insert(*parent, *elem, name);
  }
  else
    elem = (TU_Aspect*)found;
#else

  // determine the name
  string name = TI_Class::name (ci);

  // register an object with that name
  TU_Aspect *elem = (TU_Aspect*)new_aspect (parent, name);
#endif
  // set the aspect attributes
  elem->aspect_info (ai);
  if (elem->phase () == 0) // only once ...
#ifdef ACMODEL
    elem->set_intro_target (is_intro_target (ci->DefObject ()));
#else
    elem->intro_target (is_intro_target (ci->DefObject ()));
#endif
  elem->phase (_phase);
  if (_phase == 2) {
#ifdef ACMODEL
    add_source_loc (elem, ci, ci->isDefined () ? XSLK_DEF : XSLK_DECL);
#else
    add_source_loc (elem, ci, ci->isDefined () ? SLK_DEF : SLK_DECL);
#endif
  }
  return elem;
}

bool ModelBuilder::is_intro_target (CClassInfo *def) const {
  return IntroductionUnit::cast (def->SourceInfo ()->SrcUnit ()) ||
		  !(!_db->Project ()->isBelow (def->SourceInfo ()->SrcUnit ()) ||
           def->isTemplateInstance () ||
           !def->isDefined ());
}

TU_Namespace *ModelBuilder::register_namespace (CNamespaceInfo *n,
#ifdef ACMODEL
  ACM_Name *parent) {
#else
  JPL_Name *parent) {
#endif
    // not all Puma namespaces should be registered
  if (!is_valid_model_namespace (n))
    return 0;

  // find the parent model element
  if (!parent && !n->GlobalScope () && !(parent = register_scope (n)))
    return 0;

#ifdef ACMODEL
  TU_Namespace *new_elem = 0;
  if (!parent) {
    if (get_root())
      return (TU_Namespace*)get_root();
    new_elem = newNamespace();
    new_elem->set_name("::");
    new_elem->set_builtin(true);
    set_root(new_elem);
  }
  else {
    string name = n->Name().c_str();
    ACM_Name *found = map_lookup(*parent, name);
    if (found && found->type_val() == JPT_Namespace)
      return (TU_Namespace*)found;
    new_elem = newNamespace();
    new_elem->set_name(name);
    new_elem->set_builtin(false);
    parent->get_children().insert(new_elem);
    map_insert(*parent, *new_elem, name);
  }
  // TODO: source locations
#else
  // build the namespace name
  string name = TI_Namespace::name (n);

  // register this namespace as a child of its parent in the model
  TU_Namespace *new_elem = (TU_Namespace*)new_namespace (parent, name);
#endif

  // set namespace attributes
  new_elem->namespace_info (n);
#ifdef ACMODEL
  if (!(new_elem == get_root ()))
#else
  if (!new_elem->is_root ())
#endif
    add_source_loc (new_elem, n);
  return new_elem;
}


// create a new call join point in the join point model
TU_MethodCall *ModelBuilder::register_call (CFunctionInfo *called, CT_Call *call_node,
    CObjectInfo *caller, int local_id) {

  // find the called function in the join point model
#ifdef ACMODEL
  ACM_Function *called_func = register_function (called);
#else
  JPL_Function *called_func = register_function (called);
#endif
  if (!called_func) {
//  if a called function is, for instance, a member of a local class, it is
//  perfectly valid that we don't find it in the model -> ignore call join-point
//    _err << sev_error << "called function \'" << called->QualName ()
//         << "\' not found in join point model" << endMessage;
    return 0;
  }

  // what is the lexical scope of this call?
#ifdef ACMODEL
  ACM_Name *lexical = 0;
#else
  JPL_Name *lexical = 0;
#endif
  if (!caller) {
    lexical = 0; // a pseudo call join point
  }
  else if (caller->FunctionInfo ()) {
    // TODO: better cache the JPL object of the current function
#ifdef ACMODEL
    ACM_Any *loc = register_function (caller->FunctionInfo (), 0);
#else
    JoinPointLoc *loc = register_function (caller->FunctionInfo (), 0);
#endif
    if (!loc) {
      // TODO: calls in advice code are silently ignored here at the moment
//      _err << sev_error << call_node->token ()->location ()
//           << "location of function call invalid" << endMessage;
      return 0;
    }
#ifdef ACMODEL
    lexical = (ACM_Name*)loc;
#else
    lexical = (JPL_Name*)loc;
#endif
  }
  else {
    CScopeInfo *scope = caller->QualifiedScope ();
    if (!scope) scope = caller->Scope ();
    lexical = register_scope (scope);
    if (!lexical) {
      _err << sev_error << call_node->token ()->location ()
           << "location of function call invalid" << endMessage;
      return 0;
    }
  }

#ifdef ACMODEL
  TU_MethodCall *new_elem = newCall();
  new_elem->set_target(called_func);
  new_elem->set_lid(local_id);
#else
  TU_MethodCall *new_elem = (TU_MethodCall*)new_call (called_func, local_id);
#endif
  new_elem->called (called);
  new_elem->caller (caller);
  new_elem->tree (call_node);
  if (call_node)
    add_source_loc (new_elem, call_node);
  // set the parent in the join point model structure
  if (lexical) { // pseudo-calls are invisible
#ifdef ACMODEL
    lexical->get_children().insert(new_elem);
#else
    new_elem->parent (lexical);
#endif
  }

#ifdef ACMODEL
  // For functions with default arguments, not more than the number of args
  // in the call expression is used.
  bool no_operator = (call_node &&
                      call_node->NodeName () == CT_CallExpr::NodeId () &&
                      ((CT_CallExpr *)call_node)->Arguments ());
  if (no_operator) {
    // argument types are the types from the target function declaration
    CTypeList *formal_arg_types = called->TypeInfo ()->ArgTypes ();
    int args = (int)formal_arg_types->Entries ();
    CT_ExprList *current_arg_list = ((CT_CallExpr *)call_node)->Arguments ();
    int call_args = current_arg_list->Entries ();
    if (called_func->get_variadic_args()) {
      args--; // ignore the ellipsis
      for (int a = args; a < current_arg_list->Entries (); a++) {
        CTypeInfo *arg_type =
          ((CT_Expression*)current_arg_list->Entry (a))->Type ();
        new_elem->get_variadic_arg_types().insert(register_type (arg_type));
      }
    }
    else if (call_args < args) {
      new_elem->set_default_args(args - call_args);
    }
  }
#else
  // set the result type
  CTypeInfo *rtype = TI_Type::of (called_func->result_type ())->type_info ();
  new_elem->result_type (register_type (rtype));

  // argument types are the types from the target function declaration
  CTypeList *formal_arg_types = called->TypeInfo ()->ArgTypes ();
  int args = (int)formal_arg_types->Entries ();
  // For functions with default arguments, not more than the number of args
  // in the call expression is used.
  bool no_operator = (call_node &&
                      call_node->NodeName () == CT_CallExpr::NodeId () &&
                      ((CT_CallExpr *)call_node)->Arguments ());
  if (no_operator) {
    CT_ExprList *current_arg_list = ((CT_CallExpr *)call_node)->Arguments ();
    if (current_arg_list->Entries () < args) {
      args = current_arg_list->Entries ();
    }
  }
  // take the arguments types from the formal argument list
  bool have_ellipsis = false;
  int a = 0;
  for (; a < args; a++) {
    CTypeInfo *arg_type = formal_arg_types->Entry (a);
    if (arg_type->is_ellipsis ()) {
      have_ellipsis = true;
      break;
    }
    else {
      new_elem->add_arg_type (register_type (arg_type));
    }
  }

  // For functions with variable argument lists, the types that match '...'
  // in the declaration are taken from the call expression
  if (have_ellipsis && no_operator) {
    CT_ExprList *current_arg_list = ((CT_CallExpr *)call_node)->Arguments ();
    for (; a < current_arg_list->Entries (); a++) {
      CTypeInfo *arg_type =
        ((CT_Expression*)current_arg_list->Entry (a))->Type ();
      new_elem->add_arg_type (register_type (arg_type));
    }
  }

  register_elem (new_elem);
#endif
  return new_elem;
}

// create a new execution join point
#ifdef ACMODEL
TU_Method *ModelBuilder::register_execution (ACM_Function *ef) {
#else
TU_Method *ModelBuilder::register_execution (JPL_Function *ef) {
#endif
  CFunctionInfo *func = ((TI_Function*)ef->transform_info ())->func_info ();
#ifdef ACMODEL
  TU_Method *new_elem = newExecution();
  ef->get_children().insert(new_elem);
#else
  TU_Method *new_elem = (TU_Method*)new_execution (ef);
  register_elem (new_elem);
#endif
  new_elem->func_info (func);
  return new_elem;
}

// create a new construction join point
#ifdef ACMODEL
TU_Construction *ModelBuilder::register_construction (ACM_Function *cf) {
#else
TU_Construction *ModelBuilder::register_construction (JPL_Function *cf) {
#endif
  assert (cf);
  CFunctionInfo *func = ((TI_Function*)cf->transform_info ())->func_info ();
#ifdef ACMODEL
  TU_Construction *new_elem = newConstruction();
  cf->get_children().insert(new_elem);
#else
  TU_Construction *new_elem = (TU_Construction*)new_construction (cf);
  register_elem (new_elem);
#endif
  new_elem->func_info (func);
  return new_elem;
}

// create a new construction join point
#ifdef ACMODEL
TU_Destruction *ModelBuilder::register_destruction (ACM_Function *df) {
#else
TU_Destruction *ModelBuilder::register_destruction (JPL_Function *df) {
#endif
  assert (df);
  CFunctionInfo *func = ((TI_Function*)df->transform_info ())->func_info ();
#ifdef ACMODEL
  TU_Destruction *new_elem = newDestruction();
  df->get_children().insert(new_elem);
#else
  TU_Destruction *new_elem = (TU_Destruction*)new_destruction (df);
  register_elem (new_elem);
#endif
  new_elem->func_info (func);
  return new_elem;
}

// TODO: temporary solution for dac++
void ModelBuilder::register_attr_access (CAttributeInfo *attr, CTree *node) {
  _access_infos.push_back (AccessInfo (attr, node));
}

#ifdef ACMODEL
ACM_Name *ModelBuilder::register_scope (CObjectInfo *obj) {
  ACM_Name *result = 0;
#else
JPL_Name *ModelBuilder::register_scope (CObjectInfo *obj) {
  JPL_Name *result = 0;
#endif
  CScopeInfo *scope = scope_obj (obj);
  if (scope) {
    if (scope->NamespaceInfo ())
      result = register_namespace (scope->NamespaceInfo ());
#ifndef ACMODEL
    else if (scope->FunctionInfo ())
      result = register_function (scope->FunctionInfo ());
#endif
    else if (scope->ClassInfo ()) {
      ACAspectInfo *ai = _db->AspectInfo (scope->ClassInfo ()->DefObject ());
      if (ai)
        result = register_aspect (ai);
      else
        result = register_class (scope->ClassInfo ());
    }
  }

//  in some cases, e.g. join-points within local classes it can happen that
//  the scope of a join-point is not known in the model -> no error!
//  if (!result) {
//    _err << sev_error << "parent '" << scope_name (obj).c_str ()
//       << "' of model element " << obj->QualName () << " not found"
//       << endMessage;
//  }

  return result;
}

CScopeInfo *ModelBuilder::scope_obj (CObjectInfo *oi) {
  CScopeInfo *scope = 0;
  if (oi->TemplateInstance ()) {
    scope = oi->TemplateInstance ()->Template ()->CObjectInfo::QualifiedScope ();
    if (!scope) scope = oi->TemplateInstance ()->Template ()->Scope ();
  }
  else {
    scope = oi->QualifiedScope ();
    if (!scope) scope = oi->Scope ();
  }
  // if this is a template instance scope, go to the parent
  if (!(strcmp (scope->Name(), "<unnamed>") == 0)) {
    while (scope->isAnonymous () && strstr (scope->Name (), "<")) {
      scope = scope->Parent ();
    }
  }
  return scope;
}

string ModelBuilder::scope_name (CObjectInfo *oi) {
  CScopeInfo *scope = scope_obj (oi);
  assert (scope);
  // is it the globale scope
  if (scope->GlobalScope ())
    return "::";
  ostringstream scope_name;
  if (scope->TypeInfo () && !scope->TypeInfo ()->isUndefined ())
    scope_name << *scope->TypeInfo ();
  else if (scope->isAnonymous ())
    scope_name << "<noname>";
  else
    scope_name << scope->QualName ();
  return scope_name.str ();
}

// add the source location to a model element by using the syntax tree node
#ifdef ACMODEL
void ModelBuilder::add_source_loc (ACM_Any *name, CTree *tree, XSourceLocKind kind) {
#else
void ModelBuilder::add_source_loc (JoinPointLoc *name, CTree *tree, SourceLocKind kind) {
#endif

  // check if this file belong to our project
  FileUnit *funit = source_unit (tree);
  if (!funit)
    return;

  if (!_db->Project ()->isBelow (funit)) {
#ifdef ACMODEL
    if (name->type_val () & JPT_Name) {
      ACM_Name *jpl_name = (ACM_Name*)name;
      // TODO: really use linear search here?
      typedef ACM_Container<ACM_TUnit, false> Container;
      const Container &tunits = jpl_name->get_tunits ();
      bool found = false;
      for (Container::const_iterator i = tunits.begin(); i != tunits.end(); ++i) {
        if (*i == _tunit_file) {
          found = true;
          break;
        }
      }
      if (!found)
        jpl_name->get_tunits ().insert (_tunit_file);
    }
#else
    if (name->type () & JoinPointLoc::Name) {
      JPL_Name *jpl_name = (JPL_Name*)name;
      jpl_name->tunits ().insert (_tunit_file->id ());
    }
#endif
    return;
  }
  FileMap::iterator i = _file_map.find (funit);
#ifdef ACMODEL
  ACM_File *file = 0;
#else
  File *file = 0;
#endif
  if (i != _file_map.end ())
    file = i->second;
  else {
    // TODO: temporary hack
    int len = ((Token*)funit->last ())->location ().line ();
#ifdef ACMODEL
    // TODO: in the future, handle aspect headers differently
    ACM_Header *new_file = newHeader();
    get_files().insert(new_file);
    new_file->set_filename(model_filename (funit));
    new_file->set_len(len);
    new_file->get_in().insert(_tunit_file);
    new_file->set_time (modification_time (funit));
    file = new_file;
#else
    file = register_file (model_filename (funit), len, false);
    file->included_by (_tunit_file);
#endif
    _file_map.insert (FileMapPair (funit, file));
  }
  int line = tree->token ()->location ().line ();
  int len = tree->end_token ()->location ().line () - line + 1;
#ifdef ACMODEL
  // TODO: really use linear search here?
  typedef ACM_Container<ACM_Source, true> Container;
  const Container &sources = name->get_source ();
  bool found = false;
  for (Container::const_iterator i = sources.begin(); i != sources.end(); ++i) {
    if ((*i)->get_line () == line &&
        (*i)->get_file () == file &&
        (*i)->get_kind () == kind &&
        (*i)->get_len () == len) {
      found = true;
      break;
    }
  }
  if (!found) {
    ACM_Source *source = newSource();
    source->set_file(file);
    source->set_line(line);
    source->set_len(len);
    source->set_kind(kind);
    name->get_source().insert(source);
  }
#else
  SourceLoc source_loc (file->id (), line, len, kind);
  name->source_loc (source_loc);
#endif
}

FileUnit *ModelBuilder::source_unit (CTree *tree) {
  if (!tree || !tree->token ())
    return 0;

  Unit *unit   = (Unit*)tree->token ()->belonging_to ();
  while (unit && unit->isMacroExp ()) {
    unit = ((MacroUnit*)unit)->CallingUnit ();
  }
  // if the 'insert unit' is an 'introduction unit', find the intro target unit
  IntroductionUnit *intro_unit = IntroductionUnit::cast (unit);
  if (intro_unit) unit = intro_unit->final_target_unit ();

  // at this point we must have reached a file unit
  if (!unit->isFile ())
    return 0;

  return (FileUnit*)unit;
}
